///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <string>
#include <cstdlib>
#include "docview.h"
#include "textframe.h"
#include "imageframe.h"
#include "rasterframe.h"
#include "document.h"
#include "page.h"
#include <algorithm>
#include "errordialog.h"
#include "pagesel.h"
#include "icons/wait.xpm"
#include "icons/missing.xpm"
#include "icons/broken.xpm"
#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>
//#include "safeslot.h"

IVector Document_View::get_origin() const
{
  return IVector(margin_size, margin_size);
}

Gdk_Point Document_View::pt2scr(const Vector& pt) const
{
  return Gdk_Point(int(get_origin().x+pt2scr(pt.x)+0.5),
		   int(get_origin().y+page_height()-pt2scr(pt.y)+0.5));
}

Vector Document_View::scr2pt(const Gdk_Point& scr) const
{
  return Vector(scr2pt(scr->x-get_origin().x),
		scr2pt(page_height()-(scr->y-get_origin().y)));
}

int Document_View::page_height() const
{
  return (int) (document?pt2scr(document->get_height())+0.5:0);
}

int Document_View::page_width() const
{
  return (int) (document?pt2scr(document->get_width())+0.5:0);
}


void Document_View::insert_page_before()
{
  if(document)
    template_page_dialog->show_it(true, document->get_num_of_pages()?current_page_num:document->get_first_page_num());
}

void Document_View::insert_page_after()
{
  if(document)
    template_page_dialog->show_it(true, document->get_num_of_pages()?current_page_num+1:document->get_first_page_num());
}

void Document_View::delete_page()
{
  try
    {
      if(document)
	document->delete_page(current_page_num);
    }
  catch(Error::Invalid_Page_Num e)
    {
      warning << "Document_View::delete_page: " << e.what() << std::endl;
    }
}

Page *Document_View::get_page()
{
  if(document)
    {
     Page *page=document->get_page(current_page_num);
     // if page has changed, release memory from old page
     if(old_page && page!=old_page)
       {
	 try
	   {
	     document->get_page_num_of_page(old_page);
	     // If old_page has been deleted, then the line above will 
	     // throw an exception, and then line below won't be executed.
	     old_page->clear_cache();
	   }
	 catch(Error::Invalid_Page_Num)
	   {
	   }
       }
     old_page=page;
     return page;
    }
  else
    return 0;
}

void Document_View::act_on_document_change(Document *document_)
{
  int first, num;

  if(!document || document_!=document)
    return;
  first=document->get_first_page_num();
  num=document->get_num_of_pages();
  if(current_page_num<first)
    {
      current_page_num=first;
      current_page_num_changed_signal();
    }
  else if(current_page_num>first+num-1)
    {
      current_page_num=first+num-1;
      current_page_num_changed_signal();
    }
  draw();
  document_changed_signal();
}

void Document_View::set_current_page_num(int current)
{
  int first, num;
  
  if(!document)
    return;
  first=document->get_first_page_num();
  num=document->get_num_of_pages();
  if(current<first)
    current=first;
  else if(current>first+num-1)
    current=first+num-1;
  if(current!=current_page_num)
    {
      current_page_num=current;
      current_page_num_changed_signal();
      draw(); 
    }
}

void Document_View::set_document(Document *d)
{
  document=d;

  if(document)
    {
      current_page_num=document->get_first_page_num();
      Document::changed_signal.connect(slot(this, &Document_View::act_on_document_change));
      Document::size_changed_signal.connect(slot(this, &Document_View::act_on_size_change));
    }

  document_changed_signal(); //?
  document_set_signal();

  adjust_size();
  draw();
}

Document_View::Document_View(Document *d):
  old_page(0),
  template_page_dialog(0), 
  wait_pixmap(0), missing_pixmap(0), broken_pixmap(0)
{
  set_events(GDK_EXPOSURE_MASK
	     | GDK_LEAVE_NOTIFY_MASK
	     | GDK_BUTTON_PRESS_MASK
	     | GDK_BUTTON_RELEASE_MASK
	     | GDK_POINTER_MOTION_MASK
	     | GDK_POINTER_MOTION_HINT_MASK);
  Gdk_Colormap colormap_=get_default_colormap ();
  white=Gdk_Color("white");
  black=Gdk_Color("black");
  gray=Gdk_Color("gray");
  red=Gdk_Color("red");
  colormap_.alloc(white);
  colormap_.alloc(black);
  colormap_.alloc(gray);
  colormap_.alloc(red);
  zoom_factor=0.5;
  resolution=72; //ppi
  set_document(d);
  reshaping=moving=false;
  margin_size=30;
  Page::ready_to_draw_page_signal.connect(slot(this, &Document_View::maybe_page_ready_to_draw));
}

void Document_View::maybe_page_ready_to_draw(Pagent *pagent)
{
  if(pagent==get_page())
    draw();
}

Document_View::~Document_View()
{
  if(document)
    delete document;
  if(wait_pixmap)
    delete wait_pixmap;
  if(missing_pixmap)
    delete missing_pixmap;
  if(broken_pixmap)
    delete broken_pixmap;
}

void Document_View::adjust_size()
{
  if(document)
    {
      size(2*margin_size+int(pt2scr(document->get_width())+0.5), 
	   2*margin_size+int(pt2scr(document->get_height())+0.5));
    }
}

void Document_View::act_on_size_change(Document *document_)
{
  if(document && document_==document)
    {
      hide();
      adjust_size();
      show();
    }
}

void Document_View::set_zoom_factor(float factor)
{
  if(factor>0 && document)
    {
      Page *page=get_page();

      zoom_factor=factor;
      hide();
      adjust_size();
      if(page) 
	page->act_on_zoom_factor_change();
      zoom_change_signal(zoom_factor);
      show();
    }
}

bool Document_View::in_move_area(int x, int y)
{
  if(Page* page=get_page()) {
    const Document::Selection selected = get_document()->selected();
    // "paper coordinates"
    const Vector pos = scr2pt(Gdk_Point(x, y));
    const float dist = scr2pt(2);

    for(Document::Selection::const_iterator i=selected.begin(); i != selected.end(); ++i) {
      if(&getPage(**i) == page && (*i)->inside_select_area(pos, dist))
	return true;
    }
  }
  return false;
}

Pagent* Document_View::in_select_area(int x, int y)
{
  if(Page* page=get_page()) {
    const Vector pos = scr2pt(Gdk_Point(x, y));
    const float dist = scr2pt(2);

    for(Group::ChildVec::const_iterator i = page->pbegin(); 
	i != page->pend(); 
	i++) {
      if((*i)->inside_select_area(pos, dist))
	return *i;
    }
  }
  return 0;
}

gint Document_View::button_press_event_impl(GdkEventButton *event)
{
  // Cursor-browsing hack:
  //  static GdkCursorType foo=GdkCursorType(0);
  //  cerr << foo << endl;
  //  Gdk_Cursor moving_cursor(foo);
  //  foo=GdkCursorType(foo+2);
  //  win.set_cursor(moving_cursor);

  Page *page=get_page();
  if(page && event->button==1)
    {
      // "paper coordinates"
      const Vector pos = scr2pt(Gdk_Point(int(event->x), int(event->y)));
      const float dist = scr2pt(2);

      //begin reshape?
      const Document::Selection selected = get_document()->selected();
      if(Pagent* select = in_select_area(int(event->x), int(event->y)))
	{
	  // is it already selected?
	  if(find(selected.begin(), selected.end(), select)!=selected.end())
	    {
	      if(event->state & GDK_MOD1_MASK)
		{
		  get_document()->deselect(select);
		}
	      else if(!select->is_locked())
		{		
		  int j;
		  if(selected.size() == 1 
		     && selected.front()->is_reshapable() 
		     && (j = selected.front()->inside_reshape_box(pos, dist)))
		    begin_reshape(int(event->x), int(event->y), j);
		  else
		    begin_move(int(event->x), int(event->y));
		}	      
	    }
	  else
	    {
	      get_document()->select(select, !(event->state & GDK_MOD1_MASK));
	      if(!select->is_locked())
		begin_move(int(event->x), int(event->y));
	    }
	}
      else
	if((event->state & GDK_MOD1_MASK) == 0)
	  get_document()->select_all(false);

      draw();
    }

  return true;
}

void Document_View::new_text_frame()
{
  Page *page=get_page();
  if(!page || !document)
    return;
  
  int w=200;
  int h=300;
  // default to the first (alphabetically) stream
  Document::StreamVec text_streams=document->get_text_streams();
  Text_Stream *stream=text_streams.empty()?0:*text_streams.begin();
  page->addObject(new Text_Frame(page, stream, w, h));

  draw();
}

void Document_View::new_image_frame(std::string filename)
{
  Page *page=get_page();
  if(!page)
    return;
  get_document()->select_all(false);
  
  if(suffix(filename) == "eps") {
    page->addObject(new Image_Frame(page, filename, 5, 5));
  } else {
    // Assume it is some kind of raster image, and hope we support it
    // (otherwise we should get an exception).
    page->addObject(new Raster_Frame(page, filename));
  }
  draw();
}

void Document_View::select_all_frames()
{
  if(Document *doc = get_document())
    doc->select_all(true); 
}

void Document_View::unselect_all_frames()
{
  if(Document *doc = get_document())
    doc->select_all(false); 
}

void Document_View::delete_selected()
{
  if(Document *doc = get_document()) {
    doc->delete_selected();
    draw();
  }
}  

void Document_View::group_selected()
{
  if(Page *page = get_page()) {
    page->group_selected();
    draw();
  }
}

void Document_View::ungroup_selected()
{
  if(Page *page = get_page()) {
    page->ungroup_selected();
    draw();
  }
}

void Document_View::rearrange_selected(Rearrange_Action action)
{
  if(Page *page = get_page()) {
    page->rearrange_selected(action);
    draw();
  }
}  

void Document_View::refresh_streams()
{
  if(Page *page=get_page()) {
    Document::StreamVec streams;
    for(Group::ChildVec::const_iterator i=page->pbegin();
	i!=page->pend();
	i++)
      {	
	Text_Frame *tmp=dynamic_cast<Text_Frame*>(*i);
	if(tmp)
	  {
	    Text_Stream *stream=tmp->get_text_stream();
	    if(stream && 
	       find(streams.begin(), streams.end(), stream)==streams.end())
	      streams.push_back(stream);
	  }
      }
    for(Document::StreamVec::iterator i=streams.begin();
	i!=streams.end();
	i++)
      {
	(*i)->run_typesetter();
	debug << (*i)->get_name() << ", " 
	      << (*i)->get_association() << std::endl;
      }
  }
}

void Document_View::begin_reshape(int x, int y, int box)
{
  if(Page *page = get_page()) {
    last_x=x; last_y=y; reshape_box=box; reshaping=true;
    const Document::Selection selected = get_document()->selected();
    for_each(selected.begin(), selected.end(),
	     std::mem_fun(&Pagent::begin_reshape));
  }
}

void Document_View::end_reshape(bool revert)
{
  if(Document *document=get_document()) 
    {
      reshaping=false;
      const Document::Selection selected = document->selected();
      for_each(selected.begin(), selected.end(),
      	       std::bind2nd(std::mem_fun(&Pagent::end_reshape), false));
      refresh_streams();
    }
}

void Document_View::begin_move(int x, int y)
{
  if(Page *page=get_page()) {
    last_x=x; last_y=y; moving=true;  
  }
}

void Document_View::end_move(bool revert)
{
  if(Page *page=get_page()) {
    moving=false;
    refresh_streams();
  }
}

gint Document_View::button_release_event_impl(GdkEventButton *event)
{
  if(moving)
    end_move(false);
  if(reshaping)
    end_reshape(false);
  return true;
}

namespace {			// local
  bool hover_reshape(const Document_View& view, const Document& doc,
		     int x, int y, int& j) {
    const Document::Selection selected = doc.selected();
    // "paper coordinates"
    if(selected.size() != 1) return false;
    const Pagent& obj = *selected.front();
    if(!obj.is_reshapable()) return false;
    
    const Vector pos = view.scr2pt(Gdk_Point(x, y));
    const float dist = view.scr2pt(2);
    return (j = obj.inside_reshape_box(pos, dist));
  }
}

Gdk_Cursor Document_View::get_cursor(int x, int y)
{
  Document *document=get_document();
  if(!document)
    return GDK_LEFT_PTR;
    
  
  if(moving)
    return GDK_FLEUR;

  const Document::Selection selection=document->selected();
  Pagent *obj=in_select_area(x, y);
  
  if(!reshaping && obj
     && find(selection.begin(), selection.end(), obj)==selection.end())
    return GDK_TOP_LEFT_ARROW;
  
  int j=reshape_box;
  if(reshaping || hover_reshape(*this, *get_document(), x, y, j))
    switch(j)
      {
      case 1: return GDK_TOP_LEFT_CORNER;	
      case 2: return GDK_TOP_SIDE;
      case 3: return GDK_TOP_RIGHT_CORNER;
      case 4: return GDK_LEFT_SIDE;
      case 5: return GDK_RIGHT_SIDE;
      case 6: return GDK_BOTTOM_LEFT_CORNER;
      case 7: return GDK_BOTTOM_SIDE;
      case 8: return GDK_BOTTOM_RIGHT_CORNER;
      default: return GDK_QUESTION_ARROW;
      }

  if(in_move_area(x, y))
    return GDK_FLEUR;

  if(in_select_area(x, y))
    return GDK_TOP_LEFT_ARROW;
  
  return GDK_LEFT_PTR;
}

void Document_View::update_cursor(int x, int y) {
  win.set_cursor(get_cursor(x, y));
}

gint Document_View::motion_notify_event_impl(GdkEventMotion *event)
{
  Page *page=get_page();
  if(!page)
    return true;
  const Document::Selection selected = get_document()->selected();
  Document::Selection::const_iterator i=selected.begin();
  int x, y;
  GdkModifierType state;
  Gdk_Window window(event->window); 
  if (event->is_hint)
    window.get_pointer(x, y, state);
  else
    {
      x=(int) event->x;
      y=(int) event->y;
      state=(GdkModifierType) event->state;
    }
  // don't really understand the above, got it from an example. 
  // It works.

  update_cursor(int(event->x), int(event->y));

  if(moving || reshaping)
    {
      if(moving)
	while(i!=selected.end())
	  {
	    if(!(*i)->is_locked())
	      (*i)->set_matrix((*i)->get_matrix()
			       *Matrix::translation(scr2pt(x-last_x), 
						    scr2pt(last_y-y)));

	    i++;
	  }
      else if(reshaping)
	while(i!=selected.end())
	  {
	    const Vector vec(scr2pt(x-last_x), scr2pt(y-last_y));
	    if(!(*i)->is_locked())
	      (*i)->move_reshape_box(reshape_box, vec);
	    i++;
	  }
      last_x=x;
      last_y=y;
      draw();
    }
  return true;
}

void Document_View::realize_impl()
{
  Gtk::DrawingArea::realize_impl();

  template_page_dialog=manage(new Template_Page_Dialog(*get_toplevel(), 
  						       *this));
  // get_toplevel doesn't work in the constructor

  win = get_window();
  gc.create(win);
  //  GdkWindow *gdkwin=(GdkWindow*) win.gdkobj();
  Gdk_Bitmap bitmap;
  wait_pixmap.create_from_xpm_d(win, bitmap, white, 
				wait_xpm); 
  missing_pixmap.create_from_xpm_d(win, bitmap, white, 
				   missing_xpm); 
  broken_pixmap.create_from_xpm_d(win, bitmap, white, 
				  broken_xpm); 
  //  win.set_background(gray);
  //  win.clear();
}

void Document_View::draw()
{
  if(!is_realized())
    return; // will get lots of Gdk-CRITICALs otherwise

  Gtk::Style *style=get_style();
  IVector origin=get_origin();

  if(Page *page = get_page()) {
    page->draw_content(*this);
    int w=page_width()+origin.x;
    int h=page_height()+origin.y;
    style->apply_default_background(win, true, GTK_STATE_NORMAL,
				    Gdk_Rectangle(0, 0, width(), origin.y),
				    0, 0, width(), height());    
    style->apply_default_background(win, true, GTK_STATE_NORMAL,
				    Gdk_Rectangle(0, 0, origin.x, height()),
				    0, 0, width(), height());    
    style->apply_default_background(win, true, GTK_STATE_NORMAL,
				    Gdk_Rectangle(0, h, width(), height()-h),
				    0, 0, width(), height());    
    style->apply_default_background(win, true, GTK_STATE_NORMAL,
				    Gdk_Rectangle(w, 0, width()-w, height()),
				    0, 0, width(), height());    

    // ruler calibration marks at 100 and 500 points
    /*
    Gdk_Point calib=pt2scr(Vector(100, 100));
    style->draw_hline(win, GTK_STATE_NORMAL, 0, 20, calib->y);
    style->draw_vline(win, GTK_STATE_NORMAL, 0, 20, calib->x);
    calib=pt2scr(Vector(500, 500));
    style->draw_hline(win, GTK_STATE_NORMAL, 0, 20, calib->y);
    style->draw_vline(win, GTK_STATE_NORMAL, 0, 20, calib->x);
    */
  } else if(document){
    std::string message="No Pages";
    int x, y, w, h;
    Gdk_Font font=style->get_font();

    w=page_width(); 
    h=page_height();
    style->apply_default_background(win, true, GTK_STATE_NORMAL,
				    Gdk_Rectangle(0, 0, width(), height()),
				    0, 0, width(), height());    
    x=(w-font.string_width(message))/2+origin.x; 
    y=(h-font.string_height(message))/2+origin.y; 
    style->draw_string(win, GTK_STATE_NORMAL, x, y, message);
    style->draw_hline(win, GTK_STATE_NORMAL, origin.x, origin.x+w-1, origin.y);
    style->draw_hline(win, GTK_STATE_NORMAL, origin.x, origin.x+w-1, origin.y+h-1);
    style->draw_vline(win, GTK_STATE_NORMAL, origin.y, origin.y+h-1, origin.x);
    style->draw_vline(win, GTK_STATE_NORMAL, origin.y, origin.y+h-1, origin.x+w-1);
  } else
    style->apply_default_background(win, true, GTK_STATE_NORMAL,
				    Gdk_Rectangle(0, 0, width(), height()),
				    0, 0, width(), height());    
  show();
}

int Document_View::expose_event_impl (GdkEventExpose *event)
{
  draw();
  return false;
}

const Gdk_Color& Document_View::get_color(Color::Id color) const {
  switch(color) {
  case Color::bg:     return white;
  case Color::frame:  return black;
  case Color::locked: return gray;
  case Color::empty:  return gray;
  default:  throw std::runtime_error("Unknown color id get_color");
  }
}
