///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
// -*- c++ -*-
#ifndef DOCVIEW_H
#define DOCVIEW_H

#include "view.h"
#include <gdk--.h>
#include <gtk--/drawingarea.h>
#include <gtk--/style.h>
#include <list>
#include <cmath>
#include "group.h"
#include "units.h"
#include "matrix.h"

class Pagent;
class Template_Page_Dialog;
class Page;
class Document;

class Document_View: public Gtk::DrawingArea, public View {
public:
  SigC::Signal0<void> current_page_num_changed_signal;
  SigC::Signal0<void> document_changed_signal;
  SigC::Signal0<void> document_set_signal;
  SigC::Signal1<void, float> zoom_change_signal;

  Gdk_Color white, black, gray, red;

  Document_View(Document*);
  ~Document_View();
  void set_document(Document *d);
  Document *get_document()
  {return document;}
  void set_current_page_num(int);
  int get_current_page_num()
  {return current_page_num;}
  void draw();
  Page* get_page();
  float get_zoom_factor() const
  {return zoom_factor;}
  void set_zoom_factor(float);

  void insert_page_before();
  void insert_page_after();
  void delete_page();

  void delete_selected();
  void rearrange_selected(Rearrange_Action action);
  void group_selected();
  void ungroup_selected();
  void new_text_frame();
  void new_image_frame(std::string);
  void select_all_frames();
  void unselect_all_frames();
  Gdk_Window &get_win()
  {return win;}
  Gdk_GC &get_gc()
  {return gc;}
  const Gdk_Color& get_color(Color::Id color) const;

  IVector get_origin() const;
  gdouble get_scrres() const {
    return floor (resolution * zoom_factor + 0.5);
  }
  FLength pt2scr(FLength pt) const
  {
    using namespace Units;
    return floor(resolution*zoom_factor*from_pt(pt, IN)+0.5);
  }
  FLength scr2pt(FLength scr) const
  {
    using namespace Units;
    return to_pt(scr/(resolution*zoom_factor), IN);
  }  
  // Important! the scalar versions of pt2scr and scr2pt only perform
  // length conversions while the vector versions also convert between
  // right- and left-handed coordinate systems
  Gdk_Point pt2scr(const Vector& pt) const;
  Vector scr2pt(const Gdk_Point& scr) const;
  int page_height() const;
  int page_width() const;

  const Gdk_Pixmap &get_wait_pixmap() const {return wait_pixmap;}
  const Gdk_Pixmap &get_missing_pixmap() const {return missing_pixmap;}
  const Gdk_Pixmap &get_broken_pixmap() const {return broken_pixmap;}
private:
  // Undefined ctors, avoid defaults
  Document_View(const Document_View&);
  Document_View();
  void operator = (const Document_View&);

  Gdk_Window win;
  Gdk_GC gc;

  int margin_size;
  float resolution; //ppi
  float zoom_factor;
  int last_x, last_y;
  int reshape_box, current_page_num;
  bool moving, reshaping;
  Document *document;
  Page *old_page; //last page visited
  Template_Page_Dialog *template_page_dialog;
  Gdk_Pixmap wait_pixmap, missing_pixmap, broken_pixmap;

  void realize_impl();
  int expose_event_impl(GdkEventExpose*);
  gint button_press_event_impl(GdkEventButton*);
  gint button_release_event_impl(GdkEventButton*);
  gint motion_notify_event_impl(GdkEventMotion*);

  void refresh_streams();
  void begin_move(int x, int y);
  void end_move(bool revert);
  void begin_reshape(int x, int y, int box);
  void end_reshape(bool revert);

  void act_on_document_change(Document *document_);
  Gdk_Cursor get_cursor(int x, int y);
  void update_cursor(int x, int y);
  bool in_move_area(int x, int y);
  Pagent* in_select_area(int x, int y);
  void adjust_size();
  void act_on_size_change(Document *document_);
  void maybe_page_ready_to_draw(Pagent *pagent);
};

#endif







