///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "group.h"
#include "docview.h"
#include "document.h"
#include <xml++.h>
#include <util/stringutil.h>
#include <util/warning.h>
#include <gdk--.h>
//#include <float.h>
#include <algorithm>

Group::Group(Group* parent, const std::string& name)
  : Pagent(parent, name)
{
  reshapable=false;
}

Group::Group(Group* parent, const std::string& name, const Matrix& xform)
  : Pagent(parent, name)
{
  set_matrix(xform);
  reshapable=false;
}

Group::~Group() {
  // Note:  When a group is "ungrouped", the childs should not be removed.
  // Fix that by moving the childs from the group to its parent before
  // deleteing the group.
  for(ChildVec::iterator i = childs.begin(); i != childs.end(); i++)
    delete *i;
}

void Group::print(std::ostream& out) {
  out << "gsave  % group" << std::endl;
  psConcat(out, get_matrix());
  for(ChildVec::reverse_iterator i = childs.rbegin(); i != childs.rend(); i++)
    (*i)->print(out);
  out << "grestore % /group" << std::endl;
}

class SubView : public View {
public:
  SubView(View& p, const Matrix& m) : parent(p), matrix(m) {}
  
  Gdk_Window& get_win() { return parent.get_win(); }
  Gdk_GC& get_gc() { return parent.get_gc(); }
  
  gdouble get_scrres() const {
    const Matrix inv = matrix.inv();
    return dist(inv.transform(Vector(parent.get_scrres(), 0)),
		inv.transform(Vector(0,0)));
  }
  FLength pt2scr(FLength pt) const {
    return parent.pt2scr(dist(matrix.transform(Vector(pt, 0)),
			      matrix.transform(Vector(0,0))));
  }
  Gdk_Point pt2scr(const Vector& pt) const {
    return parent.pt2scr(matrix.transform(pt));
  }
  Vector scr2pt(const Gdk_Point& scr) const {
    return matrix.inv().transform(parent.scr2pt(scr));
  }
  const Gdk_Pixmap &get_wait_pixmap() const {
    return parent.get_wait_pixmap();
  }
  const Gdk_Pixmap &get_missing_pixmap() const {
    return parent.get_missing_pixmap();
  }
  const Gdk_Pixmap &get_broken_pixmap() const {
    return parent.get_broken_pixmap();
  }
  const Gdk_Color& get_color(Color::Id color) const { 
    return parent.get_color(color); 
  }
private:
  View& parent;
  const Matrix& matrix;
};

bool Group::draw_content(View& view) {
  SubView v(view, get_matrix());
  // reverse order, this is correct at least for Page
  bool ok = true;
  for(ChildVec::reverse_iterator i = childs.rbegin(); 
      i != childs.rend(); 
      i++)
    ok = (*i)->draw_content(v) && ok;
  return ok;
}

void Group::clear_cache() {
  for_each(childs.begin(), childs.end(),
	   std::mem_fun(&Pagent::clear_cache));
}

class Rectangle {
public:
  Rectangle(const Vector& v) : lo_x(v.x), lo_y(v.y), hi_x(v.x), hi_y(v.y) {}
  void grow(const Vector& v) {
    if(v.x < lo_x) lo_x = v.x;
    if(v.x > hi_x) hi_x = v.x;
    if(v.y < lo_y) lo_y = v.y;
    if(v.y > hi_y) hi_y = v.y;
  }
  Box getBox() const {
    return Box(hi_x - lo_x, hi_y - lo_y)
      *= Matrix::translation(Vector(-lo_x, -lo_y));
  }
private:
  double lo_x, lo_y, hi_x, hi_y;
};

Box Group::get_box() const {
  // If the group has no members, return a zero-size box at the right place
  if(childs.empty()) return Box(0,0) *= get_matrix();

  ChildVec::const_iterator i = childs.begin();
  const Box box = (*i)->get_box();
  Rectangle rect(box.getLL());
  rect.grow(box.getLR());
  rect.grow(box.getUL());
  rect.grow(box.getUR());
  
  for(; i != childs.end(); i++) {
    const Box box = (*i)->get_box();
    rect.grow(box.getLL());
    rect.grow(box.getLR());
    rect.grow(box.getUL());
    rect.grow(box.getUR());
  }
  return rect.getBox() *= get_matrix();
}

void Group::move_reshape_box(int, Vector) {} // Todo

// Note: maybe it's best to let a pagent ask its parent what its 
// policy on flow around is.
void Group::setFlowAround(bool) {} // Todo
bool Group::getFlowAround() const { return false; } // Todo

BoundaryVect Group::obstacle_list() const {
  BoundaryVect result;
  
  for(ChildVec::const_iterator i=childs.begin();
      i!=childs.end();
      i++)
    {
      if(Boundary* boundary = (*i)->getObstacleBoundary())
	result.push_back(boundary);
    }
  return result;
}

Boundary* Group::getObstacleBoundary() const { return 0; } // Todo

void Group::act_on_zoom_factor_change() {
  for_each(childs.begin(), childs.end(),
	   std::mem_fun(&Pagent::act_on_zoom_factor_change));
}

XMLNode* Group::save() {
  XMLNode *node = new XMLNode("frame");
  node->add_property("type", "group");
  node->add_property("transform", tostr(get_matrix()));

  saveChilds(*node);
  return node;
}

void Group::saveChilds(XMLNode& node) {
  for(ChildVec::reverse_iterator i = childs.rbegin(); 
      i != childs.rend(); 
      i++) {
    node.add_content("\n      ");
    node.add_child((*i)->save());
  }
  node.add_content("\n    ");
}

void Group::group_selected()
{
  //Todo: make this work by doing matrix operations
  const Document::Selection selection=get_document().selected();

  if(selection.size()<=1) //no one-item groups, please
    return;

  // Calculate translation:
  double minx = DBL_MAX, miny = DBL_MAX;
  for(Document::Selection::const_iterator i=selection.begin();
      i!=selection.end();
      i++)
    {
      const Matrix& m = (*i)->get_matrix();
      minx = std::min(minx, m.tr_x());
      miny = std::min(miny, m.tr_y());
    }
  
  Group *group=new Group(this, "group", Matrix::translation(minx, miny));
  
  // Use a copy or we will confuse ourselves while manipulating a container 
  // we are iterating over.
  ChildVec childs_copy=childs;

  // reparent selected
  for(ChildVec::reverse_iterator i=childs_copy.rbegin();
      i!=childs_copy.rend();
      i++)
	{
	  if(find(selection.begin(), selection.end(), *i) != selection.end()) 
	    {
	      (*i)->set_matrix((*i)->get_matrix()*group->get_matrix().inv());
	      group->add(ungroup(*i));
	    }
	}
  if(group->count())
    {
      add(group);
      get_document().select(group);
    }
  else {
    warning << "Got a strange group. Deleting it again." << std::endl;
    delete group;
  }
}

void Group::ungroup_selected()
{
  // Todo: preserve stacking order
  const Document::Selection &selection=get_document().selected();
  for(Document::Selection::const_iterator i=selection.begin();
      i!=selection.end();
      i++)
    {
      Group *group=dynamic_cast<Group*>(*i);
      if(group)
	{
	  get_document().deselect(*i);
	  while(group->pbegin()!=group->pend())
	    {
	      Pagent *pagent=group->ungroup(*(group->prbegin()));
	      if(pagent)
		{
		  pagent->set_matrix(pagent->get_matrix()*group->get_matrix());
		  add(pagent);
		  get_document().select(pagent, false);
		}
	    }
	  if(!ungroup(group))
	    warning << "Group not in this group" << std::endl;
	  delete group;
	}
    }
}

void Group::add(Pagent* obj) {
  // the front is the top
  childs.push_front(obj);
  obj->reparent(this);
}

Pagent* Group::ungroup(Pagent* obj) {
  ChildVec::iterator i = find(childs.begin(), childs.end(), obj);
  if(i==childs.end()) //won't let me compare i==pend()
    return 0;
  childs.erase(i);
  obj->reparent(parent);
  return obj;
}

void Group::rearrange_selected(Rearrange_Action action)
{
  const Document::Selection &selection=get_document().selected();
  if(action==TOP || action==BOTTOM)
    {
      ChildVec sorted_selection;
      // remove selected items from list
      for(Document::Selection::const_iterator s=selection.begin(); 
	  s!=selection.end(); 
	  s++)
	{
	  ChildVec::iterator i = find(childs.begin(), childs.end(), *s);
	  if(i != childs.end()) 
	    {
	      sorted_selection.push_back(*i);
	      childs.erase(i);
	    }
	  else
	    warning << "Selected item not on this page." << std::endl;
	}
      // append items to front or back
      childs.insert(action==TOP?childs.begin():childs.end(), 
		     sorted_selection.begin(), 
		     sorted_selection.end());
    }
  else // UP or DOWN
    {
      // This is basically a single-step bubblesort,
      // except that each element may only be swapped once.
      for(ChildVec::iterator i=childs.begin(); i!=childs.end(); i++)
	{
	  ChildVec::iterator j=i;
	  j++;
	  if(j==childs.end())
	    break;
	  bool i_selected=(find(selection.begin(), 
				selection.end(), 
				*i)!=selection.end());
	  bool j_selected=(find(selection.begin(), 
				selection.end(), 
				*j)!=selection.end());
	  if(i_selected && j_selected)
	    continue;
	  if((i_selected && action==DOWN) || (j_selected && action==UP))
	    {
	      Pagent *tmp=*j; *j=*i; *i=tmp; // swap
	      i++; // no swap next time
	    }
	}
    }
  geometry_changed_signal(this);
}

void Group::child_ready_to_draw(Pagent *pagent)
{
  //  if(pagent!=this) // no infinite loops, please
  ready_to_draw_signal(this);
}

void Group::child_props_changed(Pagent *pagent)
{
  //  if(pagent!=this) // no infinite loops, please
  props_changed_signal(this);
}

void Group::child_geometry_changed(Pagent *pagent)
{
  //  if(pagent!=this) // no infinite loops, please
  geometry_changed_signal(this);
}
