///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "imageframe.h"
#include "fileerrors.h"
#include "processman.h"
#include "config.h"
#include "globals.h"
#include <strstream>
#include <fstream>
#include <xml++.h>
#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>

Image_Frame::Image_Frame(Group *parent, 
			 const std::string& assoc, FLength w, FLength h)
  :Cached_Frame(parent, w, h, "Image " + basename(assoc)),
   association(assoc)
{
  parsed_file=process_manager.find_new_name();
}

Image_Frame::Image_Frame(Group *parent, const XMLNode& node)
  :Cached_Frame(parent, node)
{
  if(!node.property("type") || node.property("type")->value()!="image")
    throw Error::Frame_Type("Bad image-frame type: "
			    + node.property("type")->value());

  if(const XMLProperty* file = node.property("file"))
    set_association(from_relative(file->value()));
  else
    {
      set_association("");
      warning << "No \"file\" property found in image frame" << std::endl;
    }

  parsed_file=process_manager.find_new_name();
  read_size(); // try to read bounding box from file
}

Image_Frame::~Image_Frame()
{
  if(parsed_file!="")
    if(!process_manager.delete_file(parsed_file))
      warning << "Failed to delete " << parsed_file << std::endl;
}

void Image_Frame::file_modified()
{
  clear_picture();
  ready_to_draw_signal(this);
}

// Note: as long as this is only called in the constructor there is no need to
// emit a geometry_changed_signal
void Image_Frame::read_size()
{
  width=100; height=100; 
  // Must set the size to something, so we can see the frame even on failure.
  if(!association.empty())
    {
      std::ifstream in(association.c_str());
      std::string tmpline;
      while(getline(in, tmpline)) 
	{
	  if(starts_with(tmpline, "%%BoundingBox:")) {
	    std::istrstream tmp(tmpline.c_str(), tmpline.length());
	    std::string word;
	    float x1, y1, x2, y2;
	    if(tmp >> word >> x1 >> y1 >> x2 >> y2) 
	      {
		width = x2 - x1;  height = y2 - y1;
		return;
	      }
	    else
	      warning << "Bad bounding box in " << association << std::endl;
	  }
	}
    }
}

void Image_Frame::parse(std::istream &in, std::ostream &out)
{
  float x1, y1;		// width and height of actual file.
  bool boundingbox_found=false;
  bool transf_performed=false;

  std::string tmpline;
  while(getline(in, tmpline)) {
    if(starts_with(tmpline, "%%BoundingBox:")) {
      std::istrstream tmp(tmpline.c_str(), tmpline.length());
      std::string word;
      float x2, y2;
      if(tmp >> word >> x1 >> y1 >> x2 >> y2) {
	boundingbox_found = true;
	width = x2 - x1;  height = y2 - y1;
	out << "%%BoundingBox: 0 0 " << width << ' ' << height << std::endl;
      }
    } else if(!starts_with(tmpline, '%') && !transf_performed) {
      out << tmpline << std::endl << std::endl;
	  transf_performed=true;
	  if(!boundingbox_found)
	    {
	      warning << "No BoundingBox found in " << association 
		      << std::endl;
	      return; //FIXME
	    }
	  else
	    {
	      out << -x1 << ' ' << -y1 << " translate" << std::endl
		  << std::endl;
	    }
    } else out << tmpline << std::endl;
  }
}

void Image_Frame::generate_picture(View& view) {
  debug << "PID " << pid << std::endl;
  if(association.empty())
    throw Gen_Pic_Error(ASSOCIATION, "No associated file");
  if(!picture && pid==-1)
    {
      std::ostrstream tmp;

      Matrix m=get_matrix();

      {
	std::ifstream in(association.c_str());
	std::ofstream out(parsed_file.c_str());
	if(!in)
	  throw Gen_Pic_Error(ASSOCIATION, "Couldn't open "+association);
	if(!out)
	  throw Gen_Pic_Error(GENERATION, "Couldn't write to "+parsed_file);

	out << "%!PS" << std::endl;
	psConcat(out, Matrix::scaling(m.sc_x(), m.sc_y()));
	parse(in, out);
      }

      const Gdk_Point psize(int(view.pt2scr(width*m.sc_x())), 
			    int(view.pt2scr(height*m.sc_y())));
      if(psize->x==0 || psize->y==0)
	throw Gen_Pic_Error(ZEROSIZE, "Picture has zero size");

      const std::string psinterpreter=config.getPath("PSInterpreter");
      if(psinterpreter.empty())
	throw Gen_Pic_Error(GENERATION, "No PostScript interpreter specified");
      tmp<< psinterpreter //assuming it is ghostscript
	 <<" -q -dNOPAUSE -dBATCH -sDEVICE=ppmraw -r"
	 << view.get_scrres()
	 << " -g" << psize->x << 'x' << psize->y
	 <<" -sOutputFile=- "
	 <<parsed_file
	 <<" >"<<tmp_file
	 << std::ends;
      pid=process_manager.start(tmp.str());
      verbose << pid << ": " << tmp.str() << std::endl;
      if(pid==-1)
	throw Gen_Pic_Error(GENERATION, "Failed to run "+psinterpreter);
    }
}

XMLNode *Image_Frame::save()
{
  XMLNode *node=Cached_Frame::save();
  node->add_property("type", "image");
  node->add_property("file", to_relative(association));
  return node;
}

void Image_Frame::print(std::ostream &out)
{
  if(association.empty())
    {
      Cached_Frame::print(out);
      return;
    }
  
  using std::endl;
  
  out<<"save"<<endl;
  out<<"/showpage {} def"<<endl;

  psConcat(out, get_matrix());

  out<<"0 setgray 0 setlinecap 1 setlinewidth"<<endl;
  out<<"0 setlinejoin 10 setmiterlimit [] 0 setdash newpath"<<endl;
  if(true) //FIXME!!! if(level2)
    out<<"false setoverprint false setstrokeadjust"<<endl;

  out<<"%"<<association<<endl;

  out<<"%%BeginDocument: "<<association<<endl;
  std::ifstream in(association.c_str());
  if(in)
    parse(in, out);
  else
    throw Error::Print("Couldn't read "+association);
  out<<"%%EndDocument"<<endl;
  out<<"restore"<<endl;
}

void Image_Frame::set_association(std::string s)
{
  if(association == s) return;
  
  association=s;
  clear_picture();
  props_changed_signal(this);
  ready_to_draw_signal(this);
}
