///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "inspiration.h"
#include "safeslot.h"
#include <gtk--/main.h>
#include <gtk--/style.h>
#include <strstream>
#include <cstdlib>
#include "globals.h"

class Game: public Gtk::DrawingArea
{
  struct Over {};
  Gtk::Connection connection;
  Gdk_Window win;
  Gdk_Visual visual;
  Gdk_GC gc;
  Gdk_Image image;
  static const int candy_size=10;
  int score, grow_count;
  struct Vect {int x, y;};
  Vect candy_pos;
  struct Snake
  {
    static const int width=2;
    Vect direction, head, tail;
  } snake;
  virtual gint configure_event_impl(GdkEventConfigure*);
  virtual gint expose_event_impl(GdkEventExpose*);
  bool grow();
  void shrink();
  gint update();
  bool check_clean(int x, int y, int w, int h);
  void candy_found();
  guint32 snake_color, bg_color, candy_color;
  guint32 make_color(float r, float g, float b);
  void draw_rect(guint32 color, bool filled, int x, int y, int w, int h);
public:
  SigC::Signal1<void, long> score_change_signal;
  Game();
  gint key_press_event_impl(GdkEventKey*);
  void new_game();
  void stop_game();
};

Game::Game():
  Gtk::DrawingArea()
{
  set_events(GDK_EXPOSURE_MASK|
	     GDK_KEY_PRESS_MASK);
}

gint Game::configure_event_impl(GdkEventConfigure*)
{
  win = get_window();
  visual = win.get_visual();
  snake_color=make_color(0.4, 1, 0);
  candy_color=make_color(0.5, 1, 0.1);
  bg_color=make_color(0, 0, 0);

  if(!image)
    {
      gc=get_style()->get_white_gc();
      image.create(GDK_IMAGE_NORMAL, visual, width(), height());
      draw_rect(bg_color, true, 1, 1, width()-2, height()-2);
    }
  return true;
}

gint Game::key_press_event_impl(GdkEventKey *key)
{
  switch(key->keyval)
    {
    case 65361:
      if(snake.direction.x!=1)
	{
	  snake.direction.x=-1;
	  snake.direction.y=0;
	}
      return true;
      break;
    case 65362:
      if(snake.direction.y!=1)
	{
	  snake.direction.x=0;
	  snake.direction.y=-1;
	}
      return true;
      break;
    case 65363:
      if(snake.direction.x!=-1)
	{
	  snake.direction.x=1;
	  snake.direction.y=0;
	}
      return true;
      break;
    case 65364:
      if(snake.direction.y!=-1)
	{
	  snake.direction.x=0;
	  snake.direction.y=1;
	}
      return true;
      break;
    default:
      return false;
      break;
    }
}

gint Game::expose_event_impl(GdkEventExpose *event)
{
  gc=get_style()->get_fg_gc(get_state());

  if(!image)
    new_game();
  else
    win.draw_image(gc ,
		   image,
		   event->area.x, event->area.y,
		   event->area.x, event->area.y,
		   event->area.width, event->area.height);
  return false;
}

guint32 Game::make_color(float r, float g, float b)
{
  guint32 r_s, r_p, g_s, g_p, b_s, b_p;
  GdkVisual *v=dynamic_cast<GdkVisual*>(visual.gdkobj());

  r_s=v->red_shift;
  r_p=v->red_prec;

  g_s=v->green_shift;
  g_p=v->green_prec;

  b_s=v->blue_shift;
  b_p=v->blue_prec;

  guint32 rr=((guint32) (((1<<r_p)-1)*r))<<r_s;
  guint32 gg=((guint32) (((1<<g_p)-1)*g))<<g_s;
  guint32 bb=((guint32) (((1<<b_p)-1)*b))<<b_s;

  return rr|gg|bb;
}

void Game::draw_rect(guint32 color, bool filled, int x, int y, int w, int h)
{
  if(filled)
    {
      for(int i=0; i<w; i++)
	for(int j=0; j<h; j++)
	  image.put_pixel(x+i, y+j, color);
    }
  else
    {
      for(int i=0; i<w; i++)
	{
	  image.put_pixel(x+i, y, color);
	  image.put_pixel(x+i, y+h-1, color);
	}
      for(int j=0; j<h; j++)
	{
	  image.put_pixel(x, y+j, color);
	  image.put_pixel(x+w-1, y+j, color);
	}
    }
}

bool Game::check_clean(int x, int y, int w, int h)
{
  for(int i=0; i<w; i++)
    for(int j=0; j<h; j++)
      if(image.get_pixel(x+i, y+j)!=bg_color)  
	return false;
  return true;
}

void Game::candy_found()
{
  static const int maxcount=100;
  int i=0;

  grow_count=20;
  if(candy_pos.x>=0 && candy_pos.y>=0)
    {
      draw_rect(bg_color, true, 
		candy_pos.x, candy_pos.y, 
		candy_size, candy_size);
    }
  score_change_signal(score+=100);
  do
    {
      candy_pos.x=int((width()-candy_size)*(double(rand())/RAND_MAX));
      candy_pos.y=int((height()-candy_size)*(double(rand())/RAND_MAX));
    }
  while(i++<maxcount && 
	!check_clean(candy_pos.x, candy_pos.y, candy_size, candy_size));
  if(i<maxcount)
    {
      draw_rect(candy_color, true, 
		candy_pos.x, candy_pos.y, 
		candy_size, candy_size);
    }
  else
    {
    }
}

void Game::new_game()
{
  stop_game();
  grow_count=0;
  score=-100;
  image.create(GDK_IMAGE_NORMAL, visual, width(), height());
  draw_rect(bg_color, true, 0, 0, width(), height());
  draw_rect(snake_color, false, 0, 0, width(), height());
  candy_pos.x=-1;
  snake.direction.x=1; snake.direction.y=0;
  snake.head.x=width()/2; snake.head.y=height()/2;
  snake.tail=snake.head;
  for(int i=0; i<10; i++)
    grow();
  candy_found();
  connection=Gtk::Main::timeout.connect(safeslot(this, &Game::update), 20);  
}

void Game::stop_game()
{
  connection.disconnect();
  image.release();
}

bool Game::grow()
{
  snake.head.x+=snake.direction.x*snake.width;
  snake.head.y+=snake.direction.y*snake.width;
  if(snake.head.x+snake.width-1>=candy_pos.x &&
     snake.head.y+snake.width-1>=candy_pos.y &&
     snake.head.x<=candy_pos.x+candy_size-1 &&
     snake.head.y<=candy_pos.y+candy_size-1)
    candy_found();
  guint32 pixel=image.get_pixel(snake.head.x, snake.head.y);
  draw_rect(snake_color, true, snake.head.x, snake.head.y, 
	    snake.width,
	    snake.width);
  return !(pixel!=bg_color || snake.head.x<0 || snake.head.y<0 ||
	   snake.head.x>=width() || snake.head.y>=height());
}

void Game::shrink()
{
  if(grow_count>0)
    {
      --grow_count;
      return;
    }
  draw_rect(bg_color, true, snake.tail.x, snake.tail.y, 
	    snake.width,
	    snake.width);

  Vect tmp=snake.tail;
  snake.tail=tmp; snake.tail.x+=snake.width;
  if(image.get_pixel(snake.tail.x, snake.tail.y)==snake_color)
    return;
  snake.tail=tmp; snake.tail.x-=snake.width;
  if(image.get_pixel(snake.tail.x, snake.tail.y)==snake_color)
    return;
  snake.tail=tmp; snake.tail.y+=snake.width;
  if(image.get_pixel(snake.tail.x, snake.tail.y)==snake_color)
    return;
  snake.tail=tmp; snake.tail.y-=snake.width;
  if(image.get_pixel(snake.tail.x, snake.tail.y)==snake_color)
    return;
}

gint Game::update()
{
  shrink(); shrink();
  if(!grow() || !grow())
    connection.disconnect();
  GdkRectangle update_rect;
  update_rect.x=0;
  update_rect.y=0;
  update_rect.width=width();
  update_rect.height=height();
  draw(&update_rect);
  return true;
}

//**************************
//Inspiration
//**************************

Inspiration::Inspiration():
  Gtk::Window(GTK_WINDOW_TOPLEVEL),
  new_game_button(" New game "), 
  che_alert_button(" Hide ")
{
  set_events(GDK_KEY_PRESS_MASK);
  game=new Game();
  game->size(320, 240);
  set_border_width(border_width);
  set_title("Inspiration");
  set_policy(false, false, true);

  vbox.pack_start(score, true, false, 0);
  vbox.pack_start(*game, true, false, 0);
  vbox.pack_start(hbox, true, false, 5);
  hbox.pack_start(new_game_button, true, false, 0);
  hbox.pack_start(che_alert_button, true, false, 0);

  add(vbox);

  new_game_button.clicked.connect(safeslot(game, &Game::new_game));
  che_alert_button.clicked.connect(safeslot(this, &Inspiration::che_alert));
  game->score_change_signal.connect(slot(this, &Inspiration::score_change));
  
  vbox.show_all();
}

Inspiration::~Inspiration()
{
  delete game;
}

gint Inspiration::key_press_event_impl(GdkEventKey *key)
{
  return game->key_press_event_impl(key)?false:Gtk::Window::key_press_event_impl(key);
}

void Inspiration::score_change(long score_)
{
  std::ostrstream tmp;
  
  tmp << "Score: " << score_ << std::ends;
  score.set_text(tmp.str());
}

void Inspiration::che_alert()
{
  game->stop_game();
  hide();
}




