///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
// -*- c++ -*-
#ifndef MATRIX_H
#define MATRIX_H

#include <stdexcept>
#include <cmath> // sqrt

#define PI 3.14159265358979323846
template<class C> C sqr(const C& c) { return c*c; }

template <class C>
struct VectorOf
{
  C x, y;
  VectorOf(): x(C()), y(C()) {} // the zero vector
  VectorOf(C xx, C yy): x(xx), y(yy) {}

  VectorOf& operator += (const VectorOf& d)
    { x += d.x; y += d.y; return *this; }
  VectorOf& operator /= (const C& d)
    { x /= d; y /= d; return *this; }
};
typedef VectorOf<double> Vector;
typedef VectorOf<int> IVector;

template<class C>
inline VectorOf<C> operator - (const VectorOf<C>& a, const VectorOf<C> b) {
  return VectorOf<C>(a.x - b.x, a.y - b.y);
}
template<class C>
inline VectorOf<C> operator + (const VectorOf<C>& a, const VectorOf<C> b) {
  return VectorOf<C>(a) += b;
}
template<class C>
inline VectorOf<C> operator / (const VectorOf<C>& a, const C& b) {
  return VectorOf<C>(a) /= b;
}

template<class C>
inline std::ostream& operator << (std::ostream& out, const VectorOf<C> v) {
  return out << v.x << ',' << v.y;
}

template<class C>
inline C dist(const VectorOf<C>& a, const VectorOf<C>& b) {
  return sqrt(sqr(a.x - b.x) + sqr(a.y - b.y));
}

namespace Error
{
  struct Singular_Matrix : public std::runtime_error {
    Singular_Matrix() : std::runtime_error("Singular matrix") {}
  };
  // scaling a pagent by zero is not allowed
}

// transformations are performed in the following order:
// 1. Shearing
// 2. Scaling
// 3. Rotation
// 4. Translation

struct Matrix
{
  double rep[6];
  //rep[0] rep[1] 0
  //rep[2] rep[3] 0
  //rep[4] rep[5] 1
  Matrix(); // the identity matrix
  static Matrix shearing(double shcoeff);
  static Matrix scaling(double xfactor, double yfactor);
  static Matrix rotation(double angle);
  static Matrix translation(const Vector& offset);
  static Matrix translation(double xoffset, double yoffset);

  // making sense out of matrices:
  // may throw Error::Singular_Matrix
  bool is_plain() const;	// true iff this is only scale and translate.
  double tr_x() const; // translation x
  double tr_y() const; // translation y
  double rot() const; // rotation
  double sc_x() const; // scale x
  double sc_y() const; // scale y
  double sh() const; // shearing

  Matrix& set_tr_x(double val);  
  Matrix& set_tr_y(double val);
  Matrix& set_rot(double val);
  Matrix& set_scale(double sc_x, double sc_y);
  Matrix& set_sc_x(double val);
  Matrix& set_sc_y(double val);
  Matrix& set_sh(double val);

  // aliases
  double a() const {return rep[0];}
  double b() const {return rep[1];}
  double c() const {return rep[2];}
  double d() const {return rep[3];}

  // conversions
  static double rad2deg(double rad);
  static double deg2rad(double deg);

  // operations
  Vector transform(Vector v) const;
  Matrix operator * (const Matrix& m) const;
  bool operator == (const Matrix& m) const;
  double det() const;
  Matrix inv() const; // may throw Error::Singular_Matrix

  // debug function
  void print() const;
  
  friend std::ostream& operator<< (std::ostream& out, const Matrix&);
  friend std::istream& operator>> (std::istream& in, Matrix&);
};

inline Matrix& operator *= (Matrix& m1, const Matrix& m2) {
  m1 = m1 * m2;
  return m1;
}

void psConcat(std::ostream& out, const Matrix& m);

#endif



