///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "page.h"
#include "fileerrors.h"
#include "pptcore.h"
#include "config.h"
#include "globals.h"
#include "view.h"		// Todo:  Remove when possible
#include "document.h"		// Todo:  Remove if/when possible
#include <util/stringutil.h>
#include <util/warning.h>
#include <algorithm>

SigC::Signal1<void, Pagent*> Page::ready_to_draw_page_signal;

namespace {
  struct PNum_Hack { //page numbering hack
    static const bool use=false;
    static const int xpos=108; //points
    static const int ypos=28; //points
    static std::string font;
    static const int font_size=14;
  } pnum_hack;

  std::string PNum_Hack::font="Palatino-Italic";
}

int Page::get_page_num() const
{
  return document.get_page_num_of_page(this);
}

Page::Page(Document& d)
  :Group(0, ""),
   document(d)
{
}

Page::Page(const XMLNode& node, Document& d)
  :Group(0, ""),
   document(d)
{
  const XMLPropertyList properties=node.properties();
  for(XMLPropertyList::const_iterator i=properties.begin();
      i!=properties.end();
      i++)
    if((*i)->name()=="name")
      set_name((*i)->value());
    else
      warning << "Unknown property \"" << (*i)->name()
	      << "\" ignored in <page>." << std::endl;

  // read pagents:   Todo:  Avoid duplicate (groupmeta.cc:6)
  const XMLNodeList children = node.children();
  for(XMLNodeList::const_iterator i=children.begin();
      i!=children.end();
      i++)
    {
      std::string name=(*i)->name();
      if(name=="frame")
	{
	  const XMLProperty* type_node = (*i)->property("type");
	  if(!type_node)
	    throw Error::Read("No type attribute for frame");
	  
	  const std::string type = type_node->value();
	 
	  if(MetaBase* loader = core.getMeta(type))
	    add(loader->load(*i, this));
	  
	  else
	    throw Error::Read("No appropriate loader for  \"" + type + "\".");
	}
      else if(!name.empty()) 
	warning << "Unknown tag \"" << name
		<< "\" ignored in <page>." << std::endl;
    }
}

Page::~Page()
{}

XMLNode *Page::save()
{
  XMLNode *node=new XMLNode("page");
  saveChilds(*node);
  if(!name.empty())
    node->add_property("name", name);
  return node;
}

void Page::print(std::ostream &out)
{
  // Page numbering hack:
  if(pnum_hack.use)
    {
      int page_num=get_page_num();
      bool odd=page_num%2;
      int xpos=odd?(int) get_width()-pnum_hack.xpos:pnum_hack.xpos;
      int ypos=pnum_hack.ypos;
      out << '/' << pnum_hack.font << " findfont "
	  << pnum_hack.font_size << " scalefont setfont" << std::endl
	  << xpos << ' ' << ypos << " moveto (" << page_num << ") show"
	  << std::endl;
    }

  Group::print(out);
  
  out << std::endl << "showpage"
      << std::endl << std::endl;
}

std::string Page::get_name()
{
  if(name.empty())
    return tostr(get_page_num());
  else
    return name;
}

FLength Page::get_width() const
{
  return document.get_width();
}

FLength Page::get_height() const
{
  return document.get_height();
}

FLength Page::get_xpos() const
{
  return 0;
}

FLength Page::get_ypos() const
{
  return 0;
}

bool Page::draw_content(View& view)
{
  Gdk_Window &win = view.get_win();
  Gdk_GC gc = view.get_gc();
  gc.set_foreground(view.get_color(Color::bg));
  Gdk_Point origin=view.pt2scr(Vector(0, get_height()));
  win.draw_rectangle(gc, true, origin->x, origin->y, 
		     int(view.pt2scr(get_width())+0.5),
		     int(view.pt2scr(get_height())+0.5));

  // Draw page numbering hack:
  if(pnum_hack.use)
    {
      int page_num=get_page_num();
      bool odd=page_num%2;
      int xpos=odd?(int) get_width()-pnum_hack.xpos:pnum_hack.xpos;
      int ypos=pnum_hack.ypos;
      Gdk_Point pos=view.pt2scr(Vector(xpos, ypos));
      int fs=(int) view.pt2scr(pnum_hack.font_size);
      int fs30=fs*3/10;
      int fs70=fs*7/10;
      gc.set_line_attributes(1, GDK_LINE_SOLID);
      gc.set_foreground(view.get_color(Color::locked));
      win.draw_line(gc, pos->x, pos->y+fs30, pos->x+fs, pos->y+fs30);
      win.draw_line(gc, pos->x, pos->y+fs70, pos->x+fs, pos->y+fs70);
      win.draw_line(gc, pos->x+fs30, pos->y, pos->x+fs30, pos->y+fs);
      win.draw_line(gc, pos->x+fs70, pos->y, pos->x+fs70, pos->y+fs);
    }

  bool done = Group::draw_content(view); // Draws the actual content

  // Draw borders and handles after / above the content
  gc.set_line_attributes(1, GDK_LINE_ON_OFF_DASH);
  for(ChildVec::const_iterator i = pbegin(); i != pend(); i++) {
    // Todo:  Just skip this Pagent if it is selected, as those will be drawn
    // solid in the next loop.
    const Box box = (*i)->get_box();
    GdkPoint p[] = {
      view.pt2scr(box.getLL()), view.pt2scr(box.getLR()),
      view.pt2scr(box.getUR()), view.pt2scr(box.getUL()),
    };
    gc.set_foreground(view.get_color((*i)->is_locked()
				     ?Color::locked
				     :Color::frame));
    win.draw_polygon(gc, false, p, 4);
  }
  
  gc.set_line_attributes(1, GDK_LINE_SOLID);
  const Document::Selection selection = get_document().selected();
  for(Document::Selection::const_iterator i = selection.begin();
      i != selection.end(); ++i) {
    if(&getPage(**i) == this) {
      const Box box = (*i)->get_box();
      GdkPoint p[] = {
	view.pt2scr(box.getLL()), view.pt2scr(box.getLR()),
	view.pt2scr(box.getUR()), view.pt2scr(box.getUL()),
      };
      gc.set_foreground(view.get_color((*i)->is_locked()
				       ?Color::locked
				       :Color::frame));
      win.draw_polygon(gc, false, p, 4);
    }
  }
  if((selection.size() == 1)) {
    const Pagent& obj = *selection.front();
    if((&getPage(obj) == this) 
       && obj.is_reshapable() 
       && !obj.is_locked()) {
      const int size = config.getResBoxSize();
      gc.set_foreground(view.get_color(Color::frame));      
      for(unsigned int i = 1; i <= obj.num_of_reshape_points();  ++i) {
	const Gdk_Point p = view.pt2scr(obj.get_reshape_point(i));	
	win.draw_rectangle(gc, true, p->x - size/2, p->y - size/2, size, size);
      }
    }
  }
  return done;
}

void Page::act_on_zoom_factor_change()
{
  for_each(pbegin(), pend(), std::mem_fun(&Pagent::act_on_zoom_factor_change));
}

void Page::addObject(Pagent* obj) {
  add(obj);
  get_document().select(obj);
}

void Page::child_ready_to_draw(Pagent *pagent)
{
  if(pagent)
    debug << pagent->get_name() << " ready to draw itself" << std::endl;
  ready_to_draw_page_signal(this);
}

const Page& getPage(const Pagent& obj) {
  if(const Page* result = dynamic_cast<const Page*>(&obj))
    return *result;
  else
    return getPage(obj.get_parent());
}
