///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pagent.h"
#include "fileerrors.h"
#include "safeslot.h"
#include "group.h"
#include "document.h"
#include <util/stringutil.h>

Error::Invalid_Page_Num::Invalid_Page_Num(int num) 
  : logic_error("Invalid page number: " + tostr(num))
{}

Box::Box(const float& width, const float& height)
  : m(Matrix::scaling(width, height))
{}

Box& Box::operator *= (const Matrix& transform) {
  m *= transform;
  return *this;
}

Vector Box::getLL() const { return m.transform(Vector(0, 0)); }
Vector Box::getLR() const { return m.transform(Vector(1, 0)); }
Vector Box::getUL() const { return m.transform(Vector(0, 1)); }
Vector Box::getUR() const { return m.transform(Vector(1, 1)); }

void Pagent::connect_to_parent()
{
  // Anything that happens to a pagent, happens to its parent.
  // It is simpler to put this in pagent than in group, because
  // a group would need to maintain a list of connections.
  // Note: we will get multible implicit signals.
  if(parent)
    {
      draw_connection=ready_to_draw_signal.connect(slot(*parent, &Group::child_ready_to_draw));
      geometry_connection=geometry_changed_signal.connect(slot(*parent, &Group::child_geometry_changed));
      props_connection=props_changed_signal.connect(slot(*parent, &Group::child_props_changed));
    }
  //Todo: why doesn't safeslot work here?
}

Pagent::Pagent(Group *parent_, const std::string& name_)
  :parent(parent_), reshapable(true), reshaping(false), 
  locked(false), flow_around(true), name(name_) 
{
  // a geometry change implies a properties change
  geometry_changed_signal.connect(props_changed_signal.slot());
  // a geometry change does NOT imply a redraw

  connect_to_parent();
}

Pagent::~Pagent()
{}

void Pagent::reparent(Group *parent_)
{
  if(parent)
    {
      draw_connection.disconnect();
      geometry_connection.disconnect();
      props_connection.disconnect();
    }
  parent=parent_;
  connect_to_parent();
  props_changed_signal(this);
}

void Pagent::set_name(std::string name_)
{
  name=name_;
  props_changed_signal(this);
}

void Pagent::lock(bool locked_)
{
  locked=locked_;
  props_changed_signal(this);
  ready_to_draw_signal(this);
}

void Pagent::set_matrix(Matrix m) 
{
  matrix=m;
  geometry_changed_signal(this);
  ready_to_draw_signal(this);
}

unsigned int Pagent::num_of_reshape_points() const { return 8; }

Vector Pagent::get_reshape_point(int i) const {
  // Todo: This will most often get called for each legal value of i, so cache
  // the box!
  const Box box = get_box();
  switch(i) {
  case 1: return box.getUL();
  case 2: return (box.getUL() + box.getUR())/2.0;
  case 3: return box.getUR();
  case 4: return (box.getUL() + box.getLL())/2.0;
  case 5: return (box.getUR() + box.getLR())/2.0;
  case 6: return box.getLL();
  case 7: return (box.getLL() + box.getLR())/2.0;
  case 8: return box.getLR();
  default:
    throw std::runtime_error("get_reshape_point reached default");
  }
}

bool Pagent::inside_select_area(const Vector& v, const float& d) const {
  const Box box = get_box();
  const Vector i = box.into(v);
  const float d2 = dist(box.into(Vector(d, 0)), box.into(Vector(0, 0)));
  return ((i.x >= -d2 && i.x <= 1+d2 && i.y >= -d2 && i.y <= 1+d2));
}

int Pagent::inside_reshape_box(const Vector& v, const float& d) const {
  for(unsigned int i = 1; i <= num_of_reshape_points(); ++i) {
    if(dist(v, get_reshape_point(i)) < d)
      return i;
  }
  return 0;
}

void Pagent::end_reshape(bool revert) 
{
  reshaping=false;
  if(revert) 
    matrix=omatrix;
  else
    geometry_changed_signal(this);
}

void Pagent::setFlowAround(bool flow_around_)
{
  flow_around=flow_around_;
  props_changed_signal(this);
}

int Pagent::get_page_num() const {
  if(parent)
    return parent->get_page_num();
  else
    throw Error::No_Parent();
}

std::string Pagent::to_relative(std::string filename) const
{
  return get_document().to_relative(filename);
}

std::string Pagent::from_relative(std::string filename) const
{
  return get_document().from_relative(filename);
}

Group& Pagent::get_parent() {
  if(!parent) 
    throw Error::No_Parent(); 
  else
    return *parent;
}

const Group& Pagent::get_parent() const {
  if(!parent) 
    throw Error::No_Parent(); 
  else
    return *parent;
}

Document& Pagent::get_document() 
{
  if(!parent) 
    throw Error::No_Parent(); 
  return parent->get_document();
}

const Document& Pagent::get_document() const {
  if(!parent) 
    throw Error::No_Parent(); 
  return parent->get_document();
}

