///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
// -*- c++ -*-
#ifndef PAGENT_H
#define PAGENT_H

// A "pagent" is anything you can put on a page. 

// A Pagent (as opposed to a Basic_Frame) isn't necessarily rectangular 
// and doesn't have a well defined position (it might be a group).

#include "units.h"
#include "matrix.h"
#include "boundary.h"
#include "view.h"
#include <iostream>
#include <stdexcept>
#include <sigc++/basic_signal.h>

class Document;
class Group;
class XMLNode;

namespace Error
{
  struct No_Parent : public std::logic_error {
    No_Parent() : logic_error("Pagent has no parent") {}
  };
  struct Invalid_Page_Num : public std::logic_error {
    Invalid_Page_Num() : logic_error("Invalid page number") {}
    Invalid_Page_Num(int num);
  };
}

// A box in the document, in the coordinates of the page or group that the box
// is in.
class Box {
public:
  Box(const float& width, const float& height);
  
  // Apply a transform on this box.
  Box& operator *= (const Matrix& transform);

  // Note: If the transform includes rotation, these names might be strange.
  Vector getLL() const;
  Vector getLR() const;
  Vector getUL() const;
  Vector getUR() const;

  // Transform "outside" vector v into this box coordinates.  Points inside
  // the box gets coorinates inside the unit box.
  Vector into(const Vector& v) const { return m.inv().transform(v); }
private:
  Matrix m;
};

// Definition: Pagent = Page Component
class Pagent: public SigC::Object
{
public:
  Pagent(Group *parent_, const std::string& name_);
  virtual ~Pagent();
  virtual XMLNode *save()=0;
  virtual void print(std::ostream &out)=0; // should throw Print_Error

  virtual void reparent(Group *parent_);

  virtual void set_name(std::string name_);
  virtual std::string get_name()
  {return name;}

  // a locked pagent can not be moved or reshaped with the mouse
  virtual void lock(bool locked_);
  virtual bool is_locked() const
  {return locked;}

  virtual void set_matrix(Matrix m);
  virtual const Matrix& get_matrix() const {return matrix;} 
  //  virtual Matrix& get_matrix() { return matrix; }
  
  // Returns true if the content is drawn completley, false if something
  // remains.  On false, it is the callers responsibility to make sure the
  // pagent is redrawn again later.
  virtual bool draw_content(View& view)=0;

  // Release memory needed to display pagent
  // when pagent is no longer visible
  virtual void clear_cache() {} 

  virtual Box get_box() const =0;
  virtual unsigned int num_of_reshape_points() const;
  virtual Vector get_reshape_point(int i) const;
  virtual bool inside_select_area(const Vector& v, const float& dist) const;
  virtual int inside_reshape_box(const Vector& v, const float& dist) const;
  
  bool is_reshapable() const {return reshapable;}
  
  // This begin/end stuff is the begining of a "revert operation" facility
  virtual void begin_reshape() {omatrix=matrix; reshaping=true;}
  virtual void end_reshape(bool revert);
  bool is_reshaping() const {return reshaping;}
  virtual void move_reshape_box(int box, Vector v)=0;
  virtual void setFlowAround(bool flow_around_);
  virtual bool getFlowAround() const {return flow_around;}
  virtual Boundary* getObstacleBoundary() const = 0;

  virtual int get_page_num() const; // Ask the page what number it is
  // May throw Error::Invalid_Page_Num if there is a problem 
  // with the pages in the document

  std::string to_relative(std::string filename) const;
  std::string from_relative(std::string filename) const;

  virtual void act_on_zoom_factor_change()=0;

  const Group& get_parent() const;
  Group& get_parent();
  
  virtual const Document& get_document() const;
  virtual Document& get_document();

  SigC::Signal1<void, Pagent*> ready_to_draw_signal; 
  // indicated to owner that it might want to redraw the pagent
  SigC::Signal1<void, Pagent*> props_changed_signal; 
  // anything has changed, anything at all, 
  // including position and shape
  SigC::Signal1<void, Pagent*> geometry_changed_signal; 
  // the pagent has changed shape / position
protected:
  Group *parent;
  bool reshapable, reshaping, locked, flow_around;
  Matrix matrix, omatrix;
  std::string name;

  // connections to parent
  SigC::Connection draw_connection, geometry_connection, props_connection;
  void connect_to_parent();
private:  
  // Undefined ctors, avoid defaults
  Pagent(const Pagent&);
  Pagent();
  void operator = (const Pagent&);
};

#endif
