///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pixbuftransformator.h"
#include "safeslot.h"
#include <util/warning.h>
#include <gtk--/main.h>
#include <algorithm>		// just min/max

PixbufTransformator::PixbufTransformator(const Pixbuf& src, const View& view,
					 const Matrix& xform)
  :srcpix(src), cur_y(-1), is_done(false)
{
  const int src_w = srcpix.get_width(), src_h = srcpix.get_height();
  Gdk_Point corner[] = {
    view.pt2scr(xform.transform(Vector(0, 0))),
    view.pt2scr(xform.transform(Vector(0, src_h))),
    view.pt2scr(xform.transform(Vector(src_w, 0))),
    view.pt2scr(xform.transform(Vector(src_w, src_h)))
  };
  using std::min; using std::max;
  min_x = min(min(corner[0]->x, corner[1]->x), 
	      min(corner[2]->x, corner[3]->x));
  max_x = max(max(corner[0]->x, corner[1]->x), 
	      max(corner[2]->x, corner[3]->x));
  min_y = min(min(corner[0]->y, corner[1]->y), 
	      min(corner[2]->y, corner[3]->y));
  max_y = max(max(corner[0]->y, corner[1]->y), 
	      max(corner[2]->y, corner[3]->y));
  
  buf = Pixbuf(max_x - min_x, max_y - min_y);
  debug << "Creating PixbufTransformator.  " << src_w << 'x' << src_h
	<< " => " << buf.get_width() << 'x' << buf.get_height() << std::endl;

  const Matrix inv = xform.inv();
  lstart = inv.transform(view.scr2pt(Gdk_Point(min_x, min_y)));
  d_pix  = (inv.transform(view.scr2pt(Gdk_Point(min_x + 1, min_y))) - lstart);
  d_line = (inv.transform(view.scr2pt(Gdk_Point(min_x, min_y + 1))) - lstart);
  
  Gtk::Main::idle.connect(safeslot(this, &PixbufTransformator::workLine));
}

PixbufTransformator::~PixbufTransformator() {}

gint PixbufTransformator::workLine() {
  guchar* todata = buf.get_pixels();
  if(!todata)
    throw std::runtime_error("No todata for pixbuftransformator");
  const int to_bytes_per_row = buf.get_rowstride();
  const int to_bytes_per_pixel = buf.get_n_channels();

  guchar* srcdata = srcpix.get_pixels();
  if(!srcdata) {
    warning << "No srcdata!" << std::endl;
    return false;
  }
  const int src_bytes_per_row = srcpix.get_rowstride();
  const int src_bytes_per_pixel = srcpix.get_n_channels();
  
  const int src_width  = srcpix.get_width();
  const int src_height = srcpix.get_height();

  ++cur_y;
  Vector v = lstart;
  for(gint cur_x = 0; cur_x < (max_x - min_x); ++cur_x, v += d_pix) {
    // Yucky pointer arithmetic ...
    guchar* to_pix = todata + to_bytes_per_row * cur_y 
      + to_bytes_per_pixel * cur_x;
    const int sy = int(v.y), sx = int(v.x);
    if((sx > 0) && (sy > 0) && (sx < src_width) && (sy < src_height)) {
      guchar* srcp = srcdata + src_bytes_per_row * (src_height - sy)
	+ src_bytes_per_pixel * sx;
      
      to_pix[0] = srcp[0]; to_pix[1] = srcp[1];  to_pix[2] = srcp[2];
      to_pix[3] = ((src_bytes_per_pixel == 4)? srcp[3] : 255);
      
    } else {
      // Don't matter: to_pix[0] = 0; to_pix[1] = 0; to_pix[2] = 0;
      to_pix[3] = 0;          // transparent
    }
  }
  lstart += d_line;
  is_done = !((cur_y+1) < (max_y - min_y));
  return !is_done;
}

