///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "printdialog.h"

#include <fstream>

#include <gtk--/frame.h>
#include <gtk--/table.h>

#include <util/warning.h>
#include <util/filesys.h>

#include "globals.h"
#include "config.h"
#include "fileerrors.h"
#include "docview.h"
#include "safeslot.h"
#include "processman.h"
#include "document.h"

void Print_Dialog::update()
{
  using_entry.set_sensitive(using_button.get_active());
  file_entry.set_sensitive(file_button.get_active());
  eps_button.set_sensitive(current_button.get_active());
  // An EPS file may not have more than one page
  from_spinner.set_sensitive(from_button.get_active());
  to_spinner.set_sensitive(from_button.get_active());
}
  
void Print_Dialog::show_it()
{
  Document *document=document_view.get_document();
  if(!document)
    return;
  int first=document->get_first_page_num();
  int last=first+((int) document->get_num_of_pages())-1;
  save_state(); 
  update();
  from_spinner.limits(first, last);  
  to_spinner.limits(first, last);
  from_spinner.set(first);
  to_spinner.set(last);
  show();

  // Gtk::Entry::set_position doesn't seem to work 
  // unless the entry is shown first
  std::string filename=document->get_filename();
  if(filename.empty())     // the document has not been saved 
    file_entry.entry.set_text("pptout.ps");
  else
    file_entry.entry.set_text(no_suffix(filename)+".ps");
}

void Print_Dialog::save_state()
{
  saved_using_text=using_entry.get_text();
  saved_file_text=file_entry.entry.get_text();
  saved_file_not_using=file_button.get_active();
  saved_eps=eps_button.get_active();
}

void Print_Dialog::restore_state()
{
  using_entry.set_text(saved_using_text);
  file_entry.entry.set_text(saved_file_text);
  file_button.set_active(saved_file_not_using);
  eps_button.set_active(saved_eps);
}


Print_Dialog::Print_Dialog(Gtk::Window &parent, Document_View &_document_view):
  Gtk::Dialog(), ok_button(" Ok "), cancel_button(" Cancel "), 
  using_button("Print using:"), file_button("Print to file:"),
  all_button("All"), current_button("Current"), from_button("From:"),
  eps_button("Print as EPS", 0), fonts_button("Include fonts", 0),
  from_spinner(0, false), to_spinner(0, false),
  document_view(_document_view),
  file_entry("Print To File")
{
  set_transient_for(parent);
  set_border_width(border_width);
  set_modal(true);
  // Made it nonmodal first, but I realised the user's fragile little
  // mind will make a core dump if it isn't modal.
  set_title("Print");
  set_policy(false, false, true);
  const unsigned int padding=2;

  file_button.set_group(using_button.group());

  current_button.set_group(all_button.group());
  from_button.set_group(all_button.group());
  
  get_action_area()->pack_start(ok_button, true, false, 0);
  ok_button.set_flags(GTK_CAN_DEFAULT);
  get_action_area()->pack_start(cancel_button, true, false, 0);
  cancel_button.set_flags(GTK_CAN_DEFAULT);

  using_box.pack_start(using_button, false, false, 0);
  using_box.pack_start(using_entry, true, true, 0);
    
  file_box.pack_start(file_button, true, true, 0);
  file_box.pack_start(file_entry, true, true, 0);

  //  from_spinner.set_usize(40, -1);
  //  to_spinner.set_usize(40, -1);
  from_to_box.pack_start(from_button, false, true, 0);
  from_to_box.pack_start(from_spinner, false, true, 0);
  from_to_box.pack_start(*manage(new Gtk::Label(" to: ")), false, true, 0);
  from_to_box.pack_start(to_spinner, false, true, 0);

  Gtk::Frame *pages_frame=manage(new Gtk::Frame("Pages"));
  Gtk::HBox *pages_box=manage(new Gtk::HBox());
  pages_box->pack_start(all_button, true, false, padding);
  pages_box->pack_start(current_button, true, false, padding);
  pages_box->pack_start(from_to_box, true, false, padding);
  pages_box->set_border_width(border_width);
  pages_frame->add(*pages_box);

  get_vbox()->pack_start(using_box, false, false, padding);
  get_vbox()->pack_start(file_box, false, false, padding);
  get_vbox()->pack_start(*pages_frame, false, false, padding);

  Gtk::Table *format_table=manage(new Gtk::Table(2, 1));
  format_table->attach(eps_button, 0, 1, 0, 1);
  format_table->attach(fonts_button, 1, 2, 0, 1);
  get_vbox()->pack_start(*format_table, true, false, padding);


  get_vbox()->show_all();
  get_action_area()->show_all();

  using_entry.set_text(config.getPrintCmd());
  //  file_entry.entry.set_text("foo.ps");
  using_button.set_active(true);

  using_button.clicked.connect(safeslot(this, &Print_Dialog::update));
  // If there are only two buttons in a group, you only need to connect one.
  // If there are more, you need to connect them all. That is annoying.
  all_button.clicked.connect(safeslot(this, &Print_Dialog::update));
  current_button.clicked.connect(safeslot(this, &Print_Dialog::update));
  from_button.clicked.connect(safeslot(this, &Print_Dialog::update));

  ok_button.clicked.connect(safeslot(this, &Print_Dialog::ok));
  cancel_button.clicked.connect(safeslot(this, &Print_Dialog::cancel));
}

void Print_Dialog::ok()
{
  if(fonts_button.get_active())
    throw Error::Print("Font inclusion\n not yet implemented");
  Document *document=document_view.get_document();
  if(document)
    {
	  int first=document->get_first_page_num();
	  int last=first+document->get_num_of_pages()-1;
	  if(current_button.get_active())
	    first=last=document_view.get_current_page_num();
	  else if(from_button.get_active())
	    {
	      first=(int) from_spinner.get();
	      last=(int) to_spinner.get();
	      if(last<first)
		throw Error::Print("\"From\" page number must be lower\n"
				   "than \"To\" page number");
	      // We could make this error impossible to cause, 
	      // but I don't like putting leash and collar on the user.
	    }
	  // else all_button is active
	  if(file_button.get_active()) {
	    std::ofstream out(file_entry.entry.get_text().c_str());
	    if(!out) throw Error::Print("Could not print to file:\n"
					+ file_entry.entry.get_text());
	    document->print(out, first, last, eps_button.get_active());
	  } else {
	    std::string tmp_file=process_manager.find_new_name();
	    std::ofstream out(tmp_file.c_str());
	    if(!out) throw Error::Print("Could not print to file:\n"
					+ tmp_file);
	    document->print(out, first, last, eps_button.get_active());
	    int status=process_manager.system(using_entry.get_text()
					      + " < " + tmp_file);
	    if(status==-1)
	      throw Error::Print("Could not print to:\n"
				 + using_entry.get_text());
	    if(!process_manager.delete_file(tmp_file))
	      warning << "Failed to delete " << tmp_file << std::endl;
	  }
	  hide();
    }
}

void Print_Dialog::cancel()
{
  restore_state();
  hide();
}
