///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "propertiesdialog.h"
#include "propbase.h"
#include <gtk--/label.h>
#include <gtk--/separator.h>
#include <gtk--/entry.h>
#include <gtk--/checkbutton.h>
#include <algorithm>
#include "matrix.h"
#include "safeslot.h"
#include "globals.h"
#include "pptcore.h"
#include "widget/spinner.h"
#include "basicframe.h"
#include <util/stringutil.h>

// The basic properties that apply to all Pagent's.
// Currently that means some kind of name, and the position and size of the
// object.
class PropBasic : public PropBase {
public:
  // | name:  | "name of object"             |
  // -----------------------------------------
  // | left:  | "left"  | width:  | "width"  |
  // | right: | "right" | height: | "height" |
  PropBasic()
    :PropBase("Basic", 4, 7), object(0),
     l_name("Object:"),
     l_left("Left:"), l_bottom("Bottom:"), 
     l_width("Width:"), l_height("Height:"),
     c_flow("Flow around", 0), c_locked("Locked", 0)
    {
      static const unsigned int min_size=80;

      e_name.set_editable(false);
      attach(l_name,   0, 1, 0, 1);
      right_justify(l_name);      
      attach(e_name,   1, 4, 0, 1);
      attach(c_locked, 0, 2, 1, 2);
      attach(hsep1,    0, 4, 2, 3);
      
      attach(l_left,   0, 1, 3, 4);
      right_justify(l_left);      
      attach(e_left,   1, 2, 3, 4);
      e_left.set_usize(min_size, -1);
      attach(l_bottom, 0, 1, 4, 5);
      right_justify(l_bottom);      
      attach(e_bottom, 1, 2, 4, 5);
      e_bottom.set_usize(min_size, -1);
      
      attach(l_width,  2, 3, 3, 4);
      right_justify(l_width);      
      attach(e_width,  3, 4, 3, 4);
      e_width.set_usize(min_size, -1);
      attach(l_height, 2, 3, 4, 5);
      right_justify(l_height);      
      attach(e_height, 3, 4, 4, 5);
      e_height.set_usize(min_size, -1);
      
      attach(*manage(new Gtk::Label("Rotate:")), 0, 1, 5, 6);
      attach(e_rotate, 1, 2, 5, 6);
      e_rotate.set_usize(min_size, -1);
      attach(hsep2,    0, 4, 6, 7);
      
      attach(c_flow,   0, 2, 7, 8);

      set_sensitive(false);
    }
  void setObject(Pagent* pagent) {
    object = pagent;
    set_sensitive(object);
    if(is_visible())
      update();
  }
  
  void update() {
    if(!object) {
      e_name.set_text("No object selected");
    } else {
      e_name.set_text(object->get_name());

      c_locked.set_active(object->is_locked());
      
      float width = 1, height = 1;
      if(const Basic_Frame* bf = dynamic_cast<const Basic_Frame*>(object)) {
	width = bf->get_width();
	height = bf->get_height();
      }
      // Note: This is not correct!  It might be considered a placeholder, or
      // it might be corrected by changing Matrix and the page objects.
      const Matrix matrix(object->get_matrix());
      const Vector low(matrix.transform(Vector(0, 0)));
      const Vector high(matrix.transform(Vector(width, height)));
      
      e_left.set(low.x);
      e_bottom.set(low.y);
      e_width.set(high.x - low.x);
      e_height.set(high.y - low.y);
      e_rotate.set(Matrix::rad2deg(matrix.rot()));

      if(Basic_Frame* bf = dynamic_cast<Basic_Frame*>(object)) {
	c_flow.set_active(bf->getFlowAround());
      }
    }
  }
  void apply() {
    // set_matrix will send a signal that resets the fields, 
    // so we have to save them first:
    float width=e_width.get();
    float height=e_height.get();
    bool locked=c_locked.get_active();
    bool flow=c_flow.get_active();

    Matrix m=object->get_matrix();
    m.set_rot(Matrix::deg2rad(e_rotate.get()));
    m.set_tr_x(e_left.get()); m.set_tr_y(e_bottom.get());
    if(Basic_Frame* bf = dynamic_cast<Basic_Frame*>(object)) {
      m.set_scale(width / bf->get_width(),
		  height / bf->get_height());
    }

    object->set_matrix(m);
    object->lock(locked);
    object->setFlowAround(flow);
  }

private:
  Pagent* object;
  Gtk::HSeparator hsep1, hsep2;
  Gtk::Label l_name, l_left, l_bottom, l_width, l_height;
  Gtk::Entry e_name;
  Spinner e_left, e_bottom, e_width, e_height, e_rotate;
  Gtk::CheckButton c_flow, c_locked;
};

// - - - - back to the actual PropertiesDialog implementation - - - -

PropertiesDialog::PropertiesDialog():
  document(0),
  apply_button(" Apply "), close_button(" Hide ")
{
  set_title("Object properties");
  set_border_width(border_width);
  get_vbox()->pack_start(book, true, true, 0);

  PropBase* prop = new PropBasic;
  pages.push_back(prop);
  book.append_page(*prop, prop->getLabel());
  for(PptCore::MetaMap::const_iterator
	i = core.m_begin(); i != core.m_end(); ++i) {
    if(PropBase* prop = i->second->getProp()) {
      pages.push_back(prop);
      book.append_page(*prop, prop->getLabel());
    }
  }
  
  get_action_area()->pack_start(apply_button, false, false);
  get_action_area()->pack_start(close_button, false, false);
  apply_button.clicked.connect(safeslot(this, &PropertiesDialog::apply));
  close_button.clicked.connect(safeslot(this, &PropertiesDialog::close));

  // listen to selection change signals:
  Document::selection_changed_signal.connect(slot(this, &PropertiesDialog::select_change));
}

void
PropertiesDialog::showRaise() {
  show_all();
  Gdk_Window win(get_window());
  win.show();			// uniconify if necessary
  win.raise();			// raise if necessary
  
  update();
}

void
PropertiesDialog::close() {
  hide();
}

void
PropertiesDialog::apply() {
  if(document)
    for_each(pages.begin(), pages.end(), std::mem_fun(&PropBase::apply));
}

void
PropertiesDialog::set_document(Document *document_)
{
  document=document_;
  update();
}

void
PropertiesDialog::update() {
  Document::Selection all_selected;
  if(document) 
    all_selected = document->selected();

  // Note: If there is more than one object selected, no properties are shown.
  // Maybe we should show the properties that are common
  // to all selected objects.
  Pagent* pagent = (all_selected.size()!=1 ? 0
		    : all_selected.front());

  apply_button.set_sensitive(pagent);

  for(std::vector<PropBase*>::const_iterator i=pages.begin();
      i!=pages.end(); ++i)
    (*i)->setObject(pagent);
}

void
PropertiesDialog::select_change(Document* doc) {
  if(doc==document)
    update();
}
