///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "rasterframe.h"
#include <util/stringutil.h>
#include <util/filesys.h>
#include <util/warning.h>
#include <iomanip>
#include <xml++.h>
#include <ps/encode.h>
#include <ctime> // for profiling
#include <fstream>

Raster_Frame::Raster_Frame(Group *parent, const XMLNode& node)
  :Basic_Frame(parent, node), worker(0)
{
  reshapable=false;

  if(!node.property("type") || node.property("type")->value()!="raster")
    throw Error::Frame_Type("Bad raster-frame type: "
			    + node.property("type")->value());

  if(const XMLProperty* w = node.property("width"))
    width = to<float>(w->value());
  else
    throw Error::Read("No \"width\" tag found in raster frame.");
  if(const XMLProperty* h = node.property("height"))
    height = to<float>(h->value());
  else
    throw Error::Read("No \"height\" tag found in raster frame.");

  if(const XMLProperty* file = node.property("file"))
    {
      try // be a little forgiving
	{ 
	  set_association(from_relative(file->value()));
	}
      catch (Error::Read e)
	{
	  warning << e.message << std::endl;
	}
    }
  else
    throw Error::Read("\"file\" property missing in Raster_Frame");
}

Raster_Frame::Raster_Frame(Group* parent, const std::string& filename)
  :Basic_Frame(parent, 1, 1, "Raster " + basename(filename)),
   association(filename), filepixbuf(association), worker(0)
{
  reshapable=false;

  if(!filepixbuf)
    throw Error::Read("Can't load image file "+association);
  // Seems like gdk-pixbuf won't tell us _what_ the error is.
    
  loadUnitSize();
  // Assume 72ppi to begin with ...
  width = unit_size->x; height = unit_size->y;
}

Raster_Frame::~Raster_Frame() {}

void Raster_Frame::loadUnitSize() {
  if(!filepixbuf)
    throw Error::Read("Cannot load unit size from non-existing image");
  
  // Todo: care for what the image says about its resolution?
  unit_size = Gdk_Point(filepixbuf.get_width(), filepixbuf.get_height());
}

void Raster_Frame::file_modified()
{
  filepixbuf.load_file(association);
  if(filepixbuf)
    loadUnitSize();
  screenpixbuf.unref();
  ready_to_draw_signal(this);
}

bool Raster_Frame::draw_content(View& view) 
{
  const Gdk_Point sz(int(view.pt2scr(width)), int(view.pt2scr(height)));

  if(association.empty())
    draw_content_missing(view);

  if(sz->x==0 || sz->y==0)
    return true;

  if(!screenpixbuf)
    {
      if(!filepixbuf)
	{
	  draw_content_broken(view);
	  return true;
	}
      if(!worker) {
	if(get_matrix().is_plain()) {
	  screenpixbuf = filepixbuf.scale(sz->x, sz->y);
	} else {
	  worker = new PixbufTransformator
	    (filepixbuf, view, 
	     Matrix::scaling(double(width)/filepixbuf.get_width(),
			     double(height)/filepixbuf.get_height()) 
	     * get_matrix());
	  screenpixbuf = worker->getPixbuf();
	}
      } else {
	screenpixbuf = worker->getPixbuf();
      }
    }

  bool done = true;
  if(worker) {
    if(worker->isDone()) {
      delete worker;
      worker = 0;  done = true;
    } else
      done = false;
  }
  const Matrix m = Matrix::scaling(width/sz->x, height/sz->y) * get_matrix();
  const Gdk_Point corner[] = {
    view.pt2scr(m.transform(Vector(0, 0))),
    view.pt2scr(m.transform(Vector(0, sz->y))),
    view.pt2scr(m.transform(Vector(sz->x, 0))),
    view.pt2scr(m.transform(Vector(sz->x, sz->y)))
  };
  using std::min;
  const int min_x = min(min(corner[0]->x, corner[1]->x), 
			min(corner[2]->x, corner[3]->x));
  const int min_y = min(min(corner[0]->y, corner[1]->y), 
			min(corner[2]->y, corner[3]->y));
  screenpixbuf.render_to_drawable_alpha(view.get_win(), 0, 0, min_x, min_y);
  return done;
}

void Raster_Frame::act_on_zoom_factor_change() {
  screenpixbuf.unref();
}

XMLNode *Raster_Frame::save()
{
  XMLNode *node=Basic_Frame::save();
  node->add_property("type", "raster");
  node->add_property("file", to_relative(association));

  node->add_property("width", tostr(width));
  node->add_property("height", tostr(height));

  return node;
}

void Raster_Frame::print(std::ostream &out)
{
  if(!filepixbuf) {
    out << "% - - - no raster data to print for " << get_name() << std::endl;
    warning << "No raster data to print for " << get_name() << std::endl;
  } else {
    assert(filepixbuf.get_colorspace() == GDK_COLORSPACE_RGB);
    
    Pixbuf::Statistics statistics=filepixbuf.calc_stats();

    // Todo: config option for filter, and clever filter selection.
    PS::EncodeFilter *filter;
    PS::ASCIIHexEncodeFilter filterHex;
    PS::ASCII85EncodeFilter filter85;
    PS::RunLengthEncodeFilter filterRLE;
    PS::Filter_Sequence sequence;
    sequence.push_back(&filter85);
    if(statistics.suggest_rle)
      sequence.push_back(&filterRLE);
    PS::EncodeFilterCascade filterCascade(sequence);
    filter=&filterCascade;

    using std::endl;
    out << "% Raster image: " << basename(association) << endl
	<< "save" << endl;
    psConcat(out, get_matrix());
    out << (statistics.grayscale ? "/DeviceGray" : "/DeviceRGB")
	<< " setcolorspace" << endl
	<< width << ' ' << height << " scale" << endl
	<< "<<" << endl
	<< "  /ImageType 1"
	<< "  /Width " << unit_size->x
	<< "  /Height " << unit_size->y << endl
	<< "  /BitsPerComponent 8" << endl
	<< (statistics.grayscale ? "  /Decode [0 1]" : "  /Decode [0 1 0 1 0 1]")
	<< endl
	<< "  /ImageMatrix ["
	<< unit_size->x << " 0 0 " << -unit_size->y << " 0 "
	<< unit_size->y << ']' << endl
	<< "  /DataSource currentfile" << endl
	<< "  " << filter->decode_command() << endl
	<< ">> image" << endl;
    
    std::clock_t start = std::clock();
    // Dump the actual image data to postscript.
    guchar* data = filepixbuf.get_pixels();
    const int bytes_per_row = filepixbuf.get_rowstride();
    const int bytes_per_pixel = filepixbuf.get_n_channels();
    filter->begin(&out); // intialize filter
    for(int y = 0; y < unit_size->y; ++y)
      for(int x = 0; x < unit_size->x; ++x) {
	// Yucky pointer arithmetic ...
	guchar* pixel = data + bytes_per_row * y + bytes_per_pixel * x;
	// Note: This assumes that the 3 first channels are RGB, which seem to
	// be ok currently, even grayscale files are loaded to RGB pixbufs.

	// write one/three bytes:
	filter->write(pixel, statistics.grayscale ? 1 : 3); 
	// Todo; write more than 3 at a time to speed it up
      }
    filter->end(); // finalize filter
    verbose << "Time to print " << basename(association) << ": "
	    << (std::clock()-start)/1000 << " ms ("
	    << (statistics.grayscale ? "grayscale" : "color")
	    << (statistics.suggest_rle ? ", RLE" : "") << ")" << std::endl;
    out << endl << "restore" << endl;
  }
}

void Raster_Frame::set_association(const std::string& s)
{
  association=s;
  screenpixbuf.unref();
  filepixbuf.load_file(association);
  if(!filepixbuf)
    throw Error::Read("Couldn't read image file "+association);
  loadUnitSize();
  ready_to_draw_signal(this);
}

Vector Raster_Frame::get_ppi() {
  return Vector(72 * unit_size->x / width, 72 * unit_size->y / height);
}
void Raster_Frame::set_ppi(const Vector& ppi) {
  width  = 72 * unit_size->x / ppi.x;
  height = 72 * unit_size->y / ppi.y;
  screenpixbuf.unref();
  geometry_changed_signal(this);
  ready_to_draw_signal(this);
}
