///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "streamdialog.h"

#include <gtk--/table.h>
#include <vector>

#include <util/warning.h>

#include "config.h"
#include "safeslot.h"
#include "globals.h"
#include "document.h"
#include "textstream.h"

// *** Stream_Properties_Dialog Methods:

Stream_Properties_Dialog::Stream_Properties_Dialog(Gtk::Window &parent):
  Gtk::Dialog(), ok_button(" OK "), cancel_button(" Cancel "), 
  file_entry("Associated File"),
  xfrm_entry("XSL Stylesheet", config.getPath("XSLTPath")+"/"),
  document(0)
{
  set_transient_for(parent);
  get_vbox()->set_border_width(border_width);
  set_modal(true);
  //  set_policy(false, false, true);

  get_action_area()->pack_start(ok_button, true, false, 0);
  ok_button.set_flags(GTK_CAN_DEFAULT);
  get_action_area()->pack_start(cancel_button, true, false, 0);
  cancel_button.set_flags(GTK_CAN_DEFAULT);

  Gtk::Label *name_label=manage(new Gtk::Label("Name: "));
  Gtk::Label *file_label=manage(new Gtk::Label("File: "));
  Gtk::Label *xfrm_label=manage(new Gtk::Label("Stylesheet: "));
  name_label->set_alignment(1, name_label->get_yalign());
  file_label->set_alignment(1, file_label->get_yalign());
  file_entry.set_usize(300, -1);
  xfrm_label->set_alignment(1, xfrm_label->get_yalign());
  
  Gtk::Table *table=manage(new Gtk::Table(2, 3));
  
  table->set_row_spacings(3);
  table->attach(*name_label, 0, 1, 0, 1);
  table->attach(*file_label, 0, 1, 1, 2);
  table->attach(*xfrm_label, 0, 1, 2, 3);
  table->attach(name_entry, 1, 2, 0, 1);
  table->attach(file_entry, 1, 2, 1, 2);
  table->attach(xfrm_entry, 1, 2, 2, 3);

  get_vbox()->pack_start(*table, true, false, 5);

  get_vbox()->show_all();
  get_action_area()->show_all();

  ok_button.clicked.connect(safeslot(this, &Stream_Properties_Dialog::ok));
  cancel_button.clicked.connect(safeslot(this, 
					 &Stream_Properties_Dialog::cancel));
}

void Stream_Properties_Dialog::show_it(Document *document_, std::string name)
{
  if(!document_)
    return;

  document=document_;
  bool add=name.empty();
  set_title(add ? "Add Stream" : "Edit Stream");
  if(!document)
    return;
  stream = add ? 0 : document->get_text_stream(name);
  if(!add && !stream)
    throw std::runtime_error("Text stream \""+name+"\" not found");
  show();

  // Gtk::Entry::set_position doesn't seem to work 
  // unless the entry is shown first
  name_entry.set_text(add ? document->make_up_new_name() : name);
  file_entry.entry.set_text(add ? "" : stream->get_association());
  xfrm_entry.entry.set_text(add ? "" : stream->get_transform());

  if(add)
    ok_button.grab_default();
  else
    cancel_button.grab_default();

}

void Stream_Properties_Dialog::ok()
{
  if(name_entry.get_text().empty())
    {
      Error_Dialog::view("The stream has to have a name");
      return;
    }
  try
    {
      if(stream) // edit
	{
	  document->rename_text_stream(stream->get_name(), 
				       name_entry.get_text());
	  stream->set_association(file_entry.entry.get_text());
	  stream->set_transform(xfrm_entry.entry.get_text());
	  // stream->set_association does not throw a streams_changed_signal
	  // which is necessary if only the association was changed
	  Document::streams_changed_signal(document);
	}
      else //add
	document->add_text_stream(file_entry.entry.get_text(), 
				  name_entry.get_text(), 
				  xfrm_entry.entry.get_text());
    }
  catch(Error::Text_Stream_Name)
    {
      Error_Dialog::view("A stream with the name \"" 
			 + name_entry.get_text() 
			 + "\" exists already.\n"
			 + "Names must be unique.");
      return;
    }
  hide();
}

void Stream_Properties_Dialog::cancel()
{
  hide();
}

// *** Stream_Dialog Methods:

void Stream_Dialog::update(Document *document_)
{
  if(document_!=document)
    return;

  stream_list.clear();
  if(document)
    {     
      Document::StreamVec tmp=document->get_text_streams();
      for(Document::StreamVec::iterator i=tmp.begin();
	  i!=tmp.end();
	  i++)
	{
	  std::vector<Gtk::string> row;
	  row.push_back((*i)->get_name());
	  row.push_back((*i)->get_association());
	  row.push_back((*i)->get_transform());
	  stream_list.rows().push_back(row);
	}
    }

  button_box->set_sensitive(document);
  scroller.set_sensitive(document);
}

void Stream_Dialog::hide_action()
{
  hide();
}

void Stream_Dialog::delete_action()
{
  const Gtk::CList_Helpers::SelectionList &selection=stream_list.selection();
  if(document && 
     selection.begin()!=selection.end() && 
     selection.begin()->begin()!=selection.begin()->end())
    {
      document->remove_text_stream(selection.begin()->begin()->get_text());
      update(document);
    }
}

void Stream_Dialog::add_action()
{
  prop_dialog->show_it(document);
}

void Stream_Dialog::edit_action()
{
  const Gtk::CList_Helpers::SelectionList &selection=stream_list.selection();
  if(document && 
     selection.begin()!=selection.end() && 
     selection.begin()->begin()!=selection.begin()->end())
    prop_dialog->show_it(document,
			 stream_list.selection().begin()->begin()->get_text());
}

static const gchar *titles[] = { "Name", "File", "Stylesheet", NULL };

namespace{
  void foo(int a, int b, GdkEvent *c) {debug << "Ekki" << std::endl;}
  void bar(int a, int b, GdkEvent *c) {debug << "Patang!" << std::endl;}
}
 
Stream_Dialog::Stream_Dialog():
  stream_list(titles),
  add_button(" New "), edit_button(" Edit "), 
  delete_button(" Delete "), hide_button(" Hide "),
  document(0)
{
  prop_dialog=manage(new Stream_Properties_Dialog(*this));

  get_vbox()->set_border_width(border_width);
  set_modal(false);
  set_title("Text Streams");
  set_default_size(450, 300);
  //  set_policy(false, false, true);
  scroller.set_policy(GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  scroller.add(stream_list);

  button_box=manage(new Gtk::HBox());
  button_box->pack_start(add_button, true, true, 0);
  button_box->pack_start(edit_button, true, true, 0);
  button_box->pack_start(delete_button, true, true, 0);

  get_vbox()->pack_start(scroller, true, true, 0);
  get_vbox()->pack_start(*button_box, false, false, 0);

  get_action_area()->pack_start(hide_button, false, false, 0);

  hide_button.clicked.connect(safeslot(this, &Stream_Dialog::hide_action));
  add_button.clicked.connect(safeslot(this, &Stream_Dialog::add_action));
  edit_button.clicked.connect(safeslot(this, &Stream_Dialog::edit_action));
  delete_button.clicked.connect(safeslot(this, &Stream_Dialog::delete_action));

  // if they don't document it, you'll have to try everything ...
  stream_list.select_row.connect(SigC::slot(foo));
  stream_list.unselect_row.connect(SigC::slot(bar));

  Document::streams_changed_signal.connect(slot(this, 
						&Stream_Dialog::update));

  stream_list.set_column_width (0, 100);
  stream_list.set_column_width (1, 150);
  stream_list.set_column_width (2, 150);
}

void Stream_Dialog::showRaise() 
{
  show_all();
  Gdk_Window win(get_window());
  win.show();			// uniconify if necessary
  win.raise();			// raise if necessary
}

void Stream_Dialog::set_document(Document *document_)
{
  document=document_;
  update(document);
}

//*** Stream_Menu methods ***

namespace{
  static const std::string no_stream("( No Stream )"); 
}

Stream_Menu::Stream_Menu()
{
  update();
}

void Stream_Menu::set_document(Document *document_) 
{
  document=document_;
  update();
}

std::string Stream_Menu::get_stream() const 
{
  Gtk::Label *tmp=dynamic_cast<Gtk::Label*> (get_child());
  if(!tmp || tmp->get_text()==no_stream)
    return ""; 
  return tmp->get_text();
}
 
void Stream_Menu::update(std::string select_name) 
{
  Gtk::Menu dummy;
  set_menu(dummy); //so it will notice the change
  using namespace Gtk;
  using namespace Menu_Helpers;
  MenuList& menu_list=menu.items();
  menu_list.clear();
  if(!document)
    {
      menu.items().push_back(MenuElem("")); // cosmetic fix
      menu.set_active(0);
      set_menu(menu);
      return;
    }
  int index=0;
  Document::StreamVec streams=document->get_text_streams();
  for(Document::StreamVec::iterator i=streams.begin();
      i!=streams.end();
      i++)
    {
      string name=(*i)->get_name();
      menu.items().push_back(MenuElem(name));
      if(name==select_name)
	menu.set_active(index);
      index++;
    }
  menu.items().push_back(SeparatorElem());
  menu.items().push_back(MenuElem(no_stream));
  if(select_name.empty())
    menu.set_active(index+1);
  set_menu(menu);
}

