///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <strstream>
#include <fstream>

#include <xml++.h>

#include "textframe.h"
#include "document.h"
#include "processman.h"
#include "config.h"
#include "globals.h"
#include <util/warning.h>
#include <util/stringutil.h>


Text_Frame::Text_Frame(Group *parent, Text_Stream *stream,
		       FLength w, FLength h)
  :Cached_Frame(parent, w, h, "Text " + (stream?stream->get_name():"")),
   parsed_file(process_manager.find_new_name()), text_stream(stream),
   num_columns(1), gutter_width(12.0)
{
  white_is_transparent=true;
  ps_exists=false;

  if(text_stream)
    text_stream->add_frame(this);
}

Text_Frame::Text_Frame(Group *parent, const XMLNode& node)
  :Cached_Frame(parent, node),
   parsed_file(process_manager.find_new_name()),
   text_stream(0), num_columns(1), gutter_width(12.0)
{
  white_is_transparent=true;
  ps_exists=false;

  if(!node.property("type") || node.property("type")->value()!="text")
    throw Error::Frame_Type("Bad text-frame type: " 
			    + node.property("type")->value());

  if(const XMLProperty* cols = node.property("num_columns"))
    num_columns = to<int>(cols->value());
  
  // Text_Frame cannot live on matrices alone ... yet.
  if(const XMLProperty* w = node.property("width"))
    width = to<float>(w->value());
  else
    throw Error::Read("No \"width\" tag found in text frame.");
  if(const XMLProperty* h = node.property("height"))
    height = to<float>(h->value());
  else
    throw Error::Read("No \"height\" tag found in text frame.");

  // Note: This is the gutter width as a plain numeric value, it /should/ be
  // possible to give it as a value with a unit, as "1em" or "5mm" ...
  if(const XMLProperty* gutter = node.property("gutter_width"))
    gutter_width = to<float>(gutter->value());
  
  if(const XMLProperty* stream = node.property("stream")) {
    set_text_stream(get_document().get_text_stream(stream->value()));
    if(!text_stream)
      throw Error::Read("Failed to get text stream \"" + stream->value()
			+ "\" for text frame");
  } // If stream is not present text_stream is null. This is ok.
}

Text_Frame::~Text_Frame()
{
  if(text_stream)
    text_stream->remove_frame(this);
  if(!process_manager.delete_file(parsed_file))
    warning << "Couldn't delete " << parsed_file << std::endl;
}

void Text_Frame::generate_picture(View& view) 
{
  if(!text_stream)
    throw Gen_Pic_Error(ASSOCIATION, "No associated stream");
  
  pv = Gdk_Point(int(view.pt2scr(width)), int(view.pt2scr(height)));
  if(pv->x==0 || pv->y==0) 
    throw Gen_Pic_Error(ZEROSIZE, "Picture has zero size");

  if(!ps_exists)
    {
      text_stream->generate_ps_request(this);
      return;
    }

  if(pid!=-1)
    return;

  std::string psinterpreter=config.getPath("PSInterpreter");
  if(psinterpreter.empty())
    throw Gen_Pic_Error(GENERATION, "No PostScript interpreter specified");

  std::ostrstream tmp;
  tmp << psinterpreter //assuming it is ghostscript
      << " -q -dNOPAUSE -dBATCH -sDEVICE=ppmraw -r" << view.get_scrres()
      << " -g" << pv->x << 'x' << pv->y
      << " -sOutputFile=- " << parsed_file
      << " > " << tmp_file << std::ends;
  pid=process_manager.start(tmp.str());

  if(pid==-1)
    throw Gen_Pic_Error(GENERATION, "Failed to run "+psinterpreter);

  verbose << pid << ": " << tmp.str() << std::endl;
}

XMLNode *Text_Frame::save()
{
  XMLNode *node=Cached_Frame::save();
  node->add_property("type", "text");

  // Text_Frame cannot live on matrices alone ... yet.
  node->add_property("width", tostr(width));
  node->add_property("height", tostr(height));

  node->add_property("num_columns", tostr(num_columns));
  node->add_property("gutter_width", tostr(gutter_width));
  if(text_stream)
    node->add_property("stream", text_stream->get_name());

  return node;
}

void Text_Frame::print(std::ostream &out)
{
  if(!text_stream)
    {
      Cached_Frame::print(out);
      return;
    }

  using std::endl;
  
  out << "save" << endl
      << "/showpage {} def" << endl
      << "0 setgray 0 setlinecap 1 setlinewidth" << endl
      << "0 setlinejoin 10 setmiterlimit [] 0 setdash newpath" << endl;
  psConcat(out, get_matrix());
  if(false) //FIXME!!! if(level2)
    out<<"false setoverprint false setstrokeadjust"<<endl;
  
  out << "%%BeginDocument: " << parsed_file << endl;
  std::ifstream file(parsed_file.c_str());
  if(!file)
    throw Error::Print("Failed to open " + parsed_file);
    
  std::string tmp;
  while(getline(file, tmp))
    out << tmp << endl;
    
  out << "%%EndDocument" << endl
      << "restore" << endl;
}

void Text_Frame::set_text_stream(Text_Stream *new_stream, bool remove_from_old)
{
  if(text_stream == new_stream)
    return;

  if(remove_from_old && text_stream)
    text_stream->remove_frame(this);
  text_stream = new_stream;
  if(text_stream)
    text_stream->add_frame(this);
  
  // Ok, make the GUI reflect the change
  clear_picture();
  props_changed_signal(this);
  ready_to_draw_signal(this);
}

void Text_Frame::move_reshape_box(int box, Vector v)
{
  double xpos=0;
  double ypos=0;

  switch(box)
    {
    case 6:
      xpos+=v.x;
      ypos-=v.y;
      width-=v.x;
      height+=v.y;
      break;
    case 7:
      ypos-=v.y;
      height+=v.y;
      break;
    case 8:
      ypos-=v.y;
      width+=v.x;
      height+=v.y;
      break;
    case 4:
      xpos+=v.x;
      width-=v.x;
      break;
    case 5:
      width+=v.x;
      break;
    case 1:
      xpos+=v.x;
      width-=v.x;
      height-=v.y;
      break;
    case 2:
      height-=v.y;
      break;
    case 3:
      width+=v.x;
      height-=v.y;
      break;
    default:
      warning << "StupidProgrammerError:"
	      << "motion_notify_event_impl reached default!" 
	      << std::endl;
      break;
    }
  set_matrix(get_matrix()*Matrix::translation(xpos, ypos));
}

void Text_Frame::end_reshape(bool revert)
{
  ps_exists=ps_exists && revert; // ignore existing ps - it is too old
  if(!revert && text_stream)
    {
      try
	{
	  text_stream->generate_ps_request(this);
	}
      catch(Gen_Pic_Error e)
	{
	}
    }
  Cached_Frame::end_reshape(revert);
}

void Text_Frame::set_size(FLength w, FLength h) 
{
  ps_exists=false; // ignore existing ps - it is too old
  Basic_Frame::set_size(w, h);
  if(text_stream)
    {
      try
	{
	  text_stream->generate_ps_request(this);
	}
      catch(Gen_Pic_Error e)
	{
	}
    }
}

void Text_Frame::set_num_columns(int columns) 
{ 
  if(num_columns != columns)
    {
      num_columns = columns;
      props_changed_signal(this);
      if(text_stream)
	text_stream->generate_ps_request(this);
    }
}

void Text_Frame::set_gutter(const float& gutter) 
{ 
  if(gutter_width !=gutter)
    {
      gutter_width = gutter; 
      props_changed_signal(this);
      if(text_stream)
	text_stream->generate_ps_request(this);
    }
}
