///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <algorithm>
#include <strstream>
#include <fstream>

#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>

#include "textstream.h"
#include "textframe.h"
#include "globals.h"
#include "processman.h"
#include "page.h"
#include "config.h"

namespace {			// local
  class Sorting_Predicate {
  public:
    bool operator () (const Text_Frame* a, const Text_Frame* b) {
      int a_num, b_num;
      try
	{
	  a_num = a->get_page_num(); 
	  b_num = b->get_page_num();
	}
      catch (Error::Invalid_Page_Num e)
	{
	  // shouldn't happen, but it has happened
	  // not really much to do here
	}
      if(a_num < b_num)
	return true;
      if(a_num > b_num)
	return false;
      
      // Same page, look at boxes.
      const Vector a_vec = a->get_box().getUL();
      const Vector b_vec = b->get_box().getUL();
      return ((a_vec.y + b_vec.x) > (a_vec.x + b_vec.y));
    }
  };
}

void Text_Stream::file_modified()
{
  apply_stylesheet();
}

Text_Stream::Text_Stream(const string& _name, const string& _association,
			 const string& _transform)
  :name(_name), association(_association), transform(_transform),
   last_typesetter_cmdline(""),
   pid(-1), stylesheet_applied(false)
{
  proc_stop_connection = process_manager.process_stopped
    .connect(SigC::slot(this, &Text_Stream::process_stopped));
  parsed_file=process_manager.find_new_name();
  tmp_file=process_manager.find_new_name();
}

Text_Stream::~Text_Stream()
{
  proc_stop_connection.disconnect();
  if(!process_manager.delete_file(parsed_file))
    warning << "Failed to delete " << parsed_file << std::endl;
  if(!process_manager.delete_file(tmp_file))
    warning << "Failed to delete " << tmp_file << std::endl;

  // Todo:  Do this on a copy of frames, after clearing frames.
  // This makes the UGLY argument remove_from_old unneeded, so remove it.
  // remove_frame has the property that removing an unexisting frame is not an
  // error, just a no-op.
  for(FrameVec::iterator i=frames.begin();
      i!=frames.end();
      i++)
    (*i)->set_text_stream(0, false);
  // do not let the frame remove itself from this stream
}

void Text_Stream::add_frame(Text_Frame *text_frame)
{
  // Don't insert a duplicate entry
  if(find(frames.begin(), frames.end(), text_frame) == frames.end())
    frames.push_back(text_frame);
}

void Text_Stream::remove_frame(Text_Frame *text_frame)
{
  FrameVec::iterator i = find(frames.begin(), frames.end(),
				       text_frame);
  if(i != frames.end()) frames.erase(i);
}

void Text_Stream::apply_stylesheet()
{
  if(association.empty() || !access(association))
    throw Basic_Frame::Gen_Pic_Error(Basic_Frame::ASSOCIATION, 
  				     "Couldn't open "+association);

  if(transform.empty()) { // assume there is no need to apply a stylesheet
    debug << "No stylesheet to apply for " << association << std::endl;
    stylesheet_applied=true;
    stage=XML;
    last_typesetter_cmdline=""; // new content - OK do to the same thing again 
    run_typesetter();
    return;
  }

  debug << "Apply stylesheet" << std::endl;
  std::ostrstream cmd;
  //Assume xsltproc is in the path
  // Todo: It should be possible to specify extra parameters to the style.
  cmd << "xsltproc --nonet " << transform << ' ' << association
      << " > " << parsed_file << std::ends;
  stage=XML;
  if(pid!=-1)
    process_manager.stop(pid);
  pid=process_manager.start(cmd.str());
  verbose << pid<< ": " << cmd.str() << std::endl;
  if(pid==-1)
    throw Basic_Frame::Gen_Pic_Error(Basic_Frame::GENERATION, 
				     "Failed to run xsltproc");
  last_typesetter_cmdline=""; // new content - OK do to the same thing again 
}

void Text_Stream::generate_ps_request(Text_Frame *frame)
{
  if(frame)
    debug << "Request by " << frame->get_name() << std::endl;
  debug << "pid: " << pid << std::endl;
  if(pid==-1)
    {
      if(!stylesheet_applied)
	apply_stylesheet();
      else 
	run_typesetter();
    }
  else if(stage==PS)
    run_typesetter(); // may kill existing process
}

void Text_Stream::run_typesetter()
{
  if(frames.empty() || !stylesheet_applied)
    return;

  debug << "Running typesetter" << std::endl;
  FrameVec tmp(frames);
  tmp.sort(Sorting_Predicate());

  std::ostrstream cmd;
  //Assume xml2ps is in the path
  cmd << "xml2ps";
  std::string font_path = config.getPath("FontPath");
  if(!font_path.empty())
    cmd << " -f " << font_path;

  for(FrameVec::iterator i=tmp.begin();
      i!=tmp.end();
      i++)
    {
      // Todo: This might return null, in which case we will crash.
      // on the other hand; it's only used to not add a -b parameter for
      // the actual text frame, maybe that could be done in a better way.
      const Boundary* boundary = (*i)->getObstacleBoundary();
      cmd << " -p " << (*i)->get_width() << 'x' << (*i)->get_height();
	  
      if((*i)->get_num_columns() > 1)
	cmd << '/' << (*i)->get_num_columns() << '+' << (*i)->get_gutter();

      // Transform from page coords to local textframe coords.
      const Matrix xform((*i)->get_matrix().inv());
	  
      // List of boundaries the text should flow around
      const BoundaryVect obstacle_list=getPage(*(*i)).obstacle_list();
      for(BoundaryVect::const_iterator j=obstacle_list.begin();
	  j!=obstacle_list.end();
	  j++) {
	if(!boundary || (*(*j) != *boundary))
	  cmd << " -b " << xform * (*(*j));
      }

    }
  // If there is no stylesheet, assume the association is conforming to the xml2ps DTD:
  cmd << " < " << ((!transform.empty()) ? parsed_file : association) << " > " << tmp_file << std::ends;

  if(cmd.str()==last_typesetter_cmdline)
    {
      debug << "I'm not doing the same thing again!" << std::endl;
      return;
    }

  if(pid!=-1 && stage==PS)
    {
      debug << "Stopping " << pid << std::endl;
      process_manager.stop(pid);
    }

  pid=process_manager.start(cmd.str());
  last_typesetter_cmdline=cmd.str();

  stage=PS;
  if(pid==-1)
    throw Basic_Frame::Gen_Pic_Error(Basic_Frame::GENERATION, 
				     "Failed to run xml2ps");
  verbose << pid << ": " << cmd.str() << std::endl;
}

void Text_Stream::process_stopped(pid_t pid, bool exited_normally, 
				  int exit_code)
{
  if(pid!=this->pid)
    return;
  this->pid=-1;
  if(!exited_normally)
    {
      warning << "Process " << this->pid << " exited abnormally" << std::endl;
      return; // Todo: do something clever
    }
  debug << pid << ", exit code: " << exit_code << std::endl;
  if(stage==XML)
    {
      if(exit_code==127) // the shell could not find xsltproc
	{
	  warning << "xsltproc is not in $PATH" << std::endl;
	  return; // Todo: deal with this
	}
      // Todo: check that the return code 127 is not unique to Bash.

      debug << "Stylesheet done" << std::endl;
      stylesheet_applied=true;
      run_typesetter();
    }
  else // stage=PS
    {
      if(exit_code==127) // the shell could not find xml2ps
	{
	  warning << "xml2ps is not in $PATH" << std::endl;
	  return; // Todo: deal with this
	}
      // Todo: check that the return code 127 is not unique to Bash.

      debug << "xml2ps done" << std::endl;
      split_into_pages();
    }
}

void Text_Stream::split_into_pages()
{
  std::ifstream in(tmp_file.c_str());
  if(!in)
    throw Basic_Frame::Gen_Pic_Error(Basic_Frame::GENERATION, 
				     "Could not open "+tmp_file);

  FrameVec tmp(frames);
  tmp.sort(Sorting_Predicate());

  // First, we need to get the prolog and setup sections
  std::vector<std::string> ps_header;
  std::string buffer;
  while(getline(in, buffer) && !starts_with(buffer, "%%Page:"))
    ps_header.push_back(buffer);

  // Next, we split the file into pages
  for(FrameVec::iterator i=tmp.begin();
      i!=tmp.end();
      i++)
    {
      std::ofstream out((*i)->get_parsed_file().c_str());
      if(!out)
	throw Basic_Frame::Gen_Pic_Error(Basic_Frame::GENERATION, 
					 "Could not write to "
					 +(*i)->get_parsed_file());

      for(std::vector<std::string>::const_iterator j=ps_header.begin();
	  j!=ps_header.end();
	  j++)
	out << *j << std::endl;

      out << std::endl << "%%Page: \"1\" 1" << std::endl;
      
      while(getline(in, buffer)
	    && !starts_with(buffer, "%%Page:")
	    && !starts_with(buffer, "%%Trailer")
	    && !starts_with(buffer, "%%EOF"))
	out << buffer << std::endl;

      if(starts_with(buffer, "%%EOF")) // Uh-oh! We're out of pages!
	out << "showpage" << std::endl; //better do something

      out << std::endl << "%%Trailer" << std::endl;
      out.close();
      (*i)->ps_exists=true; // tell the frame it has a ps
      (*i)->clear_picture();
      (*i)->ready_to_draw_signal(*i);
    }
  while(getline(in, buffer)); // eat rest of postscript.
}

void Text_Stream::set_association(string s)
{
  // Todo: run props_changed_signal for all frames. Maybe.
  // Todo: run ready_to_draw_signal for all frames. Maybe.
  association=s;
  apply_stylesheet();
}

std::string Text_Stream::get_association() const
{
  return association;
} 

void Text_Stream::set_name(string s)
{
  // Todo: run props_changed_signal for all frames. Maybe.
  name=s;
}

std::string Text_Stream::get_name() const
{
  return name;
} 

void Text_Stream::set_transform(const string& s)
{
  if(s == transform)
    return;
  
  transform = s;
  stylesheet_applied = false;
  apply_stylesheet();		// Right on ...
}

std::string Text_Stream::get_transform() const
{
  return transform;
}
