///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
// -*- c++ -*-
#ifndef TEXTSTREAM_H
#define TEXTSTREAM_H

#include "filewatcher.h"
#include <sys/types.h>		// pid_t
#include <list>

/*
Here's how it works:

Text_Stream keeps a list of Text_Frames.  When association is set,
Text_Stream runs apply_stylesheet to convert the association into the
xml2ps dtd. This only needs to be done when the association is changed
or the files modification date has been changed. When the xsltproc
process has finished, the xml2ps process is started in run_typesetter,
but only if there are any frames associated with the stream. When
xml2ps has finished, the resulting ps is chopped ut into one file for
each frame, and a ready_to_draw_signal is emitted from each frame.

When the docview tells the page to draw itself, the text frame looks
at its ps_exists member to detemine if it should read its
parsed_file. If it is false, the frame calls
Text_Stream::generate_ps_request which will run xml2ps if
necessary. This also happens when a frame is resized.

*/

class Text_Frame;

class Text_Stream: public File_Watcher
{
public:
  typedef std::string string;
  
  Text_Stream(const string& _name, const string& _association,
	      const string& _transform);
  ~Text_Stream();

  void add_frame(Text_Frame *text_frame);
  void remove_frame(Text_Frame *text_frame);

  void generate_ps_request(Text_Frame *frame=0);
  
  void set_association(string s);
  string get_association() const;

  void set_name(string s);
  string get_name() const;

  void set_transform(const string& s);
  string get_transform() const;
  
  void run_typesetter(); // public, so Document_View can run it 

protected:
  typedef std::list<Text_Frame*> FrameVec;
  enum Stage { XML, PS };
  Stage stage;
  string name, association, transform, parsed_file, tmp_file;
  string last_typesetter_cmdline; // avoid doing the same thing more than once
  FrameVec frames;
  pid_t pid;
  SigC::Connection proc_stop_connection;
  bool stylesheet_applied;

  void apply_stylesheet();
  void split_into_pages();
  string file_to_watch() const
  {return association;}
  void file_modified();
  void process_stopped(pid_t  pid, bool exited_normally, int exit_code);

private:

  // Undefined ctors, avoid defaults
  Text_Stream(const Text_Stream&);
  Text_Stream();
  void operator = (const Text_Stream&);
};

#endif
