///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pixbufwrap.h"
#include <gdk-pixbuf/gdk-pixbuf.h>

Pixbuf::Pixbuf(const std::string imagefile)
  : buf(gdk_pixbuf_new_from_file(imagefile.c_str()))
{}

Pixbuf::Pixbuf(int width, int height)
  : buf(gdk_pixbuf_new(GDK_COLORSPACE_RGB, true, 8, width, height))
{}

Pixbuf::Pixbuf(const Pixbuf& orig)
  : buf(orig.buf)
{
  if(buf)
    gdk_pixbuf_ref(buf);
}

Pixbuf::~Pixbuf() { unref(); }

Pixbuf& Pixbuf::operator = (const Pixbuf& orig) {
  GdkPixbuf *old = buf;
  if(orig.buf)
    buf = gdk_pixbuf_ref(orig.buf);
  else
    buf = 0;

  // Unref the old one last, so we don't break on self-assignment
  if(old)
    gdk_pixbuf_unref(old);
  return *this;
}

void Pixbuf::unref() {
  if(buf)
    gdk_pixbuf_unref(buf);
  buf = 0;
}

int Pixbuf::get_width() const {
  return gdk_pixbuf_get_width(buf);
}
int Pixbuf::get_height() const {
  return gdk_pixbuf_get_height(buf);
}
GdkColorspace Pixbuf::get_colorspace() const {
  return gdk_pixbuf_get_colorspace(buf);
}

void Pixbuf::load_file(const std::string imagefile) {
  if(buf) gdk_pixbuf_unref(buf);
  buf = gdk_pixbuf_new_from_file(imagefile.c_str());
}

Pixbuf Pixbuf::scale(int width, int height) {
  Pixbuf result;
  result.buf = gdk_pixbuf_scale_simple(buf, width, height, GDK_INTERP_TILES);
  return result;
}

void Pixbuf::render_to_drawable(Gdk_Drawable& drawable, GdkGC& gc,
				int xsrc, int ysrc, int xdest, int ydest,
				int width, int height)
{
  // if width and/or height is -1, that is supposed to mean the entire
  // picture, but that doesn't seem to be implemented in gdk_pixbuf yet, so I
  // implement it here ...
  if(width == -1)
    width = get_width();
  if(height == -1)
    height = get_height();

  gdk_pixbuf_render_to_drawable(buf, drawable, &gc,
				xsrc, ysrc, xdest, ydest, width, height,
				GDK_RGB_DITHER_NORMAL, 1, 1);
}

void Pixbuf::render_to_drawable_alpha(Gdk_Drawable& drawable,
				      int xsrc, int ysrc, int xdest, int ydest,
				      int width, int height)
{
  // if width and/or height is -1, that is supposed to mean the entire
  // picture, but that doesn't seem to be implemented in gdk_pixbuf yet, so I
  // implement it here ...
  if(width == -1)
    width = get_width();
  if(height == -1)
    height = get_height();

  gdk_pixbuf_render_to_drawable_alpha(buf, drawable,
				      xsrc, ysrc, xdest, ydest, width, height,
				      GDK_PIXBUF_ALPHA_BILEVEL, 30,
				      GDK_RGB_DITHER_NORMAL, 0, 0);
}




guchar* Pixbuf::get_pixels() {
  return gdk_pixbuf_get_pixels(buf);
}
int Pixbuf::get_rowstride() const {
  return gdk_pixbuf_get_rowstride(buf);
}
int Pixbuf::get_n_channels() const {
  return gdk_pixbuf_get_n_channels(buf);
}

Pixbuf::Statistics Pixbuf::calc_stats(int samples)
{
  // Todo: check that samples is well below number of pixels
  Pixbuf::Statistics result={true, false};
  const guchar* data = get_pixels();
  const int pixnum = get_width() * get_height();
  const int bytes_per_pixel = get_n_channels();
  int rle_count=0;
  for(int i=0; i<samples; i++)
    {
      const guchar* pixel = data + bytes_per_pixel * i * pixnum / samples;

      // is it grayscale?
      bool grayscale = ((*pixel == *(pixel+1))
			&& (*pixel == *(pixel+2)));
      result.grayscale = result.grayscale && grayscale;
      // Todo: check if b/w

      // count similar consecutive pixels:
      rle_count += (grayscale 
		    && *pixel == *(pixel+bytes_per_pixel)
		    && *(pixel+1) == *(pixel+bytes_per_pixel+1)
		    && *(pixel+2) == *(pixel+bytes_per_pixel+2)
		    )?1:0;
    }

  // minimum amount of similar pixels when suggesting rle:
  const int suggest_rle_threshold=3; 
  result.suggest_rle = (rle_count>=suggest_rle_threshold);
  return result;
}

