///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "window.h"

#include <iostream>
#include <strstream>
#include <fstream>
#include <gtk--/adjustment.h>
#include <gtk--/label.h>

#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>

#include "widget/filesel.h"

#include "processman.h"
#include "safeslot.h"
#include "globals.h"
#include "textframe.h"
#include "textstream.h"
#include "page.h"
#include "errordialog.h"
#include "config.h"
#include "inspiration.h"
#include "printdialog.h"
#include "aboutdialog.h"
#include "propertiesdialog.h"
#include "docpropsdialog.h"
#include "streamdialog.h"
#include "pagesel.h"

unsigned int Frame_Window::count(0);
PropertiesDialog *Frame_Window::properties_dialog(0);
Stream_Dialog *Frame_Window::stream_dialog(0);
Frame_Window* Frame_Window::active_window(0);

namespace{

  About_Dialog* about_dialog(0);

  Gtk::Menu *page_menu, *frame_menu, *doc_menu;
  Gtk::MenuItem *save_item, *save_as_item, *print_item, *preview_item, 
    *delete_page_item;
  Gtk::Button *save_button, *text_frame_button, *image_frame_button;
  Gtk::Button *streams_button, *preview_button, *properties_button;
}

// A bit more clever than a usual conversion function
// - it understands per cent signs.
float str2float(const std::string& value)
{
  std::istrstream s(value.c_str(), value.length());
  float f, factor=1;
  if(s >> f) {
    char ch;
    if(s >> ch)
      if(ch == '%') factor = 0.01;
      else throw std::runtime_error("Strange unit in \"" + value + '"');
    return f * factor;
  } else
    throw std::runtime_error("Not a float value: \"" + value + '"');
}

void Frame_Window::do_nothing()
{
  Error_Dialog::view("Not implemented yet.\nSorry.");
}

void Frame_Window::set_filename(std::string filename)
{
  if(!filename.empty())
    {
      set_title("Passepartout ("+basename(filename)+")");
    }
  else
    set_title("Passepartout");
}

void Frame_Window::zoom_change_action(float factor)
{
  if(!locked)
    {
      Gtk::Menu_Helpers::MenuList::iterator i;
      int j;
      for(i=zoom_factor.get_menu()->items().begin(), j=0;
	  i!=zoom_factor.get_menu()->items().end();
	  i++, j++)
	{
	  if(str2float(dynamic_cast<Gtk::Label&>
		(*((*i)->get_child())).get_text())==factor)
	    {
	      zoom_factor.get_menu()->set_active(j);
	      return;
	    }
	  else if(str2float(dynamic_cast<Gtk::Label&>
		     (*((*i)->get_child())).get_text())<factor)
	    {
	      warning << "Zoom factor " << factor 
		      << " is not defined" << std::endl;
	    }
	}
    }
}

void Frame_Window::update_rulers()
{
  Document *document=document_view.get_document();
  if(!document)
    return;
  
  Vector UL=document_view.scr2pt(Gdk_Point(0, 0));
  Vector LR=document_view.scr2pt(Gdk_Point(hrule.width(), vrule.height()));

  
  hrule.set_range(UL.x, LR.x,  0, 0);
  vrule.set_range(UL.y, LR.y, 0, 0);
}

void Frame_Window::zoom_factor_changed_action()
{
  locked=true;
  // GtkOtptionMenu (GTK+) behaves in a rather perverse manner:
  // The child of the active MenuItem is reparented to
  // reside within the actual button instead of inside the
  // MenuItem. I went a little crazy before I read the GTK+ FAQ.
  // /Fredrik
  if(Gtk::Label* label = dynamic_cast<Gtk::Label*>(zoom_factor.get_child()))
    document_view.set_zoom_factor(str2float(label->get_text()));
  else ;//run around in circles and scream!

  locked=false;
  update_rulers();
  scroller.get_vadjustment()->set_value(0);
  scroller.get_hadjustment()->set_value(0);
}

Gtk::Menu* Frame_Window::create_zoom_menu()
{
  using namespace Gtk;
  using namespace Menu_Helpers;

  Menu *menu=manage(new Menu());
  MenuList& menu_list=menu->items();
  for(std::list<std::string>::const_iterator i=zoom_factors.begin();
      i!=zoom_factors.end();  ++i)
    menu_list.push_back(MenuElem(*i));

  return menu;
}

void Frame_Window::document_updated(Document *document_)
{
  Document *document=document_view.get_document();
  if(!delete_page_item || document!=document_)
    return;
  document_changed();
}

void Frame_Window::document_filename_changed(Document *document_)
{
  Document *document=document_view.get_document();
  if(document_==document)
    set_filename(document ? document->get_filename() : "");
}

void Frame_Window::document_changed()
{
  Document *document=document_view.get_document();
  // enable/disable stuff
  bool on=(document!=0);
  bool have_pages = on && document->get_num_of_pages() > 0;
  if(!(save_item && save_as_item && print_item && preview_item 
       && save_button && text_frame_button && image_frame_button 
       && preview_button
       && page_menu && frame_menu && doc_menu))
    return;
  save_item->set_sensitive(on);
  save_button->set_sensitive(on);
  save_as_item->set_sensitive(on);
  print_item->set_sensitive(have_pages);
  preview_item->set_sensitive(have_pages);
  preview_button->set_sensitive(have_pages);
  text_frame_button->set_sensitive(have_pages);
  image_frame_button->set_sensitive(have_pages);
  toolbar.set_sensitive(on);
  pagesel->set_sensitive(have_pages);
  zoom_factor.set_sensitive(on);

  for(Gtk::Menu_Helpers::MenuList::iterator i=doc_menu->items().begin();
      i!=doc_menu->items().end();
      i++)
    (*i)->set_sensitive(on);

  for(Gtk::Menu_Helpers::MenuList::iterator i=page_menu->items().begin();
      i!=page_menu->items().end();
      i++)
    (*i)->set_sensitive(on);
  delete_page_item->set_sensitive(have_pages);

  for(Gtk::Menu_Helpers::MenuList::iterator i=frame_menu->items().begin();
      i!=frame_menu->items().end();
      i++)
    (*i)->set_sensitive(on  && document->get_num_of_pages()>0);

  if(false)  // rulers are hidden by default until they work properly
    {
      hrule.show();
      vrule.show();
    }
  else
    {
      hrule.hide();
      vrule.hide();
    }

  document_filename_changed(document);
}

void Frame_Window::new_document()
{
  doc_props_dialog->show_it(true);
}

void Frame_Window::create_menus()
{
  using namespace Gtk;
  using namespace Menu_Helpers;
  using SigC::bind;

  Menu *file_menu=manage(new Menu());
  MenuList& list_file=file_menu->items();
  list_file.push_back(MenuElem("_New ...", "<control>N", 
			       safeslot(this, &Frame_Window::new_document)));
  list_file.push_back(MenuElem("_Open ...", "<control>O", 
			       safeslot(open_dialog, &Filesel::show_all)));
  list_file.push_back(MenuElem("_Save", "<control>S", 
			       safeslot(this, &Frame_Window::save)));
  save_item=list_file.back();
  list_file.push_back(MenuElem("Save _As ...", 
			       safeslot(save_dialog, &Filesel::show_all)));
  save_as_item=list_file.back();
  list_file.push_back(MenuElem("_Close", "<control>W",
			       safeslot(this, &Frame_Window::close)));
  list_file.push_back(SeparatorElem());
  list_file.push_back(MenuElem("_Print ...", "<control>P",
			       safeslot(print_dialog, &Print_Dialog::show_it)));
  print_item=list_file.back();
  list_file.push_back(MenuElem("Print Pre_view", 
			       safeslot(this, &Frame_Window::print_to_viewer)));  
  preview_item=list_file.back();
  list_file.push_back(SeparatorElem());
  list_file.push_back(MenuElem("_Quit", "<control>Q", 
			       safeslot(this, &Frame_Window::quit)));
  menubar.items().push_back(MenuElem("_File","<alt>f",*file_menu));

  doc_menu=manage(new Menu());
  MenuList& list_doc=doc_menu->items();
  list_doc.push_back(MenuElem("_Streams", 
			      safeslot(stream_dialog, 
				       &Stream_Dialog::showRaise)));
  list_doc.push_back(SeparatorElem());
  list_doc.push_back(MenuElem("P_roperties ...",
			      safeslot(this, 				   
				       &Frame_Window::show_doc_props_dialog)));
  menubar.items().push_back(MenuElem("_Doc", "<alt>d", *doc_menu));

  page_menu=manage(new Menu());
  MenuList& list_page=page_menu->items();
  list_page.push_back(MenuElem("New _Before ...", 
			       safeslot(document_view, 
					&Document_View::insert_page_before)));
  list_page.push_back(MenuElem("New _After ...", 
			       safeslot(document_view, 
					&Document_View::insert_page_after)));
  list_page.push_back(SeparatorElem());
#ifdef DEBUG
  list_page.push_back(MenuElem("_Copy", 
			       safeslot(this, 
					&Frame_Window::do_nothing)));
  list_page.push_back(MenuElem("C_ut", 
			       safeslot(this, 
					&Frame_Window::do_nothing)));
  list_page.push_back(MenuElem("Paste B_efore", 
			       safeslot(this, 
					&Frame_Window::do_nothing)));
  list_page.push_back(MenuElem("Paste A_fter", 
			       safeslot(this, 
					&Frame_Window::do_nothing)));
#endif
  list_page.push_back(MenuElem("_Delete", 
			       safeslot(document_view, 
					&Document_View::delete_page)));
  delete_page_item=list_page.back();
  //  list_page.push_back(SeparatorElem());
  //  list_page.push_back(MenuElem("_Template Page ...", safeslot(this, &Frame_Window::show_template_page_dialog)));
  // it is not yet possible to change the template of an existing page
  menubar.items().push_back(MenuElem("_Page","<alt>p",*page_menu));

  frame_menu=manage(new Menu());
  MenuList& list_frame=frame_menu->items();
  list_frame.push_back(MenuElem("_New Text Frame", 
				safeslot(document_view,
					 &Document_View::new_text_frame)));
  list_frame.push_back(MenuElem("_Import Image ...", 
				safeslot(import_dialog, &Filesel::show_all)));
  list_frame.push_back(SeparatorElem());
  list_frame.push_back(MenuElem("Select _All", "<control>A",
				safeslot(document_view, 
					 &Document_View::select_all_frames)));
  list_frame.push_back(MenuElem("Select _None",	
				safeslot(document_view, 
					 &Document_View::unselect_all_frames)));
  list_frame.push_back(SeparatorElem());
#ifdef DEBUG
  list_frame.push_back(MenuElem("_Copy",
				safeslot(this, &Frame_Window::do_nothing)));
  list_frame.push_back(MenuElem("C_ut",
				safeslot(this, &Frame_Window::do_nothing)));
  list_frame.push_back(MenuElem("P_aste",
				safeslot(this, &Frame_Window::do_nothing)));
#endif
  list_frame.push_back(MenuElem("_Delete", "<control>D", 
				safeslot(document_view, 
					 &Document_View::delete_selected)));
  list_frame.push_back(SeparatorElem());
  Menu *arrange_menu=manage(new Menu());
  MenuList& list_arrange=arrange_menu->items();
  list_arrange.push_back(MenuElem("_Group",
				  safeslot(document_view, 
					   &Document_View::group_selected)));
  list_arrange.push_back(MenuElem("_Ungroup",
				  safeslot(document_view, 
					   &Document_View::ungroup_selected)));
  list_arrange.push_back(SeparatorElem());
  list_arrange.push_back(MenuElem("Move to _top",
				  safeslot(bind(slot(document_view, &Document_View::rearrange_selected), TOP))));
  list_arrange.push_back(MenuElem("Move _up",
				  safeslot(bind(slot(document_view, &Document_View::rearrange_selected), UP))));
  list_arrange.push_back(MenuElem("Move _down",
				  safeslot(bind(slot(document_view, &Document_View::rearrange_selected), DOWN))));
  list_arrange.push_back(MenuElem("Move to _bottom",
				  safeslot(bind(slot(document_view, &Document_View::rearrange_selected), BOTTOM))));
  list_frame.push_back(MenuElem("A_rrange", *arrange_menu));
  list_frame.push_back(MenuElem("_Properties",
				safeslot(properties_dialog, 
					 &PropertiesDialog::showRaise)));
  menubar.items().push_back(MenuElem("_Object","<alt>o",*frame_menu));

  Menu *options_menu=manage(new Menu());
  MenuList& list_options=options_menu->items();
  list_options.push_back(CheckMenuElem("_Toolbar", 
				       safeslot(this, 
						&Frame_Window::toggle_toolbar)));
  // rulers are hidden by default until they work properly
  toggle_rulers();
  static_cast<CheckMenuItem*>(list_options.back())->set_active(false);
  list_options.push_back(CheckMenuElem("_Rulers", 
				       safeslot(this, 
						&Frame_Window::toggle_rulers)));
  static_cast<CheckMenuItem*>(list_options.back())->set_active(true);
  list_options.push_back(CheckMenuElem("_Statusbar", 
				       safeslot(this, 
						&Frame_Window::toggle_cafe_opera)));
  static_cast<CheckMenuItem*>(list_options.back())->set_active(true);
  menubar.items().push_back(MenuElem("Op_tions", 
				     "<alt>t", 
				     *options_menu));

  Menu *help_menu=manage(new Menu());
  MenuList& list_help=help_menu->items();
  list_help.push_back(MenuElem("_Inspiration", 
			       safeslot(inspiration, 
					&Inspiration::show_all)));
  list_help.push_back(SeparatorElem());
  list_help.push_back(MenuElem("_About", 
			       safeslot(about_dialog, &About_Dialog::show_all)));
  menubar.items().push_back(MenuElem("_Help", "<alt>h", *help_menu));
  menubar.items().back()->right_justify();
}

Frame_Window::Frame_Window(std::string filename, Doc_Props_Dialog *dp_dialog):
  toolbar(GTK_ORIENTATION_VERTICAL, GTK_TOOLBAR_ICONS),//GTK_TOOLBAR_TEXT),
  table(2, 2, false), document_table(2, 2, false),
  document_view(NULL)
{
  // temp file for gv to read from when doing a print preview:
  preview_tmp_file=process_manager.find_new_name();

  print_dialog=manage(new Print_Dialog(*this, document_view));
  doc_props_dialog=manage(new Doc_Props_Dialog(*this, document_view));
  pagesel=manage(new Pagesel(document_view));
  open_dialog=manage(new Filesel(*this, "Open"));
  save_dialog=manage(new Filesel(*this, "Save As"));
  import_dialog=manage(new Filesel(*this, "Import Image"));

  // Should we delete these before quitting?
  if(!properties_dialog)
    properties_dialog = new PropertiesDialog();
  if(!stream_dialog)
    stream_dialog = new Stream_Dialog();
  if(!about_dialog)
    about_dialog = new About_Dialog();

  create_menus(); 
  
  locked=false;
  set_default_size(450, 600);

  Error_Dialog::init(*this);

  zoom_factors.push_back("50 %");
  zoom_factors.push_back("75 %");
  zoom_factors.push_back("100 %");
  zoom_factors.push_back("150 %");
  zoom_factors.push_back("200 %");

  destroy.connect(safeslot(this, &Frame_Window::destroy_handler));

  add(main_container);

  hbox.pack_start(toolbar, false, false, 2);
  hbox.pack_end(vbox);

  vbox.pack_end(table, true, true, 0);

  main_container.pack_start(menubar, false);
  main_container.pack_start(hbox);
  main_container.pack_end(pagebar, false, false, 0);

  table.attach(scroller, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL);
  table.attach(hrule, 1, 2, 0, 1,
	       GTK_SHRINK | GTK_FILL, GTK_FILL);
  document_view.motion_notify_event.connect(hrule.motion_notify_event.slot());
  hrule.set_metric(GTK_PIXELS);
  table.attach(vrule, 0, 1, 1, 2,
	       GTK_FILL, GTK_SHRINK | GTK_FILL);
  document_view.motion_notify_event.connect(vrule.motion_notify_event.slot());
  vrule.set_metric(GTK_PIXELS);

  scroller.add_with_viewport(document_view);
  scroller.set_policy(GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  update_rulers();

  {
    using namespace Gtk::Toolbar_Helpers;
    ToolList &tools=toolbar.tools();

    tools.push_back(ButtonElem(0, icons.save, 
			       safeslot(this, &Frame_Window::save),
			       "Save file"));
    save_button=static_cast<Gtk::Button*>(tools.back()->get_content());
    tools.push_back(Space());
    tools.push_back(ButtonElem(0, icons.preview, 
			       safeslot(this, &Frame_Window::print_to_viewer),
			       "Print to external viewer"));
    preview_button=static_cast<Gtk::Button*>(tools.back()->get_content());
    tools.push_back(Space());
    //    tools.push_back(ButtonElem(0, icons.pointer, 
    //			       slot(this, &Frame_Window::do_nothing)));
    tools.push_back(ButtonElem(0, icons.new_frame, 
			       safeslot(document_view, 
					&Document_View::new_text_frame),
			       "Create new text frame"));
    text_frame_button=static_cast<Gtk::Button*>(tools.back()->get_content());
    tools.push_back(ButtonElem(0, icons.moose, 
			       safeslot(import_dialog, &Filesel::show_all),
			       "Import image file"));
    image_frame_button=static_cast<Gtk::Button*>(tools.back()->get_content());
    tools.push_back(Space());
    tools.push_back(ButtonElem(0, icons.properties, 
			       safeslot(properties_dialog, 
					&PropertiesDialog::showRaise),
			       "Show object properties"));
    properties_button=static_cast<Gtk::Button*>(tools.back()->get_content());
    tools.push_back(ButtonElem(0, icons.streams, 
			       safeslot(stream_dialog, 
					&Stream_Dialog::showRaise),
			       "Show list of streams"));
    tools.push_back(Space());
    tools.push_back(ButtonElem(0, icons.top,
			       safeslot(bind(slot(document_view, &Document_View::rearrange_selected), TOP)),
			       "Move object(s) to top"));
    tools.push_back(ButtonElem(0, icons.up,
			       safeslot(bind(slot(document_view, &Document_View::rearrange_selected), UP)),
			       "Move object(s) up"));
    tools.push_back(ButtonElem(0, icons.down,
			       safeslot(bind(slot(document_view, &Document_View::rearrange_selected), DOWN)),
			       "Move object(s) down"));
    tools.push_back(ButtonElem(0, icons.bottom,
			       safeslot(bind(slot(document_view, &Document_View::rearrange_selected), BOTTOM)),
			       "Move object(s) to bottom"));
    streams_button=static_cast<Gtk::Button*>(tools.back()->get_content());
  }

  pagebar.pack_end(*pagesel, false, false, 0);
  pagebar.pack_end(*manage(new Gtk::Label(" Page: ")), false, false, 0);
  pagebar.pack_end(zoom_factor, false, false, 0);
  pagebar.pack_end(*manage(new Gtk::Label(" Zoom: ")), false, false, 0);
  pagebar.pack_start(cafe_opera, true, true, 0);

  document_view.zoom_change_signal.connect
    (slot(this, &Frame_Window::zoom_change_action));
  zoom_factor.set_menu(create_zoom_menu());
  zoom_factor.get_menu()->selection_done.connect
    (slot(this, &Frame_Window::zoom_factor_changed_action));
  
  open_dialog->hide.connect(safeslot(this, &Frame_Window::open_dialog_done));
  save_dialog->hide.connect(safeslot(this, &Frame_Window::save_dialog_done));
  import_dialog->hide.connect(safeslot(this, &Frame_Window::import_dialog_done));

  document_view.document_set_signal.connect
    (safeslot(this, &Frame_Window::document_changed));

  Document::changed_signal.connect(slot(this, &Frame_Window::document_updated));
  Document::filename_changed_signal.connect(slot(this, &Frame_Window::document_filename_changed));

  // document_updated enables/disables some functions, 
  // so the widgets should be realised first.
  if(dp_dialog){
    if(dp_dialog->template_button.get_active())
      {
	std::string template_file=dp_dialog->file_entry.entry.get_text();
	document_view.set_document(new Document(template_file));
      }
    else
      {
	Papers::Orientation orientation=
	  dp_dialog->portrait_button.get_active()
	  ? Papers::PORTRAIT : Papers::LANDSCAPE; 
	Document *document=new Document();
	document->set_paper_name(dp_dialog->paper_size.get_entry()->get_text());
	document->set_orientation(orientation);
	document->set_doublesided(dp_dialog->double_sided_button.get_active());
	document->set_first_page_num(int(dp_dialog->first_page.get()));
	document_view.set_document(document);
      }
  } else if(!filename.empty()) {
    // If we don't specify false, it will try to find a template in filename:
    document_view.set_document(new Document(filename, false)); 
  } else 
    document_view.set_document(NULL);
  
  set_filename(basename(filename));

  ++count; 
  show_all();
  active_window=this;
  hrule.hide(); vrule.hide();

  // Make sure not everything is visible:
  document_updated(document_view.get_document()); 
}

/*
void Frame_Window::show_template_page_dialog()
{
  template_page_dialog->show_it();
}
*/

void Frame_Window::show_doc_props_dialog()
{
  doc_props_dialog->show_it(false);
}

void Frame_Window::open_dialog_done()
{
  if(!open_dialog->was_cancelled())
    {
      try
	{
	  if(document_view.get_document())
	    manage(new Frame_Window(open_dialog->get_filename()));
	  else
	    {
	      document_view.set_document(new Document(open_dialog->get_filename(),
						      false));
	      set_filename(basename(open_dialog->get_filename()));
	    }
	}
      catch(const Error::Read &e)
	{
	  Error_Dialog::view(e.message);
	}
    }
}

void Frame_Window::import_dialog_done()
{
  if(!import_dialog->was_cancelled())
    document_view.new_image_frame(import_dialog->get_filename());
}

void Frame_Window::save_dialog_done()
{
  Document *document=document_view.get_document();

  if(!save_dialog->was_cancelled() && document)
    {
      document->set_filename(save_dialog->get_filename());
      try
	{
	  document->save();
	}
      catch (Error::Write e)
	{
	  Error_Dialog::view(e.message);
	}
    }
}

void Frame_Window::close()
{
  Document *document=document_view.get_document();
  if(count>1 || !document)
    destroy();
  else
    {
      document_view.set_document(0);
      properties_dialog->set_document(0);
      stream_dialog->set_document(0);
      if(document)
	delete document;
    }
}

void Frame_Window::save()
{
  Document *document=document_view.get_document();

  if(document && !document->get_filename().empty())
    {
      try
	{
	  document->save();
	}
      catch (Error::Write e)
	{
	  Error_Dialog::view(e.message);
	}
    }
  else
    save_dialog->show();
}

void Frame_Window::toggle_rulers()
{
  if(vrule.is_visible())
    {
      vrule.hide();
      hrule.hide();
    }
  else
    { 
      vrule.show();
      hrule.show();
    }
}

void Frame_Window::toggle_toolbar()
{
  if(toolbar.is_visible())
    toolbar.hide();
  else
    toolbar.show();
}

void Frame_Window::toggle_cafe_opera()
{
  if(cafe_opera.is_visible())
    cafe_opera.hide();
  else
    cafe_opera.show();
}

int Frame_Window::delete_event_impl(GdkEventAny *event)
{
  return false;
}

int Frame_Window::focus_in_event_impl(GdkEventFocus *event)
{
  // the nonmodal windows are updated to reflect the content in the window
  // that is currently in focus

  active_window=this;
  if(properties_dialog)
    {
      properties_dialog->set_transient_for(*this);
      properties_dialog->set_document(document_view.get_document());
      //      Gdk_Window win(properties_dialog->get_window());
      //      win.raise();	
    }
  if(stream_dialog)
    {
      stream_dialog->set_transient_for(*this);
      stream_dialog->set_document(document_view.get_document());
    }
  return false;
}

void Frame_Window::destroy_handler()
{
  if(count==1)
    quit();
}

void Frame_Window::print_to_viewer()
{
  Document *document=document_view.get_document();

  // try to run gv (or whatever you want to use instead)
  if(document)
    {
      try
	{
	  std::string psviewer=config.getPath("PSViewer");
	  if(psviewer!="") {
	    std::ofstream out(preview_tmp_file.c_str());
	    if(!out)
	      {
		Error_Dialog::view("Could not print to file:\n"
				   + preview_tmp_file);
		return;
	      }
	    // print all pages:
	    document->print(out, 
			    document->get_first_page_num(), 
			    document->get_first_page_num() 
			    + document->get_num_of_pages() - 1);
	    verbose << psviewer + " " + preview_tmp_file << std::endl;
	    pid_t pid=process_manager.start(psviewer + " " + preview_tmp_file);
	    if(pid==-1)
	      {
		Error_Dialog::view("Could not run:\n"
				   + psviewer);
		return;
	      }
	  } else
	    Error_Dialog::view("No PostScript viewer defined");
	}
      catch(Error::Print e)
	{
	  Error_Dialog::view(e.message);
	}
    }
}

void Frame_Window::quit()
{
  Gtk::Main::quit();
}

Frame_Window::~Frame_Window()
{
  --count;
  active_window=0; // don't know which one it'll be
  if(!process_manager.delete_file(preview_tmp_file))
    warning << "Failed to delete " << preview_tmp_file << std::endl;
  if(count>0)
    {
      debug << "count=" << count << std::endl;
      properties_dialog->set_document(0);
      stream_dialog->set_document(0);
    }
}
