///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "filesys.h"
#include "stringutil.h"
#include <unistd.h>

using std::string;

bool access(const string& filename, int mode) {
  int c_mode = 0;
  if(mode & std::ios::in)  c_mode |= R_OK;
  if(mode & std::ios::out) c_mode |= W_OK;
  int result = access(filename.c_str(), c_mode);
  return result == 0;
}

string suffix(const string& filename) {
  const string::size_type pos = filename.find_last_of('.');
  if(pos == string::npos)
    return string();
  return filename.substr(pos+1);
}

string no_suffix(const string& filename) {
  const string::size_type pos = filename.find_last_of('.');
  if(pos == string::npos)
    return string();
  return filename.substr(0, pos);
}

string basename(const string& filename) {
  const string::size_type pos = filename.find_last_of('/');
  if(pos == string::npos)
    return filename;
  return filename.substr(pos+1);
}

string path(const string& filename) {
  const string::size_type pos = filename.find_last_of('/');
  if(pos == string::npos)
    return "./";
  return filename.substr(0, pos+1);
}

string expand_path(const string& filename) {
  unsigned int path_max;
  // from man 3 realpath:
#ifdef PATH_MAX
  path_max = PATH_MAX;
#else
  // pathconf doesn't seem to like empty strings
  path_max = pathconf (filename.empty()?".":filename.c_str(), _PC_PATH_MAX);
  if (path_max <= 0)
    path_max = 4096;
#endif
  char *resolved_path=new char[path_max];
  string result;
  if(!realpath(filename.empty()?".":filename.c_str(), resolved_path))
    {
      // Todo:  Use the warning stream, or better yet, throw an exception!
      std::cerr << "Couldn't resolve " << filename << std::endl;
      result=filename;
    }
  else
    result=resolved_path;
  delete resolved_path;
  return result;
}

string relative_path(const string& start_path, const string& filename)
{
  if(start_path.empty() || !starts_with(filename, start_path))
    return filename;
  int i = start_path.length();
  if(filename[i]=='/')
    i++;
  return filename.substr(i);
}
