///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "canvas.hh"
#include <stdexcept>
#include <algorithm>
#include <functional>

xml2ps::Canvas::Canvas(const std::vector<xml2ps::PageBoundary>& pages,
		       bool allow_extra_pages)
  :pages_(pages), cur_page_(0), allow_extra_pages_(allow_extra_pages)
{}

xml2ps::Canvas::~Canvas() {}

void xml2ps::Canvas::newPage() {
  PageVec::const_iterator op = cur_page_;
  if(cur_page_ == PageVec::const_iterator(0)) cur_page_ = pages_.begin();
  else ++cur_page_;
  
  if(cur_page_ == pages_.end()) {
    if(!allow_extra_pages_)
      throw out_of_pages();
    if(op == PageVec::const_iterator(0))
      throw std::runtime_error("No pages specified");
    cur_page_ = op;
  }
  
  cur_column = 0;
  vpos = height_ = cur_page_->height();
  margin = 0;
}

const float& xml2ps::Canvas::down(const float& step) {
  vpos -= margin;
  margin = 0;
  return vpos -= step;
}
void xml2ps::Canvas::reserveHeight(const float& h, bool span) {
  if ((vpos - margin) < h) {
    if(++cur_column >= cur_page_->columns()) {
      newPage();
    } else
      vpos = height_;
  }
  if (span)
    height_ -= h;
}

void xml2ps::Canvas::addMargin(const float& m) {
  margin = std::max(margin, m);
}

namespace {
  // Todo:  This should be possible to do with a standard binder instead!
  class IntersectAdapter {
  public:
    IntersectAdapter(xml2ps::HBox& hb) : hbox(hb) {}
    void operator() (const xml2ps::Obstacle* o) {
      o->intersect(hbox);
    }
  private:
    xml2ps::HBox& hbox;
  };
}

xml2ps::HBox xml2ps::Canvas::hbox(bool span, float margin_left,
				  float margin_right, 
				  float ascender, float descender) {
  reserveHeight(ascender + descender, span);
  const float left = cur_page_->left(cur_column);
  down(ascender);
  HBox result(vpos, ascender, descender, 
	      left, left + cur_page_->columnWidth(span),
	      margin_left, margin_right);
  down(descender);
  
  for_each(cur_page_->obegin(), cur_page_->oend(), IntersectAdapter(result));
  
  const float minimum_width = 10.0;// Todo: Don't hardcode this!
  if(result.room() < minimum_width) {
    // Something covers the whole line, recurse to take next line instead.
    return hbox(span, margin_left, margin_right, ascender, descender);
  }
  return result;
}
