///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "fontmetrics.hh"
#include <string>
#include <map>
#include <iostream>
#include <fstream>

using std::string;

class font::Metrics::Encoding {
  std::vector<string> encoding;
  typedef std::map<string, int> emap;
  emap encoding_map;
  
public:
  Encoding();
  void toPs(std::ostream &out) const;
  
  int operator() (const string &name) const {
    emap::const_iterator i = encoding_map.find(name);
    if(i != encoding_map.end()) return (*i).second;
    else return 0;
  }
private:
  Encoding(const Encoding &); // undefined.
  void operator= (const Encoding&); // undefined

  static const char* const init[]; // last in this file.
};


font::Metrics::Encoding::Encoding() {
#ifdef DEBUG_LOG
  std::cerr << "Constructing encoding ...";
#endif
  for(int i = 0; init[i] != NULL; ++i) {
    encoding.push_back(init[i]);
#ifdef DEBUG_LOG
    std::cerr << "\n   add " << encoding.back();
#endif
    encoding_map[encoding.back()] = i;
  }
#ifdef DEBUG_LOG
  std::cerr << " ... done!" << std::endl;
#endif
}

void
font::Metrics::Encoding::toPs(std::ostream &out) const {
  out << "\n% ISO Latin 1 encoding\n"
    "/Latin1Encoding [\n";
  for(std::vector<string>::const_iterator i = encoding.begin(); 
      i != encoding.end(); ++ i)
    out << '/' << *i << ' ';
  out << "] def

/Latin1Encode {
  dup findfont
  dup length
  dict begin
    {
      1 index /FID ne {def} {pop pop} ifelse
    } forall
    /Encoding Latin1Encoding def
    currentdict
    definefont
    pop
  end
} bind def\n";
}

// static in the Metrics class.
const font::Metrics::Encoding* font::Metrics::encoding;

font::Metrics::Metrics(std::istream & source) 
  :ascender(0), descender(0), underline_position(0), underline_thickness(0),
   cap_height(0), x_height(0)
{
  // Never deallocated, but there is only one, so deallocation on program
  // termintation is ok.
  if(!encoding) encoding = new Encoding();
  
  for(unsigned int i = 0; i<=255; ++i)
    width_x[i] = 0.625;		// Rather sane default for all fonts ...

  string key;
  while(source >> key) {
    if(key == "Comment");	// ignore the comment!
    else if(key == "Descender") {
      source >> descender;
      descender *= -0.001;	// Metrics use neg values here, I want positive.
    } else if(key == "Ascender") {
      source >> ascender;
      ascender *= 0.001;
    } else if(key == "CapHeight") {
      source >> cap_height;
      cap_height *= 0.001;
    } else if(key == "XHeight") {
      source >> x_height;
      x_height *= 0.001;
    } else if(key == "UnderlinePosition") {
      source >> underline_position;
      underline_position *= 0.001;
    } else if(key == "UnderlineThickness") {
      source >> underline_thickness;
      underline_thickness *= 0.001;
    } else if(key == "FontBBox") {
      float lx, ly, hx, hy;
      source >> lx >> ly >> hx >> hy;
      if(ascender == 0.0) ascender = hy * 0.001;
      if(descender == 0.0) descender = ly * -0.001;
      if(cap_height == 0.0) cap_height = hy * 0.001;

    } else if(key == "C") {
      int ch;
      float wx;
      string s1, t1, s2, t2, entity, s3;
      source >> ch >> s1 >> t1 >> wx >> s2 >> t2 >> entity >> s3;
#ifdef DEBUG_LOG
      cerr << "Find '" << entity << "' ... ";
#endif
      ch = (*encoding)(entity);
#ifdef DEBUG_LOG
      cerr << " found " << ch << "!\n";
#endif
      if(ch > 0 && ch <= 255)
	width_x[ch] = wx * 0.001;
#ifdef DEBUG_FONTMETRICS
      else
	std::cerr << "Width for illegal char " << entity << " (ignored).\n";
    } else {
      std::cerr << "Unknown key: " << key << '\n';
#endif
    }
    // Todo: maybe source.ignore(??, '\n') instead?
    getline(source, key);		// Read the rest of the line ...
  }

  width_x[160] = width_x[' '];	// 160 is &nbsp; Todo: don't fake that here,
				// use a 'cleaner' method.
}

float
font::Metrics::getWidth(const string &str) const {
  float result = 0;
  for(string::const_iterator i = str.begin(); i != str.end(); ++i) {
    const unsigned char uc = static_cast<const unsigned char>(*i);
    result += width_x[uc];
  }
  return result;
}

#ifdef DEBUG_FONTMETRICS
main(int argc, char **argv) {
  if(argc != 3) {
    cerr << "Wrong no of args. (" << argc << ")\n";
    exit(2);
  }
  font::Metrics afm(argv[1]);
  cout << "Width of '" << argv[2] << "' is: " << afm.getWidth(argv[2]);
  
}
#endif

// The ISOLatin1Encoding according to the PostScript Language Reference Manual,
// second edition. I am not sure about all glyphs.
// Note that these are the postscript names for the character entities. They
// *might* differ from the ISO-8859-1 names (but the glyphs should be the
// same).
// Note: the number are octal
const char *const font::Metrics::Encoding::init[257] = 
{
  //000 
  ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef",
  ".notdef", 
  //010
  ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef",
  ".notdef", 
  //020
  ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef",
  ".notdef", 
  //030
  ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef",
  ".notdef", 
  //040
  "space", "exclam", "quotedbl", "numbersign", "dollar", "percent",
  "ampersand", "quotesingle",
  //050
  "parenleft", "parenright", "asterisk", "plus", "comma", 
  "minus", "period", "slash",
  //060
  "zero", "one", "two", "three", "four", "five", "six", "seven",
  //070
  "eight", "nine", "colon", "semicolon", "less", "equal", "greater", 
  "question",
  //100
  "at", "A", "B", "C", "D", "E", "F", "G",
  //110
  "H", "I", "J", "K", "L", "M", "N", "O",
  //120
  "P", "Q", "R", "S", "T", "U", "V", "W", 
  //130
  "X", "Y", "Z", "bracketleft", "backslash", "bracketright", "asciicircum",
  "underscore",
  //140
  "grave", "a", "b", "c", "d", "e", "f", "g",
  //150
  "h", "i", "j", "k", "l", "m", "n", "o",
  //160
  "p", "q", "r", "s", "t", "u", "v", "w",
  //170
  "x", "y", "z", "braceleft", "bar", "braceright", 
  "asciitilde", ".notdef",
  //200
  ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef",
  ".notdef", ".notdef",
  //210
  ".notdef", ".notdef", ".notdef", ".notdef", ".notdef", ".notdef",
  ".notdef", ".notdef",
  //220
  "dotlessi", "grave", "acute", "circumflex", "tilde", "macron", //?
  "breve", "dotaccent",
  //230
  "dieresis", //?
  ".notdef", "ring", "cedilla", //?
  ".notdef", ".notdef", //?
  "ogonec", "caron", 
  //240 = &nbsp;
  "nbspace", "exclamdown", "cent", "sterling", "currency", "yen", "brokenbar",
  "section",
  //250
  "dieresis", "copyright", "ordfeminine", "guillemotleft", "logicalnot",
  "hyphen", "registered", "macron", // radicalex?
  //260
  "degree", "plusminus", "twosuperior", "threesuperior",
  "acute", // minute?
  "mu", "paragraph", "periodcentered", // middot?
  //270
  "cedilla", "onesuperior", "ordmasculine", "guillemotright", "onequarter",
  "onehalf", "threequarters", "questiondown", 
  //300
  "Agrave", "Aacute", "Acircumflex", "Atilde", "Adieresis", "Aring", "AE",
  "Ccedilla", 
  //310
  "Egrave", "Eacute", "Ecircumflex", "Edieresis", "Igrave", "Iacute",
  "Icircumflex", "Idieresis",
  //320
  "Eth", "Ntilde", "Ograve", "Oacute", "Ocircumflex", "Otilde",
  "Odieresis", "multiply",
  //330
  "Oslash", "Ugrave", "Uacute", "Ucircumflex", "Udieresis", "Yacute",
  "Thorn", "germandbls",
  //340
  "agrave", "aacute", "acircumflex", "atilde", "adieresis", "aring", "ae",
  "ccedilla",
  //350 
  "egrave", "eacute", "ecircumflex", "edieresis", "igrave", "iacute",
  "icircumflex", "idieresis",
  //360
  "eth", "ntilde", "ograve", "oacute", "ocircumflex", "otilde",
  "odieresis", "divide", 
  //370
  "oslash", "ugrave", "uacute", "ucircumflex", "udieresis", "yacute",
  "thorn", "ydieresis", 
  NULL
};
