///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <stdexcept>

#include <util/stringutil.h>

#include "fontinfo.hh"
#include "line.hh"
#include "paragraph.hh"
#include "pscanvas.hh"
#include "pdfcanvas.hh"
#include "blockcontainer.hh"

class PsConverter : public XMLParserCallback {
public:
  PsConverter(xml2ps::Canvas& output) : out(output), root(out), node(&root) {
  }
  virtual ~PsConverter() {}

  void start_element(const string& n, const XMLPropertyHash &p) {
    try {
    xml2ps::Element* newnode;
    if (n == "para") {
      newnode = new xml2ps::Paragraph(out, *node, n, p);
    } else if (n == "font") {
      newnode = new xml2ps::TextContainer(*node, n, p);
    } else if (n == "bp") {
      newnode = new xml2ps::BreakPoint(*node);
    } else if (n == "leader") {
      newnode = new xml2ps::LeaderNode(*node, p);
    } else if (n == "block-container") {
      newnode = new xml2ps::BlockContainer(out, *node, n, p);
    } else {
      throw std::runtime_error("Unknown element " + n + " in source");
    }
    node->add(newnode);
    node = newnode;
    } catch(const std::exception& err) {
      std::cerr << "problem in start_element(" << n << "): " << err.what() 
		<< std::endl;
      throw;
    }
  }
  void end_element(const string &n) {
    try {
    node->close();
    node = &node->getParent();
    } catch(const std::exception& err) {
      std::cerr << "problem in end_element(" << n << "): " << err.what()
		<< std::endl;
      throw;
    }
  }
  void characters(const string& s) {
    try {
    using xml2ps::TextContainer;
    if(TextContainer* tc = dynamic_cast<TextContainer*>(node)) {
      unsigned char* t = new unsigned char[s.length()];
      int outlen = s.length();
      int inlen = outlen;
      UTF8Toisolat1(t, &outlen, (const unsigned char*)(s.c_str()), &inlen);
      string text((char*)t, outlen);
      tc->add(text);
    }
    } catch(const std::exception& err) {
      std::cerr << "problem in characters: " << err.what() << std::endl;
      throw;
    }
    
  }

  void warning(const string &s) {
    std::cerr << "WARNING: " << s << std::endl;
  }
  void error(const string &s) {
    throw std::runtime_error(s.c_str());
  }
private:
  xml2ps::Canvas& out;
  xml2ps::RootNode root;
  xml2ps::Element * node;
};

xml2ps::Canvas *out;

template<>
XMLParser<PsConverter>::XMLParser() {
  std::cerr << "In my specialized constructor" << std::endl;
  // Note: This specification constructor _replaces_ the default, there is no
  // way to call the default from here (and I wouldn't want that, anyway).
  xmlSAXHandler sax_handler = {
    NULL,		// internalSubset
    NULL,		// isStandalone
    NULL,		// hasInternalSubset
    NULL,		// hasExternalSubset
    NULL,		// resolveEntity
    _get_entity,	// getEntity
    NULL,		// entityDecl
    NULL,		// notationDecl
    NULL,		// attributeDecl
    NULL,		// elementDecl
    NULL,		// unparsedEntityDecl
    NULL,		// setDocumentLocator
    _start_document,	// startDocument
    _end_document,	// endDocument
    _start_element,	// startElement
    _end_element,	// endElement
    NULL,		// reference
    _characters,	// characters
    NULL,		// ignorableWhitespace
    NULL,		// processingInstruction
    _comment,		// comment
    _warning,		// warning
    _error,		// error
    _fatal_error,	// fatalError
    NULL,		// getParameterEntity
    NULL,		// cdataBlock
    NULL		// externalSubset
  };
  
  if(!out) throw std::runtime_error("Ouput stream not initialized");
  _parser_callback = new PsConverter(*out);
  _context = xmlCreatePushParserCtxt(&sax_handler, this, NULL, 0, NULL);
}

enum OutputFormat {ps, pdf};

int main(int argc, char* argv[]) {
  try {
    xml2ps::FontInfo::FontPaths fontPaths;
    std::vector<xml2ps::PageBoundary> page_sizes;
    bool extra_pages = true;
    OutputFormat outputFormat=ps;
    
    for(int i = 1; i < argc; ++i){
      if(argv[i][0] == '-'){
	if(argv[i]==std::string("-of")){      // output format
	  i++;
	  if(i == argc)
	    throw std::runtime_error("The \"-of\" option needs a format"
				     " argument");
	  if(argv[i]==std::string("ps"))
	    outputFormat=ps;
	  else if(argv[i]==std::string("pdf"))
	    outputFormat=pdf;
	  else
	    throw std::runtime_error("Unknown output format: "
				     +std::string(argv[i]));
	}
	else if(argv[i]==std::string("-f")){ // Font path
	  i++;
	  if(i<argc)
	    fontPaths.push_back(argv[i]);
	  else
	    throw std::runtime_error("The \"-f\" option needs a path"
				     " argument");
	}
	else if(argv[i]==std::string("-p")){ // Page boundary
	  if(i == argc-1)
	    throw std::runtime_error("The \"-p\" option needs a page format"
				     " argument");
	  page_sizes.push_back(xml2ps::PageBoundary(argv[++i]));
	}
	else if(argv[i]==std::string("-b")){ // "block" - rectangular obstacle
	  if(page_sizes.empty())
	    throw std::runtime_error("Obstacle before page boundary"
				     " is not allowed");
	  if(i == argc-1)
	    throw std::runtime_error("The \"-b\" option needs a"
				     " boundary geometry argument");	  
	  page_sizes.back().addObstacle
	    (new xml2ps::SquareObstacle(argv[++i]));
	}
	else if(argv[i]==std::string("-e")){
	  extra_pages = false;
	}
	else{
	  throw std::runtime_error(string("Unknown argument: ") + argv[i]);
	}
      }
      else
	std::cerr << "Filename args (" << argv[i]
		  << ") not yet supported.  Sorry." << std::endl;
    }

    xml2ps::FontInfo::init(&fontPaths);

    switch(outputFormat)
      {
      case ps:
	out = new xml2ps::PsCanvas(std::cout.rdbuf(), 
				   page_sizes, extra_pages);
	break;
      case pdf:
	out = new xml2ps::PDFCanvas(std::cout.rdbuf(), 
				    page_sizes, extra_pages);
	break;
      default:
	throw std::runtime_error("Uncaught output format");
	break;
      }

    XMLParser<PsConverter> parser;
    
    // Note; xml++ doesn't know about istream, it probably should ...
    string line;
    while(getline(std::cin, line))
      parser.parse_chunk(line + ' ');

    delete out; out = 0;
  } catch (const xml2ps::out_of_pages&) {
    std::cerr << "WARNING: Out of pages, document truncated" << std::endl;
    delete out;
    
  } catch (const std::exception& err) {
    std::cerr << "Error: " << err.what() << std::endl;
    delete out;
    throw;
  } catch (...) {
    std::cerr << "Main caught something that wasn't even an exception!"
	      << std::endl;
    delete out;
    throw;
  }
}
