///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <iostream>
#include "xly.hh"

#include <stdexcept>
#include <algorithm>
#include <memory>		// auto_ptr

#include <util/stringutil.h>

using std::string;

namespace {

  bool haswhite(std::istream& in) {
    char ch;
    if(!in.get(ch))
      return false;
    in.putback(ch);
    return whitespace(ch);
  }

  template<class Value>
  class ValueUnit {
  public:
    ValueUnit() : value_(0) {}
    ValueUnit(const Value& value, const string& unit)
      : value_(value), unit_(unit) {}
    const Value& value() const { return value_; }
    const string& unit() const { return unit_; }
    
  private:
    Value value_;
    string unit_;
    friend std::istream& operator >> <> (std::istream& in, ValueUnit<Value>& vu);
  };
  
  template<class Value>
  std::istream& operator >> (std::istream& in, ValueUnit<Value>& vu) {
    // Todo: Don't eat whitespace before unit!  Or at least, dont require it.
    vu.unit_ = "";
    if(in >> vu.value_) {
      if(!in.eof()) {
	in >> vu.unit_;
	if(in.eof()) in.clear();
      }
      else
	in.clear();
    }
    return in;
  }
}

// - - - Attributes - - -

string
xml2ps::Attributes::get(const string& name, const string& defaultvalue) 
  const 
{
  XMLPropertyHash::const_iterator i = p_.find(name);
  if ( i != p_.end() )
    return i->second->value();
  else
    return defaultvalue;
}

float
xml2ps::Attributes::get(const string& name, const float& defaultvalue,
			const float embase) const {
  XMLPropertyHash::const_iterator i = p_.find(name);
  if ( i != p_.end() ) {
    ValueUnit<float> value = to<ValueUnit<float> >(i->second->value());
    if(value.unit() == "" || value.unit() == "pt")
      return value.value();
    else if(value.unit() == "%")
      return 0.01 * value.value() * defaultvalue;
    else if(value.unit() == "em"
	    // Ugly workaround: On some systems float reading eats the "e".
	    || value.unit() == "m") {
      if(embase != 0)
	return value.value() * embase;
      else return value.value() * defaultvalue;
    } else {
      std::cerr << "Bad unit: ''" <<  value.unit() << "'' in ''"
		<< i->second->value() << "''" << std::endl;
      throw std::runtime_error("Bad ''unit'': " + value.unit());
    }
  } else
    return defaultvalue;
}

// - - - Node - - -
xml2ps::Node*
xml2ps::Node::nodeBefore() const {
  Node* node = getParent().nodeBefore(this);
  if(node) return node;
  else
    // This is the first node in parent, return node before parent.
    return getParent().nodeBefore();
}

// - - - TextNode - - -
const xml2ps::FontInfo&
xml2ps::TextNode::getFont() const { return getParent().getFont(); }


// - - - Element - - -

template<>
xml2ps::Element::Align to<xml2ps::Element::Align>(const string& a) {
  if(a == "left")    return xml2ps::Element::left;
  if(a == "justify") return xml2ps::Element::justify;
  if(a == "right")   return xml2ps::Element::right;
  if(a == "center")  return xml2ps::Element::center;
  throw std::runtime_error("Bad alignment \"" + a + "\"");
}

namespace {
  xml2ps::FontInfo getFont(const xml2ps::Attributes& attr, 
			   xml2ps::Element& parent) {
    const string name(attr.get("font-family", parent.getFontName()));
    const float size(attr.get("font-size", parent.getFontSize(), parent.getFontSize()));
    const float letter_spacing(attr.get("letter-spacing", 0, size));
    return xml2ps::FontInfo(name, size, letter_spacing);
  }
}

xml2ps::Element::Element(Element& parent, const string& n, const Attributes& attr)
  : Node(&parent), name(n), 
    font_info(::getFont(attr, parent)),
    align(to<Align>(attr.get("align", "left"))),
    underline(attr.get("underline", 0) > 0),
    baseline(attr.get("baseline", 0, font_info.getSize()))
{}

xml2ps::Element::Element(Element& parent, const string&n, const FontInfo& fi)
  : Node(&parent), name(n), font_info(fi),
    align(parent.getAlign()), underline(false), baseline(0)
{}

void xml2ps::Element::add(Node* node) {
  nodes.push_back(node);
}
void xml2ps::Element::debug(std::ostream& out, bool nl) {
  getParent().debug(out, false);
  out << '|' << name;
  if(nl) out << '>' << std::endl;
}
void xml2ps::Element::close() {
//   cerr << '/';
//   debug(cerr);
}

xml2ps::Node*
xml2ps::Element::nodeBefore(const xml2ps::Node* node) const {
  if(!node) return Node::nodeBefore();
  NodeVect::const_iterator i = find(nodes.begin(), nodes.end(), node);
  if(i == nodes.end()) throw std::runtime_error("Node before non-child node");
  
  if(i == nodes.begin()) return 0;
  return *(--i);
};

float
xml2ps::Element::getWidth() const {
  float width = 0;
  for(NodeVect::const_iterator i = nodes.begin(); i != nodes.end(); ++i)
    width += (*i)->getWidth();

  // Todo: Add support for margins / padding ...
  return width;
}

namespace {
  // Special marker-object
  class Underline {
  public:
    Underline(xml2ps::Canvas &c, const xml2ps::FontInfo& font)
      : canvas(c),
	pos(font.getUnderlinePos()), thick(font.getUnderlineThickness())
      { canvas.underlineFrom(pos); }
    ~Underline() {
      canvas.underlineTo(pos, thick);
    }
  private:
    xml2ps::Canvas& canvas;
    float pos, thick;
  };
  
  class BaseShift {
  public:
    BaseShift(xml2ps::Canvas& c, const float& shift)
      : canvas(c), dy(shift)
      { canvas.moverel(0, dy); }
    ~BaseShift() { canvas.moverel(0, -dy); }
  private:
    xml2ps::Canvas& canvas;
    float dy;
  };
}

xml2ps::Element::CharSpaceCount
xml2ps::Element::countChars(const xml2ps::Node* from, const xml2ps::Node* to)
  const 
{
  int chars = 0, spaces = 0;
  const Node* actualfrom = from;
  while(actualfrom && &actualfrom->getParent() != this)
    actualfrom = &actualfrom->getParent();
  NodeVect::const_iterator start = (!actualfrom
				    ? nodes.begin()
				    : find(nodes.begin(), nodes.end(), 
					   actualfrom));
  NodeVect::const_iterator end = std::find(start, nodes.end(), to);
  if(end != nodes.end()) ++end;	// to is inclusive!

  const Node* n = 0;
  for(NodeVect::const_iterator i = start; i != end; ++i) {
    n = *i;
    if(dynamic_cast<WhiteSpaceNode*>(*i)) {
      ++spaces;
      
    } else if(TextNode* tn = dynamic_cast<TextNode*>(*i)) {
      chars += tn->getContent().length();
      
    } else if(Element *elem = dynamic_cast<Element*>(*i)) {
      if(from == actualfrom) from = 0; // level of the from
      CharSpaceCount t = elem->countChars(from, to);
      chars += t.first;
      spaces += t.second;
    }
    from = 0;
  }
  return std::make_pair(chars, spaces);
}

const xml2ps::Node* 
xml2ps::Element::printPart(Canvas& canvas, const Node* from, const Node* to,
			   const float& whitewidth, const float& cwidth) const
{
  const Node* actualfrom = from;
  while(actualfrom && &actualfrom->getParent() != this)
    actualfrom = &actualfrom->getParent();
  NodeVect::const_iterator start = (!actualfrom
				    ? nodes.begin()
				    : find(nodes.begin(), nodes.end(), 
					   actualfrom));
  NodeVect::const_iterator end = std::find(start, nodes.end(), to);
  if(end != nodes.end()) ++end;	// to is inclusive!
  
  const FontInfo font = FontInfo::WidenFont(getFont(), cwidth);
  canvas.setfont(font);
  std::auto_ptr<Underline> ul(underline? new Underline(canvas, font): 0);
  std::auto_ptr<BaseShift> bs(baseline!=0? new BaseShift(canvas, baseline): 0);
  
  const Node* n = 0;
  for(NodeVect::const_iterator i = start; i != end; ++i) {
    n = *i;
    if(dynamic_cast<WhiteSpaceNode*>(*i)) {
      canvas.whitespace(font.getWidth(" ") * whitewidth);
      
    } else if(TextNode* tn = dynamic_cast<TextNode*>(*i)) {
      canvas.show(tn->getContent());
      
    } else if(Element *elem = dynamic_cast<Element*>(*i)) {
      if(from == actualfrom) from = 0; // level of the from
      if(elem->printPart(canvas, from, to, whitewidth, cwidth) == to)
	return to;
      
      canvas.setfont(font);
    }
    from = 0;
  }
  return n;
}

string
xml2ps::Element::d() const { return name + "(" + tostr(nodes.size()) + ")"; }


// - - - BreakPoint - - -

// noop
const xml2ps::Node*
xml2ps::BreakPoint::printPart(xml2ps::Canvas& canvas, 
			      const xml2ps::Node* from, const xml2ps::Node* to,
			      const float& whitewidth, const float& cwidth)
  const
{
  return this;
}

// - - - LeaderNode - - -

xml2ps::LeaderNode::LeaderNode(Element& parent, const Attributes& attr)
  : Element(parent, "leader", attr), 
    // Todo: Get column width for percentages
    width_(attr.get("width", 100 /*%*/, getFont().getSize()))
{}


// Neither the whitespace- or the char width is of any concern, and there are
// no child nodes.
const xml2ps::Node*
xml2ps::LeaderNode::printPart(xml2ps::Canvas& canvas, 
			      const xml2ps::Node*, const xml2ps::Node*,
			      const float&, const float&) const {
  canvas.moverel(width_, 0);
  return this;
}
// - - TextContainer - -

void xml2ps::TextContainer::add(Node* node) {
  makeTextParts();
  Element::add(node);
}
void xml2ps::TextContainer::close() {
  makeTextParts();
  Element::close();
}

void xml2ps::TextContainer::makeTextParts() {
#ifdef USE_OLD_STRSTREAM
  std::istrstream in(text.c_str(), text.length());
#else
  std::istringstream in(text);
#endif
  
  // Do not put a whitespace at the start
  if(haswhite(in) && !nodes.empty()) Element::add(new WhiteSpaceNode(*this));
  
  string word;
  while(in >> word) {
    Element::add(new TextNode(*this, word));
    if(haswhite(in)) {
      Element::add(new WhiteSpaceNode(*this));
    }
  }
  text = "";
}
