///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
// XML for the layout engine
#ifndef X2P_XLY
#define X2P_XLY
#include <string>
#include <vector>
#include <xml++.h>
#include "fontinfo.hh"
#include "canvas.hh"

namespace xml2ps {
  
  class Attributes {
  public:
    Attributes(const XMLPropertyHash& p) : p_(p) {}
    
    string get(const string& name, const string& defaultvalue = "") const;    
    float get(const string& name, const float& defaultvalue,
	      const float embase = 0) const;
    
  private:
    const XMLPropertyHash p_;
  };
  
  class Element;
  
  class Node {
  public:
    Node(Element* parent) : parent_(parent) {}
    
    virtual ~Node() {}
    Element& getParent() const   { 
      assert(parent_);
      return *parent_; 
    }
    Node* nodeBefore() const;
    virtual float getWidth() const = 0;
  private:
    Element* parent_;
    
    Node();
    Node(const Node&);
    void operator= (const Node&);
  };

  class TextNode : public Node {
  public:
    TextNode(Element& parent, const string& text)
      : Node(&parent), text_(text) {}
    const string& getContent() const { return text_; }
    const FontInfo& getFont() const;
    float getWidth() const { return getFont().getWidth(text_); }
    
  private:
    string text_;
  };

  class WhiteSpaceNode : public TextNode {
  public:
    explicit WhiteSpaceNode(Element& parent) : TextNode(parent, " ") {}
  };
  
  class Element : public Node {
  public:
    typedef std::vector<Node*> NodeVect;
    typedef enum { left, justify, right, center } Align;
    typedef std::pair<int, int> CharSpaceCount;
    
    Element(Element& parent, const string& n, const Attributes& attr);
    Element(Element& parent, const string&n, const FontInfo& fi);
    
    virtual void add(Node* node);
    virtual void debug(std::ostream& out, bool nl = true);
    virtual void close();

    NodeVect::const_iterator begin() const { return nodes.begin(); }
    NodeVect::const_iterator end() const { return nodes.end(); }
    Node* nodeBefore(const Node* node = 0) const;
    
    const FontInfo& getFont() const { return font_info; }
    const string& getFontName() const { return font_info.getName(); }
    const float& getFontSize() const { return font_info.getSize(); }
    Align getAlign() const { return align; }
    
    float getWidth() const;

    // Returns (number of non-whitespace characters, number of whitespace)
    CharSpaceCount countChars(const Node* from, const Node* to) const;

    virtual const Node* printPart(Canvas& canvas, const Node* from,
				  const Node* to, const float& wh_width,
				  const float& cwidth) const;
    string d() const;
  private:
    string name;
    FontInfo font_info;
    Align align;
    bool underline;
    float baseline;		// relative to parent / hbox
  protected:
    NodeVect nodes;
  };
  
  class BreakPoint : public Element {
  public:
    BreakPoint(Element& parent) : Element(parent, "bp", parent.getFont()) {}
    const Node* printPart(Canvas& canvas, const Node* from, const Node* to,
			  const float& whitewidth, const float& cwidth) const;
  };
  
  class LeaderNode : public Element {
  public:
    LeaderNode(Element& parent, const Attributes& attr);
    float getWidth() const { return width_; }

    const Node* printPart(Canvas& canvas, const Node* from, const Node* to,
			  const float& whitewidth, const float& cwidth) const;
  private:
    float width_;
  };

  // An element that can contain both text and other Elements
  // text contents is automatically split into textnodes.
  class TextContainer : public Element {
  public:
    TextContainer(Element& parent, const string& n, const Attributes& attr)
      : Element(parent, n, attr) {}

    void add(const string& newtext) { text += newtext; }
    virtual void add(Node* node);
    virtual void close();
    
  private:
    void makeTextParts();
    
    string text;
  };
  
  
}

#endif
