///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "fontmanager.hh"

#include <fstream>
#include <iostream>
#include <sys/types.h>
#include <dirent.h>

#include <util/stringutil.h>
#include <util/filesys.h>

namespace font{
FontManager::FontManager() {
}

// TODO: should we really allow more than one font to
// have the same alias?
std::string
FontManager::unalias(std::string fontname, bool recursive) const {
  FontAliases::const_iterator i = fontaliases.find(fontname);
  if(i == fontaliases.end())
    return fontname;
  else if(recursive)
    return unalias(i->second); // recursive aliases possible
  else
    return i->second;
}

void
FontManager::fontAlias(std::string alias, std::string fontname) {
  fontaliases.insert(std::make_pair(alias, fontname));
}

void
FontManager::reportFonts(std::ostream &out) const {
  using std::endl;
  out << "Fonts:" << endl;
  for(FontFiles::const_iterator i = fontfiles.begin();
      i != fontfiles.end();
      i++)
    out << "\t   Name: " << i->first << endl
	<< "\tMetrics: " << i->second.first << endl
	<< "\t   File: " << i->second.second << endl << endl;

  out << endl << "Aliases:" << endl;
  for(FontAliases::const_iterator i = fontaliases.begin();
      i != fontaliases.end();
      i++)
    out << "\t" << i->first << ": " << i->second << endl;
} 

void
FontManager::addFontPath(std::string path) {
  if(path.empty())
    return;
  if(path[path.length() - 1] != '/')
    path += '/';
  DIR *dir = opendir(path.c_str());
  if(!dir) {
    std::cerr << "Failed to open " << path << std::endl;
    return;
  }
  dirent *d_ent;
  while((d_ent = readdir(dir))) {
      std::string filename(d_ent->d_name); // the only field defined in POSIX
      std::string ext = suffix(filename);
      filename = path + filename;
      if(ext == "afm") // Adobe Font Metrics
	checkOutAFM(filename);
      else if(ext == "pfa" || ext == "pfb") // Type 1
	checkOutType1(filename);
      else if(ext == "ttf") { // TrueType
	// We don't support TrueType yet.
      }
  }
  closedir(dir);
}

void
FontManager::checkOutAFM(std::string filename) {
  std::ifstream in(filename.c_str());
  if(!in) {
    std::cerr << "Failed to open " << filename << std::endl;
    return;
  }
  std::string tmp;
  std::getline(in, tmp);
  if(!starts_with(tmp, "StartFontMetrics")) {
    std::cerr << filename << " is not an AFM" << std::endl;
    return;
  }
  while(in && !starts_with(tmp, "FontName"))
    std::getline(in, tmp);
  if(starts_with(tmp, "FontName")) {
    std::string fontname(strip_whitespace(tmp.substr(9), true, true));
    FontFiles::iterator i = fontfiles.find(fontname);
    if(i == fontfiles.end())
      fontfiles[fontname] = std::make_pair(filename, std::string(""));
    else if(i->second.first.empty()) // don't overwrite
      i->second.first = filename;
  } else
    std::cerr << filename << " has no FontName" << std::endl;
}

void
FontManager::checkOutType1(std::string filename) {
  std::ifstream in(filename.c_str());
  if(!in) {
    std::cerr << "Failed to open " << filename << std::endl;
    return;
  }
  std::string tmp;
  std::getline(in, tmp);
  while(in && !starts_with(tmp, "/FontName"))
    std::getline(in, tmp);
  if(starts_with(tmp, "/FontName")) {
    std::string fontname(tmp.substr(tmp.find('/', 1) + 1));
    fontname = fontname.substr(0, fontname.find(' '));
    FontFiles::iterator i = fontfiles.find(fontname);
    if(i == fontfiles.end())
      fontfiles[fontname] = std::make_pair(std::string(""), filename);
    else if(i->second.second.empty() 
	    || (suffix(filename) == "pfa" 
		&& suffix(i->second.second) == "pfb")) // prefer pfa over pfb
      i->second.second = filename;
  } else
    std::cerr << filename << " has no FontName" << std::endl;
}

std::pair<std::string, std::string>
FontManager::getFontFiles(const string &fontname) const {
  FontFiles::const_iterator i = fontfiles.find(fontname);
  std::string filename;
  if(i != fontfiles.end())
    return std::make_pair(i->second.first, i->second.second);
  else {
    FontAliases::const_iterator l = fontaliases.lower_bound(fontname);
    FontAliases::const_iterator u = fontaliases.upper_bound(fontname);
    while(l != u){
      try{
	return getFontFiles(l->second);
      } catch (FontError) {}
      l++;
    }
  }
  throw FontError("No font files found for " + fontname);
}

void
FontManager::loadFont(const string &fontname) {
  std::string filename(getFontFiles(fontname).first);
  std::ifstream metricsfile(filename.c_str());

  if(!metricsfile)
    throw FontError("No font metrics " + filename + " found for "
		    + fontname);
  else {
    font::Metrics *metrics = new font::Metrics(metricsfile);
    // Todo: delete metrics in destructor
    metrics_map[fontname] = metrics; // Todo: check if it already exists
  }
}

const font::Metrics* 
FontManager::getFont(const string &fontname){
  MetricsMap::const_iterator i = metrics_map.find(fontname);
  if(i == metrics_map.end())
    {
      loadFont(fontname);
      i = metrics_map.find(fontname);
      if(i == metrics_map.end())
	throw FontError("No font metrics found for "
			+ fontname);
    }
  return i->second;
}
}

