///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "basicframe.h"
#include "util/rectboundary.h"
#include <util/stringutil.h>
#include <util/warning.h>
#include <sigc++/sigc++.h>
#include <libxml++/libxml++.h>
#include <gdkmm.h>
#include <fstream>
#include <algorithm>

Basic_Frame::Basic_Frame(Group *parent, float w, float h, 
			 const Glib::ustring& name)
  :Pagent(parent, name)
{
  set_size(w, h);
}

Basic_Frame::Basic_Frame(Group *parent, const xmlpp::Element& node)
  :Pagent(parent, "unnamed")
{
  if(const xmlpp::Attribute* name = node.get_attribute("name"))
    set_name(name->get_value());

  if(const xmlpp::Attribute* locked = node.get_attribute("lock"))
    set_lock(to<bool>(locked->get_value()));

  if(const xmlpp::Attribute* flow = node.get_attribute("flowaround"))
    flow_around = to<bool>(flow->get_value());

  if(const xmlpp::Attribute* margin = node.get_attribute("obstaclemargin")) {
    try{
      obstacle_margin = to<float>(margin->get_value());
    } catch( std::runtime_error e){
      throw Error::Read("\"" + margin->get_value() 
			+ "\" is not a valid numerical value"
			" for obstaclemargin");
    }
  }
  
  if(const xmlpp::Attribute* w = node.get_attribute("width"))
    width = to<float>(w->get_value());
//   else
//     throw Error::Read("No \"width\" attribute found in frame");
  if(const xmlpp::Attribute* h = node.get_attribute("height"))
    height = to<float>(h->get_value());
//   else
//     throw Error::Read("No \"height\" attribute found in frame");

  if(const xmlpp::Attribute* matrix = node.get_attribute("matrix")) {
    try{
      set_matrix(to<Matrix>(matrix->get_value()));
    } catch(std::runtime_error) {
      throw Error::Read("Error reading \"matrix\" attribute in frame.");
    }
  } else
    throw Error::Read("No \"matrix\" attribute found in <frame>");
}

xmlpp::Element *Basic_Frame::save(xmlpp::Element& parent_node) const {
  xmlpp::Element *node = parent_node.add_child("frame");
  if(name != "")
    node->add_attribute("name", name);

  if(resizable) {
    node->add_attribute("width", tostr(width));
    node->add_attribute("height", tostr(height));
  }

  node->add_attribute("matrix", tostr(get_matrix()));
  node->add_attribute("lock", tostr(get_lock()));
  node->add_attribute("flowaround", tostr(flow_around));
  node->add_attribute("obstaclemargin", tostr(obstacle_margin));

  return node;
}

Basic_Frame::~Basic_Frame() {
}

void Basic_Frame::print(std::ostream &out, bool grayscale) const {
}

Boundary Basic_Frame::get_box() const {
  return RectBoundary::create(get_matrix(), width, height);
}

Boundary Basic_Frame::get_obstacle_boundary() const {
  if(flow_around) {
    // Todo: Isn't this bogus? width / height is before matrix, but the
    // addition for the obstacle should be after.
    return RectBoundary::create(Matrix::translation(-obstacle_margin, 
						    -obstacle_margin)
				* get_matrix(),
				width + 2 * obstacle_margin, 
				height + 2 * obstacle_margin);
    
  } else
    return Boundary();
};

void Basic_Frame::draw_content_error(View& view, 
				     Glib::RefPtr<const Gdk::Pixmap> pixmap) 
{
  if(pixmap) {
    // p is the top left corner, since that is what draw_pixmap wants.
    const Gdk::Point p = 
      view.pt2scr(get_matrix().transform(Vector(0, height)));
    view.get_win()->draw_drawable(view.get_gc(), pixmap, 
				  0, 0, p.get_x(), p.get_y());
  }
}

void Basic_Frame::draw_content_wait(View& view) {
  draw_content_error(view, view.get_wait_pixmap());
}

void Basic_Frame::draw_content_missing(View& view) {
  draw_content_error(view, view.get_missing_pixmap());
}

void Basic_Frame::draw_content_broken(View& view) {
  draw_content_error(view, view.get_broken_pixmap());
}
