///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "cachedframe.h"
#include "fileerrors.h"
#include "globals.h"
#include "processman.h"
#include <gdkmm.h>
#include <sigc++/sigc++.h>
#include <util/warning.h>
#include <util/os.h>
#include <util/filesys.h>
#include "config.h"
#include "safeslot.h"

void Cached_Frame::clear_picture() {
  if(picture)
    picture.clear();
}

bool Cached_Frame::draw_content(View& view, bool reshaping) {
  if(no_picture_generated) {
    no_picture_generated = false; // or it will never be generated again
    draw_content_broken(view);
    return true;
  }
  if(picture) {
    // p is the top left corner, since that is what draw_pixmap wants.
    const Gdk::Point p = 
      view.pt2scr(get_matrix().transform(Vector(0, height)));
    view.get_win()->draw_pixbuf(view.get_gc(), picture, 0, 0,
				p.get_x(), p.get_y(), -1, -1,
				Gdk::RGB_DITHER_NORMAL, 0, 0);
    return true;
  } else {
    // TODO: is_rehaping has been removed, what to do?
    if(!reshaping)  {	// generate_picture takes too long time
      try {
	generate_picture(view);
	draw_content_wait(view);
      } catch(Gen_Pic_Error e) {
	debug << get_name() << ": " << e.what() << std::endl;
	switch(e.type) {
	case ASSOCIATION: draw_content_missing(view); break;
	case GENERATION:  draw_content_broken(view);  break;
	case ZEROSIZE:                                break;
	}
      }
    } else // if reshaping
      draw_content_wait(view);
    return false;
  }
}

namespace {
  void do_clear_picture(Pagent* frame) {
    if(Cached_Frame *cf = dynamic_cast<Cached_Frame*>(frame)) {
      cf->clear_picture();
      // TODO: we don't really want this for all kinds of geometry changes
    }
  }
}

Cached_Frame::Cached_Frame(Group *parent, 
			   float w, float h, const Glib::ustring &name) :
  Basic_Frame(parent, w, h, name), no_picture_generated(false),
  white_is_transparent(false),
  picture(0), pid(-1)
{
  proc_stop_connection = process_manager.process_stopped.connect
    (safeslot(*this, &Cached_Frame::process_stopped));
  tmp_file = process_manager.find_new_name();
  
  geometry_changed_signal.connect(slot(do_clear_picture));
}

Cached_Frame::Cached_Frame(Group* parent, const xmlpp::Element& node)
  :Basic_Frame(parent, node), no_picture_generated(false),
   white_is_transparent(false),
   picture(0), pid(-1)
{
  proc_stop_connection = process_manager.process_stopped.connect
    (safeslot(*this, &Cached_Frame::process_stopped));
  tmp_file = process_manager.find_new_name();

  geometry_changed_signal.connect(slot(do_clear_picture));
}

Cached_Frame::~Cached_Frame() {
  proc_stop_connection.disconnect();
  try { unlink(tmp_file); }
  catch(std::exception& err) {
    warning << "Failed to delete parsed_file: " << err.what()
	    << std::endl;
  }
}

void Cached_Frame::act_on_zoom_factor_change() {
  clear_picture();
  if(pid != -1) {
    process_manager.stop(pid);
    pid = -1;
  }
}

void Cached_Frame::process_stopped(pid_t _pid, bool exited_normally, 
				   int exit_code)
{
  if(_pid != pid)
    return;

  pid = -1;
  debug << _pid << ", exit code: " << exit_code << std::endl;
  if(!exited_normally) {
    warning << "Process " << _pid << " exited abnormally" << std::endl;
    return;
  }

  clear_picture();
  if(exists(tmp_file)) {
    try {
      picture = Gdk::Pixbuf::create_from_file(tmp_file);
    } catch (Glib::Error e) { 
      warning << get_name() << ": " << e.what() << std::endl; 
    }
  
    // Make all white pixels transparent.
    if(config.FakeTransparency.values.front()
       && picture && !picture->get_has_alpha())
      picture = picture->add_alpha(true, 255, 255, 255); 
    // had some trouble with some images becoming totally transparent
  } else warning << get_name() << ": " << tmp_file 
		 << " does not exist" << std::endl; 
  
  no_picture_generated = !picture;
  ready_to_draw_signal(this);
}
