///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "config.h"
#include "widget/wmisc.h"
#include <sstream>

#include <gtkmm/label.h>
#include <gtkmm/button.h>
#include <gtkmm/tooltips.h>
#include <gtkmm/table.h>
#include <gtkmm/entry.h>
#include <gtkmm/stock.h>

#include <util/warning.h>
#include <util/filesys.h>
#include <util/os.h>
#include <defines.h>

#include "safeslot.h"

#define _Str(x) #x
#define Str(x) _Str(x)

P_File::P_File(): 
  ReshapeBoxSize("ReshapeBoxSize", "Size of reshape handles"),
  ScreenResolution("ScreenResolution", "Screen resolution"),
  DefaultResolution("DefaultResolution", "Default image resolution"),
  Landscape("Landscape", "Landscape format is default"),
  FakeTransparency("FakeTransparency", "White is transparent"),
  PaperName("PaperName", "Default paper"),
  PrintCommand("PrintCommand", "Printing command"),
  FontPath("FontPath", "Path(s) to look for font files in"),
  StylesheetPath("StylesheetPath", "Path(s) to look for XSLT files in"),
  PSInterpreter("PSInterpreter", "Command to run Ghostscript"),
  PSViewer("PSViewer", "Command to run external PS viewer"),
  //  fonts_to_include("Fonts", "fonts to include"),
  first_error(true)
{
  // set default values
  
  declare_var(&ReshapeBoxSize);
  ReshapeBoxSize.values.push_back(7);

  declare_var(&ScreenResolution);
  ScreenResolution.values.push_back(72);

  declare_var(&DefaultResolution);
  DefaultResolution.values.push_back(72);
  
  declare_var(&PaperName);
  PaperName.values.push_back("A4");
  declare_var(&Landscape);
  Landscape.values.push_back(false);

  declare_var(&FontPath);
  FontPath.values.push_back(Str(FONTPATH));
  declare_var(&StylesheetPath);
  StylesheetPath.values.push_back(XMLPATH);

  declare_var(&PrintCommand);
  PrintCommand.values.push_back("lpr");

  declare_var(&PSInterpreter);
  PSInterpreter.values.push_back("gs");
  declare_var(&PSViewer);
  // -safer makes gv send the -dSAFER option to gs
  PSViewer.values.push_back("gv -safer");

  declare_var(&FakeTransparency);
  FakeTransparency.values.push_back(true);

  // Try to get the $HOME location
  std::string home=os::get_env("HOME");
  user_file = home.empty() ? "" : home + "/.pptout";
  if(home.empty())
    verbose << "$HOME environment variable is not set" << std::endl;
  global_file = "/etc/pptout";
}

P_File::~P_File()
{
  // Disable writing until we have something to write
  //  if(!user_file.empty())
  //    write(user_file);
  delete dialog;
}

void P_File::read()
{
  if(!user_file.empty() && access(user_file)) {
    debug << "Reading configuration from " << user_file << std::endl;
    Config::File::read(user_file);
  } 
  else {
    if(access(global_file)) {
      debug << "Reading configuration from " << global_file << std::endl;
      Config::File::read(global_file);
    }
    else
      debug << "No configuration file found" << std::endl;
    debug << "Creating configuration file " << user_file << std::endl;
    if(!user_file.empty())
      write(user_file);
  }
}

Gtk::Dialog &P_File::get_dialog()
{
  if(dialog)
    return *dialog;
  dialog=new Gtk::Dialog();
  dialog->set_title("Preferences");
  dialog->set_border_width(border_width);
  //  dialog->set_modal(true);

  dialog->get_vbox()->set_border_width(border_width);
  dialog->get_vbox()->set_spacing(border_width);

  Gtk::Table *table=manage(new Gtk::Table(vars.size(), 2));

  Gtk::Label *instructions = manage(new Gtk::Label());
  dialog->get_vbox()->pack_start(*instructions,
				 Gtk::SHRINK, 5);
  instructions->set_markup("<i>Sorry,"
			   "the preferences are read-only in this version,"
			   "\nbut you can change them by editing</i>"
			   " <tt>$HOME/.pptout</tt>");

  dialog->get_vbox()->pack_start(*table, Gtk::SHRINK, 5);
  int row=0;
  for(Config::Vars::const_iterator i=vars.begin(); i!=vars.end(); i++, row++)
    {
      Gtk::Tooltips tips;
      Gui_Var *g_var;
      if((g_var=dynamic_cast<Gui_Var*>(*i)))
	{
	  Gtk::Label *label=manage(new Gtk::Label(g_var->desc+": "));
	  label->set_alignment(1, label->property_yalign());
	  table->attach(*label, 0, 1, row, row+1);
	  Gtk::Entry *entry=manage(new Gtk::Entry());
	  tips.set_tip(*entry, (*i)->name);
	  table->attach(*entry, 1, 2, row, row+1);
	  std::ostringstream tmp;
	  (*i)->print(tmp);
	  entry->set_text(tmp.str());
	  entry->set_sensitive(false);
	}
    }

  Gtk::Button *cancel_button=manage(new Gtk::Button(Gtk::Stock::CLOSE));
  dialog->get_action_area()->pack_start(*cancel_button, 
					Gtk::PACK_EXPAND_PADDING, 0);
  cancel_button->set_flags(Gtk::CAN_DEFAULT);
  cancel_button->signal_clicked().connect(safeslot(*this, &P_File::dialog_cancel));

  dialog->get_action_area()->show_all();
  cancel_button->grab_default();

  dialog->signal_delete_event().connect(slot(*this, &P_File::dialog_delete));

  return *dialog;
}

void P_File::dialog_show()
{
  get_dialog(); // create it if necessary
  if(dialog)
    dialog->show_all();
}

void P_File::dialog_ok()
{
  if(dialog)
    dialog->hide();
}

void P_File::dialog_cancel()
{
  if(dialog)
    dialog->hide();
}

bool P_File::dialog_delete(GdkEventAny*)
{
  dialog_cancel();
  return true;
}

void P_File::error(const std::string &message)
{
  if(first_error)
    {
      warning << "Errors in config file:" << std::endl;
      first_error=false;
    }
  warning << message << std::endl;
}
