///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "docpropsdialog.h"
#include <list>

#include <gtkmm/box.h>
#include <gtkmm/label.h>
#include <gtkmm/separator.h>
#include <gtkmm/radiobutton.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/combo.h>
#include <gtkmm/stock.h>
#include <gtkmm/image.h>

#include <util/warning.h>

#include "globals.h"
#include "safeslot.h"
#include "paper_sizes.h"
#include "document.h"
#include "window.h"
#include "errordialog.h"
#include "fileerrors.h"
#include "config.h"

#include "widget/filesel.h"
#include "widget/spinner.h"
#include "widget/wmisc.h"

Doc_Props_Dialog::Doc_Props_Dialog():
  DialogWrap(""), // don't know title or parent
  view(0)
{
  template_button = manage(new Gtk::CheckButton("_Use template:", true));

  portrait_button = manage(new Gtk::RadioButton("_Portrait", true));
  landscape_button = manage(new Gtk::RadioButton("_Landscape", true));

  single_sided_button = manage(new Gtk::RadioButton("_Single-sided", true));
  double_sided_button = manage(new Gtk::RadioButton("_Double-sided", true));

  paper_size = manage(new Gtk::Combo());
  file_entry = manage(new File_Entry("Select template"));
  first_page = manage(new Spinner(1, false));

  Gtk::HBox *template_box = manage(new Gtk::HBox(false, double_space));
  Gtk::VBox *sidedness_box = manage(new Gtk::VBox(false, single_space)),
    *format_box3 = manage(new Gtk::VBox(false, single_space));
  Gtk::HBox *first_page_box = manage(new Gtk::HBox(false, single_space));
  format_box = manage(new Gtk::VBox(false, double_space));
    
  Gtk::VBox *vbox = manage(new Gtk::VBox(false, triple_space));
  vbox->set_border_width(border_width);
  set_modal(true);
  set_resizable(false);

  {
    std::string& default_paper = config.PaperName.values.front();
    bool default_is_known = false;
    std::list<std::string> paper_list;
    for(std::map<std::string, Papers::Size>::const_iterator i 
	  = papers.sizes.begin();
	i!=papers.sizes.end();
	i++) {
      paper_list.push_back(i->first);
      if(i->first == default_paper)
	default_is_known = true;
    }
    paper_size->set_popdown_strings(paper_list);
    if(!default_is_known)
      {
	warning << "Paper type \"" << default_paper 
		<< "\" is unknown." << std::endl;
	default_paper = "A4";
	config.PaperName.values.clear();
	config.PaperName.values.push_back(default_paper);
      }
    paper_size->get_entry()->set_text(default_paper);
  }

  {
    Gtk::RadioButton::Group group = landscape_button->get_group();
    portrait_button->set_group(group);
  }

  {
    Gtk::RadioButton::Group group = double_sided_button->get_group();
    single_sided_button->set_group(group);
  }

  template_box->pack_start(*template_button, Gtk::PACK_SHRINK);
  template_box->pack_start(*file_entry, Gtk::PACK_EXPAND_WIDGET);

  format_box3->pack_start(*paper_size, Gtk::PACK_SHRINK);
  format_box3->pack_start(*portrait_button, Gtk::PACK_SHRINK);
  format_box3->pack_start(*landscape_button, Gtk::PACK_SHRINK);

  sidedness_box->pack_start(*single_sided_button, Gtk::PACK_SHRINK);
  sidedness_box->pack_start(*double_sided_button, Gtk::PACK_SHRINK);

  Gtk::Label *format_label = 
    manage(new Gtk::Label("<b>Document _format</b>", 0.0, 0.5, true));
  format_label->set_use_markup();
  format_label->set_mnemonic_widget(*paper_size->get_entry());
  format_box->pack_start(*format_label);
  Gtk::HBox *format_box2 = manage(new Gtk::HBox());
  format_box2->pack_start(*manage(new Gtk::Label("  ")));
  format_box2->pack_start(*format_box3);
  format_box2->pack_start(*manage(new Gtk::VSeparator()), Gtk::PACK_SHRINK,
			  double_space);
  format_box2->pack_start(*sidedness_box);
  format_box->pack_start(*format_box2);

  first_page_box->pack_end(*first_page, Gtk::PACK_SHRINK);
  Gtk::Label *first_page_label = 
    manage(new Gtk::Label("First page _number:", true));
  first_page_label->set_mnemonic_widget(first_page->get_spinbutton());
  first_page_box->pack_end(*first_page_label, 
			   Gtk::PACK_SHRINK);

  vbox->pack_start(*template_box);
  vbox->pack_start(*manage(new Gtk::HSeparator()));
  vbox->pack_start(*format_box);
  vbox->pack_start(*manage(new Gtk::HSeparator()));
  vbox->pack_start(*first_page_box);

  get_vbox()->pack_start(*vbox);

  template_button->set_active(true);
  portrait_button->set_active(!config.Landscape.values.front());
  double_sided_button->set_active(true);

  paper_size->get_entry()->set_editable(false);

  add_button(Gtk::Stock::CANCEL, 0);
  ok_button = add_button(Gtk::Stock::OK, 1);
  create_button = add_button(Gtk::Stock::NEW, 1);

  template_button->signal_clicked().connect
    (safeslot(*this, &Doc_Props_Dialog::update_mode));
  update_mode();

  get_vbox()->show_all();
  get_action_area()->show_all();
}

void Doc_Props_Dialog::update_mode() {
  bool active = template_button->get_active();
  format_box->set_sensitive(!active);
  file_entry->set_sensitive(active);
}

void Doc_Props_Dialog::show_it(Document_View *_view, bool create_new) {
  new_document = create_new;
  view = _view;
  if(!new_document) {
    Document *document = view->get_document();
    if(!document)
      return;
    std::string template_file = document->get_template_file();
    template_button->set_active(!template_file.empty());
    first_page->set(document->get_first_page_num());
    portrait_button->set_active(document->get_orientation()
				== Papers::PORTRAIT);
    double_sided_button->set_active(document->is_doublesided());
    paper_size->get_entry()->set_text(document->get_paper_name());

    create_button->hide(); ok_button->show();
    show();
    // Gtk::Entry::set_position doesn't seem to work 
    // unless the entry is shown first
    file_entry->entry.set_text(template_file);
  } else {
    create_button->show(); ok_button->hide();
    template_button->set_active(true);
    show();
  }

  // have to show() first
  set_title(new_document
	    ? "Create new document"
 	    : "Document properties");
}


void Doc_Props_Dialog::on_response(int response_id) {
  if(response_id == 0) {
    hide();
  } else if(response_id == 1) {
    Papers::Orientation orientation =
      portrait_button->get_active()
      ? Papers::PORTRAIT 
      : Papers::LANDSCAPE; 

    if(template_button->get_active() && file_entry->entry.get_text().empty()) {
      Error_Dialog::view("No template file was selected");
      return;
    }
    if(new_document) { // Create new document
      Document *document = 0;
      try {
	if(template_button->get_active()) { // Create from template
	  document = new Document(file_entry->entry.get_text());
	  document->set_first_page_num(int(first_page->get()));
	} else {
	  document = new Document();
	  document->set_paper_name(paper_size->get_entry()->get_text());
	  document->set_orientation(orientation);
	  document->set_doublesided(double_sided_button->get_active());
	  document->set_first_page_num(int(first_page->get()));
	}
	if(view && !view->get_document())
	  view->set_document(document);
	else
	  new Frame_Window("", document);
	hide();
      }
      catch(const Error::Read &e) {
	delete document;
	Error_Dialog::view(e.message);
      }
    } else { // Modify existing document
      try {
	Document *document = view->get_document();
	if(!document)
	  return;
	if(template_button->get_active()) { // create from template
	  document->set_template(file_entry->entry.get_text());
	  document->set_first_page_num(int(first_page->get()));
	} else {
	  document->set_template("");
	  document->set_paper_name(paper_size->get_entry()->get_text());
	  document->set_orientation(orientation);
	  document->set_doublesided(double_sided_button->get_active());
	  document->set_first_page_num(int(first_page->get()));
	}
	hide();
      }
      catch(const Error::Read &e) {
	Error_Dialog::view(e.message);
      }
    }
  }
}
