///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <sstream>
#include <algorithm>
#include <stdexcept>
#include <ctime>
#include <fstream>

#include <libxml++/libxml++.h>

#include "document.h"
#include "fileerrors.h"
#include "page.h"
#include "globals.h"
#include "textstream.h"
#include <xml2ps/psstream.hh>
#include <util/stringutil.h>
#include <util/filesys.h>
#include <util/warning.h>
#include <util/os.h>
#include <fonts/fontmanager.hh>
#include <ps/pfb2pfa.h>
#include <defines.h>

SigC::Signal1<void, Document*> Document::changed_signal;
SigC::Signal1<void, Document*> Document::size_changed_signal;
SigC::Signal1<void, Document*> Document::filename_changed_signal;
SigC::Signal1<void, Document*> Document::selection_changed_signal;
SigC::Signal1<void, Document*> Document::streams_changed_signal;
SigC::Signal1<void, Document*> Document::deleted_signal;
SigC::Signal1<void, Document*> Document::undo_changed_signal;

namespace {			// local
  class same_name {
  public:
    same_name(const std::string& name) : name_(name) {};
    bool operator() (const Text_Stream* stream) {
      return stream->get_name() == name_;
    }
  private:
    const std::string name_;
  };
  class NameOrder {
  public:
    bool operator() (const Text_Stream* x, const Text_Stream* y) const {
      return x && y && (x->get_name() < y->get_name());
    }
  };
}

Document::Document()
  :is_template(false), paper_name("A4"), filename(""),
   orientation(Papers::PORTRAIT), 
   the_template(0)
{
  pages.push_back(new Page(*this));
  set_doublesided(true);
  set_first_page_num(1);
}

Document::Document(std::string filename_, bool is_template_)
  :is_template(is_template_), the_template(0)
{
  filename = expand_path(filename_);
  open();
}

Document::Document(std::string template_file_)
  :is_template(false), filename(""), the_template(0)
{
  set_first_page_num(1);
  set_template(template_file_);
}

Document::~Document() {
  deleted_signal(this);

  // delete template
  if(the_template)
    delete the_template;

  // delete pages
  while(!pages.empty()) {
    Page *page = *pages.begin();
    pages.pop_front();
    delete page;
  }

  // delete streams
  while(!text_streams.empty()) {
    Text_Stream *text_stream = *text_streams.begin();
    text_streams.pop_front();
    delete text_stream;
  }
}

std::string Document::make_up_new_name() {
  static unsigned int num = 1;
  while(num < UINT_MAX) {
    std::ostringstream tmp;
    tmp << "stream" << num;
    if(find_if(text_streams.begin(), text_streams.end(), same_name(tmp.str()))
       == text_streams.end())
      return tmp.str();
    // up the number only afterwards if no match, so if the user creates a
    // stream, renames it, and creates another one, he don't loose numers.
    ++num;
  }
  // Todo:  Try again from 1, in case a stream was deleted or renamed, but
  // only once, so we don't get stuck in an eternal loop.  Don't start from 1
  // each time, to avoid confusing the user
  throw std::runtime_error("Out of automatical stream names");
}

int Document::count_selected() const {
  return selection.size();
}

const Document::Selection& Document::selected() const {
  return selection;
}

void Document::select_all(bool select) {
  selection.clear();

  if(select)
    for(PageVec::const_iterator p = pages.begin(); p != pages.end(); ++p)
      for(Group::ChildVec::const_iterator i = (*p)->pbegin(); 
          i != (*p)->pend(); ++i)
        selection.push_back(*i);
  
  selection_changed_signal(this);
}

void Document::select_all_on_page(Page *page, bool select) {
  if(!page)
    return;

  selection.clear();

  if(select)
    for(Group::ChildVec::const_iterator i = page->pbegin(); 
	i != page->pend(); ++i)
      selection.push_back(*i);
  
  selection_changed_signal(this);
}

void Document::deselect(Pagent* obj) {
  Selection::iterator i = find(selection.begin(), selection.end(), obj);
  if(i != selection.end()) {
    selection.erase(i);
    selection_changed_signal(this);
  }
}

void Document::select(Pagent* obj, bool deselect_old) {
  if(deselect_old)
    selection.clear();
  selection.push_back(obj);
  selection_changed_signal(this);
}

void Document::delete_selected() {
  for(Selection::const_iterator i = selection.begin(); 
      i != selection.end(); ++i) 
    {
      Pagent* obj = (*i)->get_parent().ungroup(*i);
      if (!obj)
	throw std::logic_error("Tried to remove pagent from bad parent.");
      delete obj;
    }
  selection.clear();

  selection_changed_signal(this);
}

Document::StreamVec Document::get_text_streams() {
  StreamVec tmp = text_streams;
  tmp.sort(NameOrder());
  return tmp;
}

void Document::rename_text_stream(std::string old_name, std::string new_name) {
  Text_Stream *stream = get_text_stream(old_name);
  if(!stream)
    throw Error::Text_Stream_Name();
  if(old_name == new_name)
    return; //not an error
  if(get_text_stream(new_name))
    throw Error::Text_Stream_Name();
  stream->set_name(new_name);
  streams_changed_signal(this);
}

void Document::add_text_stream(Text_Stream* new_stream) {
  for(StreamVec::const_iterator i = text_streams.begin();
      i != text_streams.end(); i++)
    if((*i)->get_name() == new_stream->get_name())
      throw Error::Text_Stream_Name();

  text_streams.push_back(new_stream);
  streams_changed_signal(this);
}

std::string Document::add_text_stream(std::string file, 
				      std::string name, 
				      std::string transform)
{
  if(name == "") 
    name = make_up_new_name();
  for(StreamVec::const_iterator i = text_streams.begin();
      i != text_streams.end(); i++)
    if((*i)->get_name() == name)
      throw Error::Text_Stream_Name();
  text_streams.push_back(new Text_Stream(name, file, transform));
  streams_changed_signal(this);
  return name;
}

Text_Stream* Document::get_text_stream(std::string name) {
  for(StreamVec::iterator i = text_streams.begin();
      i != text_streams.end();
      i++)
    if((*i)->get_name() == name)
      return *i;
  return 0;
}

void Document::remove_text_stream(std::string name) {
  for(StreamVec::iterator i = text_streams.begin();
      i != text_streams.end();
      i++)
    if((*i)->get_name() == name) {
      delete *i;
      text_streams.erase(i);
      streams_changed_signal(this);  
      return;
    }
}

std::string Document::to_relative(std::string filename) const {
  return relative_path(path(this->filename), filename);
}

std::string Document::from_relative(std::string filename) const {
  if(filename.empty() || filename[0] == '/')
    return filename;
  else
    return path(this->filename)+filename;
}

unsigned int Document::get_num_of_pages() const {
  return pages.size();
}

void Document::delete_page(int page_num) {
  int num_of_pages = get_num_of_pages();

  if(page_num >= first_page_num 
     && page_num < first_page_num + num_of_pages) {
    int j = first_page_num;
    PageVec::iterator i = pages.begin();
    
    while(j < page_num && j < first_page_num + num_of_pages) {
      i++;
      j++;
    }
    delete *i;
    pages.erase(i);
  }
  else throw Error::Invalid_Page_Num(page_num);
  
  // Todo: temporary fix to make sure deleted objects 
  //are not still selected:
  select_all(false); 

  changed_signal(this);
}

Page *Document::new_page(int page_num, Page *original) {
  PageVec::iterator i = pages.begin();
  int num_of_pages = get_num_of_pages();
  Page *the_new_page;

  if(original) {
    xmlpp::Document tmpdoc;
    xmlpp::Element *root = tmpdoc.create_root_node("template");
    xmlpp::Element *tmp = original->save(*root);
    the_new_page = new Page(*tmp, *this);
    the_new_page->set_name("");  // the name should not be inherited
    // Todo: Check that Document desctructor destructs entire tree so we
    // don't leak memory here!
  } else {
    the_new_page = new Page(*this);
  }

  if(num_of_pages == 0) {
    pages.push_front(the_new_page);
  } else if(page_num >= first_page_num 
	    && page_num <= first_page_num + num_of_pages) {
    int j = first_page_num;

    while(j < page_num && j < first_page_num + num_of_pages) {
      i++;
      j++;
    }
      
    pages.insert(i, the_new_page);
  } else {
    delete the_new_page;
    the_new_page = 0;
    throw Error::Invalid_Page_Num(page_num);
  }

  changed_signal(this);
  return the_new_page;
}

int Document::get_page_num_of_page(const Page *page) const {
  int j = get_first_page_num();
  for(PageVec::const_iterator i = pages.begin(); i != pages.end(); i++, j++) {
    if(page == *i)
      return j;
  }
  throw Error::Invalid_Page_Num();
}

Page *Document::get_page(int page_num) {
  int j = first_page_num;
  PageVec::iterator i = pages.begin();
  int num_of_pages = get_num_of_pages();

  if(num_of_pages == 0)
    return 0;

  while(j < page_num && j < first_page_num + num_of_pages - 1) {
    i++;
    j++;
  }
  
  return *i;
}

Page *Document::get_page(std::string page_name) {
  for(PageVec::iterator i = pages.begin();
      i != pages.end(); i++)
    if((*i)->get_name() == page_name)
      return *i;

  return 0;
}

std::list<std::string> Document::get_template_pages() {
  std::list<std::string> tmp;
  if(the_template) {
    Document *t = the_template;
    for(int i = t->get_first_page_num(); 
	i < t->get_first_page_num() + int(t->get_num_of_pages());
	i++)
      tmp.push_back(t->get_page(i)->get_name());
  }
  return tmp;
}

void Document::open() {
  try {
    xmlpp::DomParser tree(filename);
    // Todo:  Check if get_document is guaranteed to not return 0.
    xml_open(tree.get_document()->get_root_node());
  } catch(xmlpp::exception e) {
    throw Error::Read("Failed to open " + filename +
		      "\n" + e.what());
  }
}

void Document::save() {
  xmlpp::Document *tree = xml_save();
  try {
    tree->write_to_file(filename);
  } catch (xmlpp::exception e) {
    delete tree;
    throw Error::Write("Could not write to " + filename + 
		       "\n" + e.what());
  }
  delete tree;
}

void Document::xml_open(xmlpp::Element *root) {
  // TODO: this function is too long and unreadable
  std::string temp_template = "";

  if(!root)
    throw Error::Read("Root node is NULL.");
  if(root->get_name() != "document")
    throw Error::Read("Root node is not <document>");

  //default values if attribute is not encountered:
  set_doublesided(true); 
  set_orientation(Papers::PORTRAIT);
  set_first_page_num(1);
  set_paper_name("A4");

  //read document attributes:
  xmlpp::Element::AttributeList attributes = root->get_attributes();
  for(xmlpp::Element::AttributeList::iterator i = attributes.begin();
      i != attributes.end();
      i++)
    {
      const std::string name = (*i)->get_name();
      if(name == "template")
	temp_template = (*i)->get_value();
      else if(name == "doublesided")
	set_doublesided(to<bool>((*i)->get_value()));
      else if(name == "landscape") {
	if(to<bool>((*i)->get_value()))
	  set_orientation(Papers::LANDSCAPE);
      } else if(name == "paper_name") {
	try {
	  set_paper_name((*i)->get_value());
	} catch (Error::Paper_Name e) {
	  throw Error::Read("There is no paper called \""+e.name+"\".");
	}
      } else if(name == "first_page_num") {
	set_first_page_num(to<int>((*i)->get_value()));
      } else
	warning << "Unknown attribute \"" << (*i)->get_name()
		<< "\" ignored in <document>." << std::endl;
    }

  if(!is_template) // templates can't have templates
    set_template(from_relative(temp_template)); 
  // the template overrides anything  explicitly stated in the document

  // read text streams and pages:
  xmlpp::Element::NodeList children = root->get_children();
  for(xmlpp::Node::NodeList::iterator i = children.begin();
      i != children.end();
      i++) {
    if(xmlpp::Element *elem = dynamic_cast<xmlpp::Element*>(*i)) {
      std::string name = elem->get_name();
      if(name == "text_stream") {
	// if no name we can't add it because a name would be assigned for it
	std::string name, file, transform;
	if(const xmlpp::Attribute* n_prop = elem->get_attribute("name"))
	  name = n_prop->get_value();
	if(const xmlpp::Attribute* f_prop = elem->get_attribute("file"))
	  file = f_prop->get_value();
	if(const xmlpp::Attribute* x_prop = elem->get_attribute("transform"))
	  transform = x_prop->get_value();
	  
	if(name.empty()) //permit streams without an association
	  throw Error::Read("Could not load text stream, name \""
			    + name + "\", file \"" + file + "\"");
	  
	// template streams override document streams
	// if two streams have the same name, the second will be ignored
	if(!get_text_stream(name)) {
	  Text_Stream *stream = new Text_Stream(name, from_relative(file), 
						from_relative(transform));
	  xmlpp::Element::NodeList children = elem->get_children();
	  for(xmlpp::Element::NodeList::iterator i = children.begin();
	      i != children.end();
	      ++i) {
	    if(xmlpp::Element *elem = dynamic_cast<xmlpp::Element*>(*i))
	      if(elem->get_name() == "parameter") {
		stream->set_parameter
		  (elem->get_attribute("name")->get_value(),
		   elem->get_attribute("value")->get_value());
	      }
	  }
	  add_text_stream(stream);
	}
      } else if(name == "page") {
	Page *page = new Page(*elem, *this);
	if(!page)
	  throw Error::Read("Could not add page");
	pages.push_back(page);
      } else
	warning << "Unknown node <" << elem->get_name()
		<< "> ignored in <document>" << std::endl;
    }
  }
}

void Document::set_filename(std::string filename_) {
  filename = expand_path(path(filename_)) + "/" + basename(filename_);
  debug << filename << std::endl;
  filename_changed_signal(this);
}

void Document::set_template(std::string template_file_) {
  template_file = template_file_;
  if(!template_file.empty()) {
    if(the_template)
      delete the_template;
    the_template = new Document(template_file, true);
    set_doublesided(the_template->is_doublesided());
    set_orientation(the_template->get_orientation());
    set_paper_name(the_template->get_paper_name());
    StreamVec ts = the_template->get_text_streams();
    for(StreamVec::iterator i = ts.begin();
	i != ts.end(); i++)
      {
	try {
	  Text_Stream* tmp = get_text_stream((*i)->get_name());
	  if(!tmp)
	    add_text_stream((*i)->get_association(), (*i)->get_name(),
			    (*i)->get_transform());
	  else // override
	    tmp->set_association((*i)->get_association());
	} catch (Error::Text_Stream_Name e) {
	  warning << e.what() << std::endl;
	}
      }
  }
}

void Document::set_doublesided(bool ds) {
  doublesided = ds;
  changed_signal(this);
}

void Document::set_first_page_num(int num) {
  first_page_num = num;
  changed_signal(this);
}

void Document::set_orientation(Papers::Orientation _orientation) {
  orientation = _orientation;
  changed_signal(this);
  size_changed_signal(this);
}

void Document::set_paper_name(std::string _paper_name) {
  if(papers.sizes.find(_paper_name) == papers.sizes.end())
    throw Error::Paper_Name(_paper_name);
  paper_name = _paper_name;
  changed_signal(this);
  size_changed_signal(this);
}

xmlpp::Document *Document::xml_save() {
  xmlpp::Document *tree=new xmlpp::Document();
  xmlpp::Element *root = tree->create_root_node("document");
  root->add_attribute("paper_name", get_paper_name());

  if(the_template)
    root->add_attribute("template", to_relative(template_file));
  root->add_attribute("doublesided", tostr<bool>(is_doublesided()));
  
  root->add_attribute("landscape", 
		      tostr<bool>(get_orientation() == Papers::LANDSCAPE));
  
  root->add_attribute("first_page_num", tostr(get_first_page_num()));
  root->add_content("\n");
  for(StreamVec::iterator i = text_streams.begin(); 
      i != text_streams.end(); i++)
    {
      root->add_content("  ");
      xmlpp::Element *tmp = root->add_child("text_stream");
      root->add_content("\n");
      tmp->add_attribute("name", (*i)->get_name());
      tmp->add_attribute("file", to_relative((*i)->get_association()));
      tmp->add_attribute("transform", to_relative((*i)->get_transform()));
      
      for(Text_Stream::ParamIter j = (*i)->param_begin();
	  j != (*i)->param_end(); ++j) {
	tmp->add_content("\n    ");
	xmlpp::Element *par = tmp->add_child("parameter");
	par->add_attribute("name", j->first);
	par->add_attribute("value", j->second);
      }
    }
  for(PageVec::iterator i = pages.begin(); i != pages.end(); i++) {
    root->add_content("  ");
    (*i)->save(*root);
    root->add_content("\n");
  }
  return tree;
}

void Document::print(std::ostream& out, int first_page, int last_page,
		     bool eps, bool include_fonts, bool grayscale) const
{
  const font::FontManager *fm = font::FontInfo::getFontManager();
  if(!fm)
    throw Error::Print("FontManager not found");

  // merge required fonts from all streams
  font::Fonts used_fonts;
  for(StreamVec::const_iterator j = text_streams.begin();
      j != text_streams.end();
      j++) {
    const font::Fonts &fonts = (*j)->get_used_fonts();
    for(font::Fonts::const_iterator i = fonts.begin();
	i != fonts.end();
	i++) {
      used_fonts.insert(*i);
    }
  }

  using std::endl;
  
  time_t the_time = std::time(0);

  if(!(first_page >= first_page_num 
       && first_page <= last_page
       && last_page <= first_page_num + int(get_num_of_pages()) - 1))
    throw Error::Print("Bad page interval");

  // ignore request to print multible-page EPS 
  eps = eps && first_page == last_page;

  int w = int(get_width() + 0.5);
  int h = int(get_height() + 0.5);
  
  if(eps) {
    out << "%!PS-Adobe-3.0 EPSF-3.0\n"
	<< "%%BoundingBox: 0 0 " << w << " " << h << '\n';
  } else {
    out << "%!PS-Adobe-3.0\n";
  }

  
    //out << "%%DocumentData: Clean8Bit" << endl
    // <<"%%LanguageLevel: 2"<<endl
    // actually, we don't really know much the eps files ...
  out << "%%Orientation: "
      << (orientation == Papers::PORTRAIT ? "Portrait\n" : "Landscape\n")
      << "%%Pages: " << get_num_of_pages() << '\n'
      << "%%PageOrder: Ascend\n"
      << "%%Title: " << basename(filename) << '\n'
      << "%%CreationDate: " << std::ctime(&the_time)
    // ctime seems to add a newline
      << "%%Creator: Passepartout " << version
      << " by Fredrik Arnerup & Rasmus Kaj\n";

  if(true) // perhaps the user is into cloak and dagger stuff ...
    out << "%%For: " << os::fullname()
	<< " <" << os::username() << "@" << os::hostname() << ">"
	<< " (" << os::machine() << ", " << os::sysname()
	<< " " << os::release() << ")\n";

  out << "%%EndComments\n\n"
      << "%%BeginProlog\n";
  xml2ps::PsStream::psProlog(out);
  out << "%%EndProlog\n\n"
      << "%%BeginSetup" << endl;
  
  // Resource comments
  int line = 0;
  for(font::Fonts::const_iterator i = used_fonts.begin();
      i != used_fonts.end();
      i++) {
    out << (line++ ? "%%+ "
	    : (include_fonts 
	       ? "%%DocumentSuppliedResources: " 
	       : "%%DocumentNeededResources: "))
	<< "font " << fm->unalias(*i) << std::endl;
  }

  // %%IncludeResource comments
  if(!include_fonts) {
    for(font::Fonts::const_iterator i = used_fonts.begin();
	i != used_fonts.end();
	i++) {
      out << "%%IncludeResource: font " << fm->unalias(*i) << std::endl;
    }
  } else { // include fonts
    for(font::Fonts::const_iterator i = used_fonts.begin();
	i != used_fonts.end();
	i++) {
      std::string fontfile = fm->getFontFile(*i);
      if(fontfile.empty()) {
	warning << "Couldn't find font file for " << *i << std::endl;
	continue;
      }
      std::ifstream in(fontfile.c_str());
      if(!in) {
	warning << "Couldn't read " << fontfile << std::endl;
	continue;
      }
      out << "%%BeginResource: font " << fm->unalias(*i) << std::endl;
      if(suffix(fontfile) == "pfa") { // ascii
	std::string tmpline;
	while(getline(in, tmpline)) {
	  out << tmpline << std::endl;
	}
      } else { // assume it is pfb
	try {
	  PS::pfb2pfa(in, out);
	} catch(std::runtime_error e) {
	  warning << "error in " << fontfile 
		  << " : " << e.what() << std::endl;
	}
      }
      out << "%%EndResource" << std::endl;
    }
  }

  out << "%%EndSetup" << endl
      << endl;
  
  int page_num = first_page_num;
  for(PageVec::const_iterator i = pages.begin(); i != pages.end(); i++) {
    if(page_num >= first_page && page_num <= last_page) {
      out << endl << "%%Page: \""
	  <<(*i)->get_name()<<"\" "<<page_num<<endl;

      // If we have a bounding box, we may not draw outside of it.
      // Shouldn't really be a problem, but one never knows ...
      if(eps)
	out << "\ngsave\n"
	    << "0 0 moveto "            << w << " 0 rlineto\n"
	    << "0 " << h <<" rlineto "  << -w << " 0 rlineto\n"
	    << "closepath clip newpath\n\n";
      //clip doesn't make implicit newpath

      (*i)->print(out, grayscale);

      if(eps)
	out << "\ngrestore\n";
    }
    page_num++;
  }
  out << "%%EOF" << endl;
}

Document& Document::containing(Pagent& obj) {
  try {
    Page& page = Page::containing(obj);
    return page.document;
    
  } catch(const Error::No_Parent& err) {
    throw std::logic_error
      ("Tried to get Document containing pagent that was not in a Document.");
  }
}

const Document& Document::containing(const Pagent& obj) {
  try {
    const Page& page = Page::containing(obj);
    return page.document;
    
  } catch(const Error::No_Parent& err) {
    throw std::logic_error
      ("Tried to get Document containing pagent that was not in a Document.");
  }
}
