#ifndef DOCUMENT_H		// -*- c++ -*-
#define DOCUMENT_H
///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "page.h"
#include "paper_sizes.h"
#include "util/undo.h"
#include <sigc++/object.h>
#include <list>
#include <string>
#include <stdexcept>

class Pagent;
class Text_Stream;
namespace xmlpp { class Node; }
class Page;

namespace Error {
  struct Empty_Document : public std::logic_error {
    Empty_Document() : logic_error("Empty document") {}
  };
  struct Text_Stream_Name : public std::logic_error {
    Text_Stream_Name() : logic_error("Text stream name clash") {}
  };
}

class Document: public undo::Undoable {
public:
  typedef std::list<Pagent*> Selection;
  typedef std::list<Text_Stream*> StreamVec;

  static SigC::Signal1<void, Document*> changed_signal;
  static SigC::Signal1<void, Document*> size_changed_signal;
  static SigC::Signal1<void, Document*> filename_changed_signal;
  static SigC::Signal1<void, Document*> selection_changed_signal;
  static SigC::Signal1<void, Document*> streams_changed_signal;
  static SigC::Signal1<void, Document*> deleted_signal;
  static SigC::Signal1<void, Document*> undo_changed_signal;

  Document();
  Document(std::string filename_, bool is_template_); // throws Error::Open
  Document(std::string template_file_); // throws Error::Open
  ~Document();
  void open(); // throw Error::Open
  void save();

  void print(std::ostream& out, int first_page, 
	     int last_page, 
	     bool eps = false, bool include_fonts = false, 
	     bool grayscale = false) const;// throw Error::Print

  float get_width() const
  {return papers.sizes[get_paper_name()].get_width(orientation);}
  float get_height() const
  {return papers.sizes[get_paper_name()].get_height(orientation);}

  void set_filename(std::string filename_);
  const std::string &get_filename() const {return filename;}

  void set_template(std::string template_file_);
  std::string get_template_file()
  {return template_file;}
  Document *get_template()
  {return the_template;}
  std::list<std::string> get_template_pages(); // throws Error::Read
  void set_paper_name(std::string _paper_name);
  std::string get_paper_name() const
  {return paper_name;}
  void set_orientation(Papers::Orientation _orientation);
  Papers::Orientation get_orientation() const
  {return orientation;}
  void set_first_page_num(int num);
  int get_first_page_num() const
  {return first_page_num;}
  void set_doublesided(bool ds);
  bool is_doublesided() const
  {return doublesided;}

  unsigned int get_num_of_pages() const;
  // a page does not know its own number
  int get_page_num_of_page(const Page *page) const;
  // throws Error::Invalid_Page_Num
  Page *get_page(int page_num);
  Page *get_page(std::string page_name);
  // NULL, if page does not exist
  void delete_page(int page_num); 
  // throws Error::Invalid_Page_Num
  // emits document_changed_signal
  Page *new_page(int page_num, Page *original=0);
  // inserts page at requested page number, possibly a copy of original
  // if existing pages are e.g. 4, 5, 6 then valid numbers are 4, 5, 6, 7
  // if no pages, then first_page_num is set to page_num
  // throws Error::Invalid_Page_Num
  // emits document_changed_signal

  int count_selected() const;
  const Selection& selected() const;
  void select_all(bool select);	// false for select none
  void select_all_on_page(Page *page, bool select); // false for select none
  void select(Pagent* obj, bool deselect_old = true);
  void deselect(Pagent* obj);
  void delete_selected();

  StreamVec get_text_streams(); //sorted list
  std::string make_up_new_name();

  /**
   * Add a stream, while checkin that it has a unique name (otherwise an
   * exception is thrown.  If sucessfull, a streams_changed_signal is raised.
   */
  void add_text_stream(Text_Stream* new_stream);
  // old method, deprecated ...
  std::string add_text_stream(std::string file, std::string name,
			      std::string transform);
  //throws Error::Text_Stream_Name
  void rename_text_stream(std::string old_name, std::string new_name);
  //throws Error::Text_Stream_Name
  Text_Stream* get_text_stream(std::string name);
  void remove_text_stream(std::string name);

  // convert to and from path relative to document path:
  std::string to_relative(std::string filename) const;
  std::string from_relative(std::string filename) const;
  
  /** Find the document containing a specific Pagent. */
  static Document& containing(Pagent& obj);
  /** Find the document containing a specific Pagent. */
  static const Document& containing(const Pagent& obj);
  
private:
  typedef std::list<Page*> PageVec;
  int first_page_num;
  PageVec pages;
  StreamVec text_streams;
  bool doublesided, is_template;
  std::string paper_name, template_file, filename;
  Papers::Orientation orientation;
  Document *the_template;
  Selection selection;

  void xml_open(xmlpp::Element *root); // throw Error::Open
  xmlpp::Document *xml_save(); 

  // overrides undo::Undoable::on_stack_change
  void on_stack_change() {undo_changed_signal(this);}
};

#endif



