#ifndef DOCVIEW_H		// -*- c++ -*-
#define DOCVIEW_H
///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "view.h"
#include "group.h"
#include <gdkmm.h>
#include <gtkmm/drawingarea.h>

class Pagent;
class Template_Page_Dialog;
class Page;
class Document;

namespace snap {
  enum Mode {GRID, GUIDE, NONE};
}

class Document_View: public Gtk::DrawingArea, public View {
public:
  SigC::Signal0<void> current_page_num_changed_signal;
  SigC::Signal0<void> document_changed_signal;
  SigC::Signal0<void> document_set_signal;
  SigC::Signal1<void, float> zoom_change_signal;

  Gdk::Color white, black, gray, red;

  Document_View(Document*);
  ~Document_View();
  void set_document(Document *d);
  Document *get_document() const {return document;}
  void set_current_page_num(int);
  int get_current_page_num() const {return current_page_num;}
  void draw(bool reshaping = false);
  Page* get_page();
  float get_zoom_factor() const {return zoom_factor;}
  void set_zoom_factor(float);

  void insert_page_before();
  void insert_page_after();
  void delete_page();

  void undo();
  void redo();

  void copy();
  void cut();
  void paste();

  void delete_selected();
  void rearrange_selected(Rearrange_Target target);
  void group_selected();
  void ungroup_selected();
  void new_text_frame();
  void new_image_frame(std::string, float res = -1, bool catch_errors = true);
  void select_all_frames();
  void unselect_all_frames();
  Glib::RefPtr<Gdk::Drawable> get_win()
  {return win;}
  Glib::RefPtr<Gdk::GC> get_gc()
  {return gc;}
  const Gdk::Color& get_color(Color::Id color) const;

  void set_snap(snap::Mode mode) {snap_mode = mode;}
  snap::Mode get_snap() const {return snap_mode;}
		
  IVector get_origin() const;
  double get_scrres() const {return resolution * zoom_factor;}
  float pt2scr(float pt) const;
  float scr2pt(float scr) const;
  // Important! the scalar versions of pt2scr and scr2pt only perform
  // length conversions while the vector versions also convert between
  // right- and left-handed coordinate systems
  Gdk::Point pt2scr(const Vector& pt) const;
  Vector scr2pt(const Gdk::Point& scr) const;
  int page_height() const;
  int page_width() const;

  Glib::RefPtr<const Gdk::Pixmap> get_wait_pixmap() const 
  {return wait_pixmap;}
  Glib::RefPtr<const Gdk::Pixmap> get_missing_pixmap() const 
  {return missing_pixmap;}
  Glib::RefPtr<const Gdk::Pixmap> get_broken_pixmap() const 
  {return broken_pixmap;}

  bool on_key_press_event(GdkEventKey*); // called from Window

private:
  // Undefined ctors, avoid defaults
  Document_View(const Document_View&);
  Document_View();
  void operator = (const Document_View&);

  Glib::RefPtr<Gdk::Window> win;
  Glib::RefPtr<Gdk::GC> gc;

  snap::Mode snap_mode;
  int margin_size;
  float resolution; //ppi
  float zoom_factor;
  float old_width, old_height;
  Matrix old_matrix;
  IVector offset;
  int reshape_box, current_page_num;
  bool moving, reshaping;
  Document *document;
  Page *old_page; //last page visited
  Template_Page_Dialog *template_page_dialog;
  Glib::RefPtr<Gdk::Pixmap> wait_pixmap, missing_pixmap, broken_pixmap;

  void on_clipboard_get(Gtk::SelectionData& selection_data, guint info);
  void on_clipboard_clear();
  void on_clipboard_received(const Gtk::SelectionData& selection_data);

  void on_realize();
  bool on_expose_event(GdkEventExpose*);
  bool on_button_press_event(GdkEventButton*);
  bool on_button_release_event(GdkEventButton*);
  bool on_motion_notify_event(GdkEventMotion*);
  void on_drag_data_received(const Glib::RefPtr<Gdk::DragContext>& context, 
			     int x, int y, GtkSelectionData* selection_data, 
			     guint info, guint time);

  void refresh_streams();
  void begin_move(int x, int y);
  void end_move(bool revert);
  void begin_reshape(int x, int y, int box);
  void end_reshape(bool revert);
  void move_reshape_box(int x, int y);

  void act_on_document_change(Document *document_);
  Gdk::Cursor get_cursor(int x, int y);
  void update_cursor(int x, int y);
  bool in_move_area(int x, int y);
  Pagent* in_select_area(int x, int y);
  void adjust_size();
  void act_on_size_change(Document *document_);
  void maybe_page_ready_to_draw(Pagent *pagent);
};

#endif







