///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "group.h"
#include "docview.h"
#include "document.h"
#include "util/rectboundary.h"
#include <util/stringutil.h>
#include <util/warning.h>
#include <ps/misc.h>
#include <gdkmm.h>
#include <algorithm>

Group::Group(Group* parent, const Glib::ustring& name)
  : Pagent(parent, name)
{}

Group::Group(Group* parent, const Glib::ustring& name, const Matrix& xform)
  : Pagent(parent, name)
{
  set_matrix(xform);
}

Group::~Group() {
  // Note:  When a group is "ungrouped", the childs should not be removed.
  // Fix that by moving the childs from the group to its parent before
  // deleteing the group.
  for(ChildVec::iterator i = childs.begin(); i != childs.end(); i++)
    delete *i;
}

void Group::print(std::ostream& out, bool grayscale) const {
  out << "gsave  % group\n"
      << PS::Concat(get_matrix());
  for(ChildVec::const_reverse_iterator i = childs.rbegin(); 
      i != childs.rend(); 
      i++)
    (*i)->print(out, grayscale);
  out << "grestore % /group" << std::endl;
}

class SubView : public View {
public:
  SubView(View& p, const Matrix& m) : parent(p), matrix(m) {}
  
  Glib::RefPtr<Gdk::Drawable> get_win() { return parent.get_win(); }
  Glib::RefPtr<Gdk::GC> get_gc() { return parent.get_gc(); }
  
  gdouble get_scrres() const {
    const Matrix inv = matrix.inv();
    return dist(inv.transform(Vector(parent.get_scrres(), 0)),
		inv.transform(Vector(0,0)));
  }
  // transforming lengths only make sense if the axes are 
  // scaled by the same amount
  float pt2scr(float pt) const {
    return parent.pt2scr(dist(matrix.transform(Vector(pt, 0)),
			      matrix.transform(Vector(0,0))));
  }
  Gdk::Point pt2scr(const Vector& pt) const {
    return parent.pt2scr(matrix.transform(pt));
  }
  Vector scr2pt(const Gdk::Point& scr) const {
    return matrix.inv().transform(parent.scr2pt(scr));
  }
  float scr2pt(float scr) const { // untested code!
    return dist(scr2pt(Gdk::Point(int(scr + 0.5), 0)), 
		scr2pt(Gdk::Point(0, 0)));
  }
  Glib::RefPtr<const Gdk::Pixmap> get_wait_pixmap() const {
    return parent.get_wait_pixmap();
  }
  Glib::RefPtr<const Gdk::Pixmap> get_missing_pixmap() const {
    return parent.get_missing_pixmap();
  }
  Glib::RefPtr<const Gdk::Pixmap> get_broken_pixmap() const {
    return parent.get_broken_pixmap();
  }
  const Gdk::Color &get_color(Color::Id color) const { 
    return parent.get_color(color); 
  }
private:
  View& parent;
  const Matrix& matrix;
};

bool Group::draw_content(View& view, bool reshaping) {
  SubView v(view, get_matrix());
  // reverse order, this is correct at least for Page
  bool ok = true;
  for(ChildVec::reverse_iterator i = childs.rbegin(); 
      i != childs.rend(); 
      i++)
    ok = (*i)->draw_content(v, reshaping) && ok;
  return ok;
}

void Group::clear_cache() {
  for_each(childs.begin(), childs.end(),
	   std::mem_fun(&Pagent::clear_cache));
}

class Rectangle {
public:
  Rectangle(const Vector& v) : lo_x(v.x), lo_y(v.y), hi_x(v.x), hi_y(v.y) {}
  void grow(const Vector& v) {
    if(v.x < lo_x) lo_x = v.x;
    if(v.x > hi_x) hi_x = v.x;
    if(v.y < lo_y) lo_y = v.y;
    if(v.y > hi_y) hi_y = v.y;
  }
  Boundary getBox() const {
    return RectBoundary::create(Matrix::translation(Vector(-lo_x, -lo_y)),
				hi_x - lo_x, hi_y - lo_y);
  }
private:
  double lo_x, lo_y, hi_x, hi_y;
};

Boundary Group::get_box() const {
  // If the group has no members, return a zero-size box at the right place
  // Hmm.  It seems zero-sized boxes isn't supported ...
  if(childs.empty()) return RectBoundary::create(get_matrix(), 1, 1);
  
  ChildVec::const_iterator i = childs.begin();
  const Boundary box = (*i)->get_box();
  Rectangle rect(box->getCorner(Corner::LL));
  rect.grow(box->getCorner(Corner::LR));
  rect.grow(box->getCorner(Corner::UL));
  rect.grow(box->getCorner(Corner::UR));
  
  for(; i != childs.end(); i++) {
    const Boundary box = (*i)->get_box();
    rect.grow(box->getCorner(Corner::LL));
    rect.grow(box->getCorner(Corner::LR));
    rect.grow(box->getCorner(Corner::UL));
    rect.grow(box->getCorner(Corner::UR));
  }
  return rect.getBox() * get_matrix();
}

// Note: maybe it's best to let a pagent ask its parent what its 
// policy on flow around is.
void Group::set_flow_around(bool) {} // Todo
bool Group::get_flow_around() const { return false; } // Todo

BoundaryVect Group::obstacle_list() const {
  BoundaryVect result;
  
  for(ChildVec::const_iterator i=childs.begin();
      i!=childs.end();
      i++)
    {
      if(Boundary boundary = (*i)->get_obstacle_boundary())
	result.push_back(boundary);
    }
  return result;
}

Boundary Group::get_obstacle_boundary() const {
  return Boundary(); // Todo
}

void Group::act_on_zoom_factor_change() {
  for_each(childs.begin(), childs.end(),
	   std::mem_fun(&Pagent::act_on_zoom_factor_change));
}

xmlpp::Element* Group::save(xmlpp::Element& parent_node) const {
  xmlpp::Element *node = parent_node.add_child("frame");
  node->add_attribute("type", "group");
  node->add_attribute("transform", tostr(get_matrix()));

  save_childs(*node);
  return node;
}

void Group::save_childs(xmlpp::Element& node) const {
  for(ChildVec::const_reverse_iterator i = childs.rbegin(); 
      i != childs.rend(); 
      i++) {
    node.add_content("\n      ");
    (*i)->save(node);
  }
  node.add_content("\n    ");
}

void Group::group_selected() {
  //Todo: make this work by doing matrix operations
  const Document::Selection selection=Document::containing(*this).selected();

  if(selection.size() <= 1) //no one-item groups, please
    return;

  // Calculate translation:
  double minx = DBL_MAX, miny = DBL_MAX;
  for(Document::Selection::const_iterator i=selection.begin();
      i!=selection.end();
      i++)
    {
      const Matrix& m = (*i)->get_matrix();
      minx = std::min(minx, m.tr_x());
      miny = std::min(miny, m.tr_y());
    }
  
  Group *group = new Group(this, "group", Matrix::translation(minx, miny));
  
  // Use a copy or we will confuse ourselves while manipulating a container 
  // we are iterating over.
  ChildVec childs_copy = childs;

  // reparent selected
  for(ChildVec::reverse_iterator i = childs_copy.rbegin();
      i != childs_copy.rend();
      i++)
	{
	  if(find(selection.begin(), selection.end(), *i) != selection.end()) {
	    (*i)->set_matrix((*i)->get_matrix()*group->get_matrix().inv());
	    group->add(ungroup(*i));
	  }
	}
  if(group->count()) {
    add(group);
    Document::containing(*this).select(group);
  } else {
    warning << "Got a strange group. Deleting it again." << std::endl;
    delete group;
  }
}

void Group::ungroup_selected() {
  // Todo: preserve stacking order
  const Document::Selection &selection
    = Document::containing(*this).selected();
  
  for(Document::Selection::const_iterator i = selection.begin();
      i != selection.end();
      i++)
    {
      Group *group = dynamic_cast<Group*>(*i);
      if(group) {
	Document::containing(*this).deselect(*i);
	while(group->pbegin() != group->pend()) {
	  Pagent *pagent = group->ungroup(*(group->prbegin()));
	  if(pagent) {
	    pagent->set_matrix(pagent->get_matrix()*group->get_matrix());
	    add(pagent);
	    Document::containing(*this).select(pagent, false);
	  }
	}
	if(!ungroup(group))
	  warning << "Group not in this group" << std::endl;
	delete group;
      }
    }
}

void Group::add(Pagent* obj) {
  // the front is the top
  childs.push_front(obj);
  obj->set_parent(this);
}

Pagent* Group::ungroup(Pagent* obj) {
  ChildVec::iterator i = find(childs.begin(), childs.end(), obj);
  if(i == childs.end()) //won't let me compare i==pend()
    return 0;
  childs.erase(i);
  obj->set_parent(parent);
  return obj;
}

void Group::rearrange_selected(Rearrange_Target target) {
  const Document::Selection &selection
    = Document::containing(*this).selected();
  
  if(target == TOP || target == BOTTOM) {
    ChildVec sorted_selection;
    // remove selected items from list
    for(Document::Selection::const_iterator s = selection.begin(); 
	s!=selection.end(); 
	s++)
      {
	ChildVec::iterator i = find(childs.begin(), childs.end(), *s);
	if(i != childs.end()) {
	  sorted_selection.push_back(*i);
	  childs.erase(i);
	} else
	  warning << "Selected item not on this page." << std::endl;
      }
    // append items to front or back
    childs.insert(target == TOP ? childs.begin() : childs.end(), 
		  sorted_selection.begin(), 
		  sorted_selection.end());
  } else { // UP or DOWN
    // This is basically a single-step bubblesort,
    // except that each element may only be swapped once.
    for(ChildVec::iterator i = childs.begin(); i != childs.end(); i++) {
      ChildVec::iterator j = i;
      j++;
      if(j == childs.end())
	break;
      bool i_selected = (find(selection.begin(), selection.end(), *i) 
			 != selection.end());
      bool j_selected = (find(selection.begin(), selection.end(), *j) 
			 != selection.end());
      if(i_selected && j_selected)
	continue;
      if((i_selected && target == DOWN) || (j_selected && target == UP)) {
	Pagent *tmp = *j; *j = *i; *i = tmp; // swap
	i++; // no swap next time
      }
    }
  }
  geometry_changed_signal(this);
}

void Group::child_ready_to_draw(Pagent *pagent) {
  //  if(pagent!=this) // no infinite loops, please
  ready_to_draw_signal(this);
}

void Group::child_props_changed(Pagent *pagent) {
  //  if(pagent!=this) // no infinite loops, please
  props_changed_signal(this);
}

void Group::child_geometry_changed(Pagent *pagent) {
  //  if(pagent!=this) // no infinite loops, please
  geometry_changed_signal(this);
}
