///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "imageframe.h"
#include "fileerrors.h"
#include "processman.h"
#include "config.h"
#include "globals.h"
#include <sstream>
#include <fstream>
#include <libxml++/libxml++.h>
#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>
#include <util/os.h>
#include <ps/misc.h>
#include <ps/wineps.h>
#include "action.h"
#include "document.h"

// TODO: move the magic to the ps library
namespace {
  bool check_windows_magic(const char *magic) {
    unsigned long tmp = 0;
    for(int i = 0; i < 4; i++) {
      tmp <<= 8;  tmp |= static_cast<unsigned char>(magic[i]);
    }
    if(tmp == 0xC5D0D3C6)
      debug << "Blechh! A Windows EPS :-(" << std::endl;
    else
      debug << "Not a Windows EPS :-D" << std::endl;
    return tmp == 0xC5D0D3C6;  // magic number for "windows" eps
  }
}

bool Image_Frame::is_postscript(std::string filename) {
  std::ifstream in(filename.c_str());
  char magic[5];
  return in && in.get(magic, 5) 
    && (std::string(magic) == "%!PS" || check_windows_magic(magic));
}

Image_Frame::Image_Frame(Group *parent, 
			 const Glib::ustring& assoc, float w, float h) :
  Cached_Frame(parent, w, h, "Image " + basename(assoc)),
  association(assoc)
{
  parsed_file = process_manager.find_new_name();
}

Image_Frame::Image_Frame(Group *parent, const xmlpp::Element& node) :
  Cached_Frame(parent, node)
{
  if(!node.get_attribute("type") 
     || node.get_attribute("type")->get_value() != "image")
    throw Error::Frame_Type("Bad image frame type: "
			    + node.get_attribute("type")->get_value());

  if(const xmlpp::Attribute* file = node.get_attribute("file"))
    set_association(Document::containing(*this)
		    .from_relative(file->get_value()));
  else {
    set_association("");
    warning << "No \"file\" attribute found in image frame" << std::endl;
  }

  parsed_file = process_manager.find_new_name();
  read_size(); // try to read bounding box from file
}

Image_Frame::~Image_Frame() {
  if(!parsed_file.empty())
    try { unlink(parsed_file); }
    catch(const std::exception& err) {
      warning << "Failed to clean up: " << err.what() << std::endl;
    }
}

xmlpp::Element *Image_Frame::save(xmlpp::Element& parent_node) const {
  xmlpp::Element *node = Cached_Frame::save(parent_node);
  node->add_attribute("type", "image");
  node->add_attribute("file", Document::containing(*this)
		      .to_relative(association));
  return node;
}


namespace {
  Vector parse(PS::WinEPSFilter &in, std::ostream &out) {
    float x1, y1;		// width and height of actual file.
    float _width, _height;
    bool boundingbox_found = false;
    bool transf_performed = false;
    
    std::string tmpline;
    while(getline(in, tmpline)) {
      if(starts_with(tmpline, "%%BoundingBox:")) {
	std::istringstream tmp(tmpline.c_str());
	std::string word;
	float x2, y2;
	if(tmp >> word >> x1 >> y1 >> x2 >> y2) {
	  boundingbox_found = true;
	  _width = x2 - x1;  _height = y2 - y1;
	  out << "%%BoundingBox: 0 0 " << _width << ' ' << _height << '\n';
	}
      } else if(!starts_with(tmpline, '%') && !transf_performed) {
	out << tmpline << "\n\n";
	transf_performed = true;
	if(!boundingbox_found)
	  throw Basic_Frame::Gen_Pic_Error(Basic_Frame::GENERATION, 
					   "No BoundingBox found");
	else {
	  out << -x1 << ' ' << -y1 << " translate\n\n";
	}
      } else out << tmpline << '\n';
    }
    return Vector(_width, _height);
  }
}

void Image_Frame::print(std::ostream &out, bool grayscale) const {
  if(association.empty()) {
    Cached_Frame::print(out);
    return;
  }
  
  out << "save\n"
      << "/showpage {} def\n"
      << PS::Concat(get_matrix())
      << "0 setgray 0 setlinecap 1 setlinewidth\n"
      << "0 setlinejoin 10 setmiterlimit [] 0 setdash newpath\n";
  if(true) //FIXME!!! if(level2)
    out << "false setoverprint false setstrokeadjust\n";

  // redefine color operators
  // NOTE: does not work for images
  // NOTE: this does not work for CIE-based or special color spaces
  if(grayscale) {
    out << "/setrgbcolor {setrgbcolor currentgray setgray} bind def\n"
	<< "/sethsbcolor {sethsbcolor currentgray setgray} bind def\n"
	<< "/setcmykcolor {setcmykcolor currentgray setgray} bind def\n"
	<< "/setcolor {setcolor currentgray setgray} bind def\n";
  }

  out << "%%BeginDocument: " << association << '\n';
  PS::WinEPSFilter in(association.c_str());
  if(in) {
    try {
      parse(in, out);
    } catch(Gen_Pic_Error e){
      throw Error::Print(get_name() + ": " + e.what());
    }
  } else
    throw Error::Print(get_name() + ": Couldn't read " + association);
  out << "%%EndDocument\n"
      << "restore" << std::endl;
}

namespace {
  class Set_Association_Action: public Action {
  public:
    Set_Association_Action(const Glib::ustring &_new_ass, 
			   Image_Frame &_frame) :
      Action("Change associated EPS"), frame(_frame),
      old_ass(frame.get_association()), new_ass(_new_ass)
    {}
    void undo() const {frame.set_association(old_ass);}
    void redo() const {frame.set_association(new_ass);}
  private:
    Image_Frame &frame;
    Glib::ustring old_ass, new_ass;
  };
}

void Image_Frame::set_association(const Glib::ustring &s, bool push_undo) {
  if(association == s) 
    return;
  
  if(push_undo)
    Document::containing(*this).push_action
      (new Set_Association_Action(s, *this));
  // important to create action object before association is set

  association = s;
  clear_picture();
  props_changed_signal(this);
  ready_to_draw_signal(this);
}

void Image_Frame::generate_picture(View& view) {
  debug << "PID " << pid << std::endl;
  if(association.empty())
    throw Gen_Pic_Error(ASSOCIATION, "No associated file");
  if(!picture && pid == -1) {
      std::ostringstream tmp;

      const Matrix &m = get_matrix();

      {
	PS::WinEPSFilter in(association);
	std::ofstream out(parsed_file.c_str());
	if(!in)
	  throw Gen_Pic_Error(ASSOCIATION, "Couldn't open " + association);
	if(!out)
	  throw Gen_Pic_Error(GENERATION, "Couldn't write to " + parsed_file);

	// Ghostscript needs a showpage
	out << "%!PS\n"
	    << "save\n"
	    << "/showpage {} def\n"
	    << PS::Concat(Matrix::scaling(m.sc_x(), m.sc_y()));
	Vector size = parse(in, out); width = size.x; height = size.y;
	out << "restore\n"
	    << "showpage" << std::endl;
      }

      const Gdk::Point psize(int(view.pt2scr(width * m.sc_x())), 
			     int(view.pt2scr(height * m.sc_y())));
      if(psize.get_x() == 0 || psize.get_y() == 0)
	throw Gen_Pic_Error(ZEROSIZE, "Picture has zero size");

      const std::string &psinterpreter = config.PSInterpreter.values.front();
      if(psinterpreter.empty())
	throw Gen_Pic_Error(GENERATION, "No PostScript interpreter specified");
      // -dSAFER prevents postscript code from writing to files
      // there is also a -dPARANOIDSAFER option that prevents read access
      tmp << psinterpreter //assuming it is ghostscript
	  << " -q -dSAFER -dNOPAUSE -dBATCH -sDEVICE=ppmraw -r"
	  << view.get_scrres()
	  << " -g" << psize.get_x() << 'x' << psize.get_y()
	  << " -sOutputFile=- "
	  << parsed_file
	  << " >" << tmp_file;
      pid = process_manager.start(tmp.str());
      verbose << pid << ": " << tmp.str() << std::endl;
      if(pid == -1)
	throw Gen_Pic_Error(GENERATION, "Failed to run " + psinterpreter);
  }
}

void Image_Frame::file_modified() {
  clear_picture();
  ready_to_draw_signal(this);
}

// Note: as long as this is only called in the constructor there is no need to
// emit a geometry_changed_signal
void Image_Frame::read_size() {
  width = 100; height = 100; 
  // Must set the size to something, so we can see the frame even on failure.
  if(!association.empty()) {
    PS::WinEPSFilter in(association);
    std::string tmpline;
    while(getline(in, tmpline)) {
      if(starts_with(tmpline, "%%BoundingBox:")) {
	std::istringstream tmp(tmpline.c_str());
	std::string word;
	float x1, y1, x2, y2;
	if(tmp >> word >> x1 >> y1 >> x2 >> y2) {
	  width = x2 - x1;  height = y2 - y1;
	  return;
	}
	else
	  warning << "Bad bounding box in " << association << std::endl;
      }
    }
  }
}
