///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "page.h"
#include "fileerrors.h"
#include "pptcore.h"
#include "config.h"
#include "globals.h"
#include "view.h"		// Todo:  Remove when possible
#include "document.h"		// Todo:  Remove if/when possible
#include <util/stringutil.h>
#include <util/warning.h>
#include <algorithm>

SigC::Signal1<void, Pagent*> Page::ready_to_draw_page_signal;

namespace {
  struct PNum_Hack { //page numbering hack
    static const bool use=false;
    static const int xpos=40; //points
    static const int ypos=40; //points
    static std::string font;
    static const int font_size=10;
  } pnum_hack;

  //  std::string PNum_Hack::font="Palatino-Italic";
  std::string PNum_Hack::font="URWPalladioL-Ital";
}

int Page::get_page_num() const {
  return document.get_page_num_of_page(this);
}

Page::Page(Document& d)
  :Group(0, ""),
   document(d)
{}

Page::Page(const xmlpp::Element& node, Document& d)
  :Group(0, ""),
   document(d)
{
  const xmlpp::Element::AttributeList attributes=node.get_attributes();
  for(xmlpp::Element::AttributeList::const_iterator i = attributes.begin();
      i != attributes.end(); i++)
    if((*i)->get_name() == "name")
      set_name((*i)->get_value());
    else
      warning << "Unknown attribute \"" << (*i)->get_name()
	      << "\" ignored in <page>." << std::endl;

  // read pagents:   Todo:  Avoid duplicate (groupmeta.cc:6)
  const xmlpp::Node::NodeList children = node.get_children();
  for(xmlpp::Node::NodeList::const_iterator i = children.begin();
      i != children.end(); i++) {
    if(const xmlpp::Element* elem = dynamic_cast<const xmlpp::Element*>(*i)) {
      std::string name = elem->get_name();
      if(name == "frame") { 
	const xmlpp::Attribute* type_attr = elem->get_attribute("type");
	if(!type_attr)
	  throw Error::Read("No type attribute for frame");
	
	const std::string type = type_attr->get_value();
	 
	if(MetaBase* loader = core.getMeta(type))
	  add(loader->load(*elem, this));
	else
	  throw Error::Read("No appropriate loader for  \"" + type + "\".");
      } else if(name == "guide") {
	guides.push_back(Guide(false, 0));
	const xmlpp::Attribute* orientation_attr = 
	  elem->get_attribute("orientation");
	if(orientation_attr) {
	  const std::string o = orientation_attr->get_value();
	  if(o == "h")
	    guides.back().is_horizontal = true;
	  else if(o != "v") // "v" is default
	    throw Error::Read("\"" + o + "\" is not a proper " 
			      "guide orientation.\n"
			      "Allowed values are \"v\" and \"h\"");
	}
	const xmlpp::Attribute* position_attr = 
	  elem->get_attribute("pos");
	if(!position_attr)
	  throw Error::Read("Guide position is undefined");
	try {
	  guides.back().position = to<float>(position_attr->get_value());
	} catch(std::runtime_error e) {
	  throw Error::Read("Bad guide position: " 
			    + position_attr->get_value());
	}
      } else if(!name.empty()) {
	warning << "Unknown node <" << name
		<< "> ignored in <page>." << std::endl;
      }
    }
  }
  // Todo: Maybe throw an error if *i was a text node (but not if it was a
  // comment) ...
}

Page::~Page()
{}

xmlpp::Element *Page::save(xmlpp::Element& parent_node) {
  xmlpp::Element *node = parent_node.add_child("page");
  for(Guides::const_iterator i = guides.begin(); i != guides.end(); i++) {
    node->add_content("\n      "); // Todo:  Just tell libxml++ to indent?
    xmlpp::Element *guide_node = node->add_child("guide");
    guide_node->add_attribute("orientation", i->is_horizontal ? "h" : "v");
    guide_node->add_attribute("pos", tostr(i->position));
  }
  save_childs(*node);
  if(!name.empty())
    node->add_attribute("name", name);
  return node;
}

void Page::print(std::ostream &out, bool grayscale) {
  // Page numbering hack:
  if(pnum_hack.use) {
    int page_num=get_page_num();
    bool odd=page_num%2;
    int xpos=odd?(int) get_width()-pnum_hack.xpos:pnum_hack.xpos;
    int ypos=pnum_hack.ypos;
    out << '/' << pnum_hack.font << " findfont "
	<< pnum_hack.font_size << " scalefont setfont" << std::endl
	<< xpos << ' ' << ypos << " moveto (" << page_num << ") show"
	<< std::endl;
  }

  Group::print(out, grayscale);
  
  out << std::endl << "showpage"
      << std::endl << std::endl;
}

std::string Page::get_name() {
  // Todo: is this a bad idea?
  return name.empty() ? tostr(get_page_num()) : std::string(name);
}

float Page::get_width() const {
  return document.get_width();
}

float Page::get_height() const {
  return document.get_height();
}

float Page::get_xpos() const {
  return 0;
}

float Page::get_ypos() const {
  return 0;
}

void Page::draw_guide(View &view,
		      const Guide &guide) 
{
  Glib::RefPtr<Gdk::Drawable> win = view.get_win();
  Glib::RefPtr<Gdk::GC> gc = view.get_gc();
  gc->set_line_attributes(1, Gdk::LINE_ON_OFF_DASH, Gdk::CAP_BUTT, 
			  Gdk::JOIN_MITER);
  gc->set_foreground(view.get_color(Color::guide));
  Gdk::Point start, end;
  if(guide.is_horizontal) {
    start = view.pt2scr(Vector(0, guide.position));
    end = view.pt2scr(Vector(get_width(), guide.position));
  }
  else {
    start = view.pt2scr(Vector(guide.position, 0));
    end = view.pt2scr(Vector(guide.position, get_height()));
  }
  win->draw_line(gc, 
		 start.get_x(), start.get_y(),
		 end.get_x(), end.get_y());
}

typedef std::vector<Gdk::Point> ScrPolygon;

ScrPolygon polygonToScreen(const View& view, const Polygon& poly) {
  // Note; this should be done with the generic copy algorithm
  ScrPolygon result;
  for(Polygon::const_iterator i = poly.begin(); i != poly.end(); ++i)
    result.push_back(view.pt2scr(*i));
  return result;
}

bool Page::draw_content(View& view, bool reshaping) {
  Glib::RefPtr<Gdk::Drawable> win = view.get_win();
  Glib::RefPtr<Gdk::GC> gc = view.get_gc();

  // draw background color
  gc->set_foreground(view.get_color(Color::bg));
  Gdk::Point origin = view.pt2scr(Vector(0, get_height()));
  win->draw_rectangle(gc, true, origin.get_x(), origin.get_y(), 
		      int(view.pt2scr(get_width()) + 0.5),
		      int(view.pt2scr(get_height()) + 0.5));

  // Draw page numbering hack:
  if(pnum_hack.use) {
      int page_num = get_page_num();
      bool odd = page_num % 2;
      int xpos = odd ? int(get_width() - pnum_hack.xpos) : pnum_hack.xpos;
      int ypos = pnum_hack.ypos;
      Gdk::Point pos = view.pt2scr(Vector(xpos, ypos));
      int fs = int(view.pt2scr(pnum_hack.font_size));
      int fs30 = fs * 3 / 10;
      int fs70 = fs * 7 / 10;
      gc->set_line_attributes(1, Gdk::LINE_SOLID, Gdk::CAP_BUTT, 
			      Gdk::JOIN_MITER);
      gc->set_foreground(view.get_color(Color::locked));
      win->draw_line(gc, pos.get_x(), pos.get_y()+fs30, 
		     pos.get_x()+fs, pos.get_y()+fs30);
      win->draw_line(gc, pos.get_x(), pos.get_y()+fs70, 
		     pos.get_x()+fs, pos.get_y()+fs70);
      win->draw_line(gc, pos.get_x()+fs30, pos.get_y(), 
		    pos.get_x()+fs30, pos.get_y()+fs);
      win->draw_line(gc, pos.get_x()+fs70, pos.get_y(), 
		    pos.get_x()+fs70, pos.get_y()+fs);
    }

  // Draw the actual content
  bool done = true;

  const Document::Selection &selection
    = Document::containing(*this).selected();
  
  for(ChildVec::const_reverse_iterator i = prbegin(); 
      i != prend(); 
      i++) {
    // draw content
    done = (*i)->draw_content(view, reshaping) && done;
    
    // draw border
    bool selected = 
      find(selection.begin(), selection.end(), *i) != selection.end();
    gc->set_line_attributes
      (1, selected ? Gdk::LINE_SOLID : Gdk::LINE_ON_OFF_DASH, 
       Gdk::CAP_BUTT, 
       Gdk::JOIN_MITER);
    gc->set_foreground(view.get_color((*i)->get_lock() 
				      ? Color::locked
 				      : Color::frame));
    win->draw_polygon(gc, false,
		      polygonToScreen(view, (*i)->get_box()->get_polygon()));
  }

  for(Guides::const_iterator i = guides.begin(); i != guides.end(); i++)
    draw_guide(view, *i);

  return done;
}

void Page::act_on_zoom_factor_change() {
  for_each(pbegin(), pend(), std::mem_fun(&Pagent::act_on_zoom_factor_change));
}

void Page::addObject(Pagent* obj) {
  add(obj);
  Document::containing(*this).select(obj);
}

void Page::child_ready_to_draw(Pagent *pagent) {
  if(pagent)
    debug << pagent->get_name() << " ready to draw itself" << std::endl;
  ready_to_draw_page_signal(this);
}

void Page::select_all(bool select) {
  document.select_all_on_page(this, select);
}

Page& Page::containing(Pagent& obj) {
  try {
    if(Page* result = dynamic_cast<Page*>(&obj))
      return *result;
    else
      return Page::containing(obj.get_parent());
  } catch(const Error::No_Parent& err) {
    throw std::logic_error
      ("Tried to get Page containing pagent that was not on a Page.");
  }
}

const Page& Page::containing(const Pagent& obj) {
  try {
    if(const Page* result = dynamic_cast<const Page*>(&obj))
      return *result;
    else
      return Page::containing(obj.get_parent());
  } catch(const Error::No_Parent& err) {
    throw std::logic_error
      ("Tried to get Page containing pagent that was not on a Page.");
  }
}
