#ifndef PAGENT_H		// -*- c++ -*-
#define PAGENT_H
///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "view.h"
#include "util/matrix.h"
#include "util/boundary.h"
#include <sigc++/object.h>
#include <libxml++/nodes/element.h>
#include <stdexcept>

class Group;

namespace Error {
  struct No_Parent : public std::logic_error {
    No_Parent() : logic_error("Pagent has no parent") {}
  };
  struct Invalid_Page_Num : public std::logic_error {
    Invalid_Page_Num() : logic_error("Invalid page number") {}
    Invalid_Page_Num(int num);
  };
}

/**
 * A Pagent is a Page Component, something that can be placed on a Page.  Most
 * Pagent's are also Basic_Frames (the exact distinction between a Pagent and
 * a Basic_Frame is currently rather muddy).
 */
class Pagent: public SigC::Object {
public:
  Pagent(Group *_parent, const Glib::ustring& _name);
  virtual ~Pagent();

  virtual xmlpp::Element *save(xmlpp::Element& parent_node) const = 0;
  virtual void print(std::ostream &out, bool grayscale = false) const = 0; 
  // should throw Print_Error


  // *** undoable actions ***
  virtual void set_translation(const Vector& v, bool push_undo = false);
  virtual void set_rotation(float angle, bool push_undo = false);
  virtual void set_scaling(float xfactor, float yfactor, 
			   bool push_undo = false);
  virtual void set_size(float w, float h, bool push_undo = false);
  virtual void set_lock(bool _locked, bool push_undo = false);
  // a locked pagent cannot be moved or reshaped with the mouse
  virtual void set_flow_around(bool _flow_around, bool push_undo = false);
  virtual void set_obstacle_margin(float margin, bool push_undo = false);
  virtual void set_name(const Glib::ustring &_name, bool push_undo = false);
  

  // *** probe state ***
  virtual const Glib::ustring &get_name() const {return name;}

  virtual bool get_lock() const {return locked;}
  virtual Boundary get_box() const = 0;
  virtual const Matrix& get_matrix() const {return matrix;} 
  bool is_resizable() {return resizable;}
  virtual float get_width() const {return width;}
  virtual float get_height() const {return height;}

  virtual bool get_flow_around() const {return flow_around;}
  virtual Boundary get_obstacle_boundary() const = 0;
  virtual float get_obstacle_margin() const {return obstacle_margin;}

  virtual int get_page_num() const; // Ask the page what number it is
  // May throw Error::Invalid_Page_Num if there is a problem 
  // with the pages in the document


  // *** non-undoable actions ***
  virtual void set_parent(Group *parent_);
  virtual void set_matrix(Matrix m);


  // *** yucky gui stuff that shouldn't even be in this class ***

  // Returns true if the content is drawn completely, false if something
  // remains.  On false, it is the callers responsibility to make sure the
  // pagent is redrawn again later.
  virtual bool draw_content(View& view, bool reshaping = false) = 0;
  // Release memory needed to display pagent
  // when pagent is no longer visible
  virtual void clear_cache() {} 
  virtual void act_on_zoom_factor_change() = 0;

  // *** access parent objects ***
  const Group& get_parent() const;
  Group& get_parent();

  SigC::Signal1<void, Pagent*> ready_to_draw_signal; 
  // indicated to owner that it might want to redraw the pagent
  SigC::Signal1<void, Pagent*> props_changed_signal; 
  // anything has changed, anything at all, 
  // including position and shape
  SigC::Signal1<void, Pagent*> geometry_changed_signal; 
  // the pagent has changed shape / position

protected:
  Group *parent;
  bool resizable, locked, flow_around;
  Matrix matrix;
  Glib::ustring name;
  float obstacle_margin;
  float width, height;

  // connections to parent
  SigC::Connection draw_connection, geometry_connection, props_connection;
  void connect_to_parent();

private:  
  // Undefined ctors, avoid defaults
  Pagent(const Pagent&);
  Pagent();
  void operator = (const Pagent&);
};

#endif
