///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "processman.h"
#include "safeslot.h"
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <stdexcept>
#include "util/warning.h"
#include "util/os.h"
#include "util/filesys.h"

extern char **environ;
namespace {
  std::string get_template(std::string base) {
    base += "XXXXXX";
    if(base[0] != '/') {
      const std::string tmpdir(os::get_env("TMPDIR"));
      if(tmpdir != "") {
	base = tmpdir + '/' + base;
      } else {
	base = "/tmp/" + base;
      }
    }
    // Todo: verbose is not initialized when the constructor is executed
    verbose << "Temp file base is " << base << std::endl;
    return base;
  }
}

Tempfile::~Tempfile() {
  // delete the temp file
  try {
    unlink(filename);
  } catch(std::exception& err) {
    warning << "Failed to delete " << filename 
	    << ", " << err.what() << std::endl;
  }
}

Process_Manager::Process_Manager(const std::string& filename_base)
  : filename_template(get_template(filename_base))
{
  // Get a callback to run_check every 100ms.
  Glib::signal_timeout().connect
    (safeslot(*this, &Process_Manager::run_check), 100);
}

Process_Manager::~Process_Manager() {
  stop_all();
}

// Note: This method creates the temp file with suitable permissions, but then
// it closes it for anything to reopen rather than using the actual fd where
// the file is open.
// Todo: Create and use a new version
// tmpfile() is said to be more portable
std::string Process_Manager::find_new_name() {
  char* templ = strdup(filename_template.c_str());
  int fd = mkstemp(templ);
  if(fd == -1)
    throw std::runtime_error("Failed to create temp file.");
  close(fd);
  std::string result(templ);
  delete templ;
  return result;
}

pid_t Process_Manager::start(std::string command) {
  pid_t pid = fork();
  if(pid == -1)
    return -1;
  if(pid == 0) {
    char *tmp = new char[command.length() + 1];
    command.copy(tmp, std::string::npos); tmp[command.length()] = 0;
    char *argv[4] = {"sh", "-c", tmp, 0};
    execve("/bin/sh", argv, environ);
    delete[] tmp;
    exit(127);
  }
  processes[pid] = Process(command);
  return pid;
}

int Process_Manager::system(std::string command) {
  int status; 
  pid_t pid = fork();
  if(pid == -1)
    return -1;
  if(pid == 0) {
    char *tmp = new char[command.length() + 1];
    command.copy(tmp, std::string::npos); tmp[command.length()] = 0;
    char *argv[4] = {"sh", "-c", tmp, 0};
    execve("/bin/sh", argv, environ);
    delete[] tmp;
    exit(127);
  }
  if(waitpid(pid, &status, 0) == -1 || !WIFEXITED(status))
    return -1; // no process or abnormal termination
  else
    return WEXITSTATUS(status); // return exit status
}

bool Process_Manager::stop_all() {
  bool tmp = true; 
  while(!processes.empty())
    tmp = tmp && stop(processes.begin()->first);
  return tmp;
}

bool Process_Manager::stop(pid_t pid) {
  Processes::iterator i = processes.find(pid);
  if(i != processes.end()) {
    bool tmp = kill(i->first, 9) == 0;
    debug << i->first << " killed" << std::endl;
    processes.erase(i);
    return tmp;
  }
  else return false;
}

Process *Process_Manager::get_process(pid_t pid) {
  Processes::iterator i = processes.find(pid);
  if(i != processes.end())
    return &(i->second);
  return 0;
}

bool Process_Manager::run_check() {
  pid_t tmp;
  int status;
  while((tmp = waitpid(-1, &status, WNOHANG)) != 0 && tmp != -1) {
    const Processes::iterator i = processes.find(tmp);
    if(i != processes.end()) {
      debug << tmp << " done" << std::endl;
      processes.erase(i);
      process_stopped(tmp, WIFEXITED(status), 
		      WIFEXITED(status) ? WEXITSTATUS(status) : 0);
    }
  }
  return true;
}
