///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "rasterframe.h"
#include <util/stringutil.h>
#include <util/filesys.h>
#include <util/warning.h>
#include <iomanip>
#include <libxml++/libxml++.h>
#include <ps/encode.h>
#include <ps/misc.h>
#include <ctime> // for profiling
#include <fstream>
#include "config.h"
#include "globals.h"
#include "document.h"
#include "action.h"

Raster_Frame::Raster_Frame(Group *parent, const xmlpp::Element& node)
  :Basic_Frame(parent, node), worker(0)
{
  resizable = true;

  if(!node.get_attribute("type") 
     || node.get_attribute("type")->get_value() != "raster")
    throw Error::Frame_Type("Bad raster-frame type: "
			    + node.get_attribute("type")->get_value());

  if(const xmlpp::Attribute* w = node.get_attribute("width"))
    width = to<float>(w->get_value());
  else
    throw Error::Read("No \"width\" attribute found in raster frame");
  if(const xmlpp::Attribute* h = node.get_attribute("height"))
    height = to<float>(h->get_value());
  else
    throw Error::Read("No \"height\" attribute found in raster frame");

  if(const xmlpp::Attribute* file = node.get_attribute("file")) {
    try { // be a little forgiving
      set_association(Document::containing(*this)
		      .from_relative(file->get_value()));
    } catch (Error::Read e) {
      warning << e.message << std::endl;
    }
  } else
    throw Error::Read("\"file\" attribute missing in Raster_Frame");
}

Raster_Frame::Raster_Frame(Group* parent, const Glib::ustring& filename, 
			   float res)
  :Basic_Frame(parent, 1, 1, "Raster " + basename(filename)),
   association(filename), 
   worker(0)
{
  resizable = true;
 
 try {
    filepixbuf = Gdk::Pixbuf::create_from_file(association);
  } catch(Glib::Error e) {
    throw Error::Read("Cannot load raster image file " + association
		      + "\n" + e.what());
  }
    
  load_unit_size();
  _set_ppi(res > 0 ? Vector(res, res) : get_default_ppi());
}

Raster_Frame::~Raster_Frame() {}

void Raster_Frame::load_unit_size() {
  if(!filepixbuf)
    throw Error::Read("Cannot load unit size from non-existing image");
  
  // Todo: care for what the image says about its resolution?
  unit_size = Gdk::Point(filepixbuf->get_width(), filepixbuf->get_height());
}

void Raster_Frame::file_modified() {
  try {
    filepixbuf = Gdk::Pixbuf::create_from_file(association);
    load_unit_size();
  } catch(Glib::Error e){
    warning << get_name() << ": " << e.what() << std::endl;
  }
  screenpixbuf.clear();
  ready_to_draw_signal(this);
}

bool Raster_Frame::draw_content(View& view, bool reshaping) {
  const Gdk::Point sz(int(view.pt2scr(width)), int(view.pt2scr(height)));

  if(association.empty()) {
    draw_content_missing(view);
    return true;
  }

  if(sz.get_x() == 0 || sz.get_y() == 0)
    return true;

  if(!screenpixbuf) {
    if(!filepixbuf) {
      draw_content_broken(view);
      return true;
    }
    if(!worker) {
      if(get_matrix().is_plain()) {
	screenpixbuf = filepixbuf->scale_simple(sz.get_x(), sz.get_y(),
						Gdk::INTERP_BILINEAR);
      } else {
	worker = new PixbufTransformator
	  (filepixbuf, view, 
	   Matrix::scaling(double(width)/filepixbuf->get_width(),
			   double(height)/filepixbuf->get_height()) 
	   * get_matrix());
	screenpixbuf = worker->getPixbuf();
      }
    } else {
      screenpixbuf = worker->getPixbuf();
    }
  }

  bool done = true;
  if(worker) {
    if(worker->isDone()) {
      delete worker;
      worker = 0;  done = true;
    } else
      done = false;
  }
  const Matrix m = Matrix::scaling(width / sz.get_x(),
				   height / sz.get_y()) * get_matrix();
  const Gdk::Point corner[] = {
    view.pt2scr(m.transform(Vector(0, 0))),
    view.pt2scr(m.transform(Vector(0, sz.get_y()))),
    view.pt2scr(m.transform(Vector(sz.get_x(), 0))),
    view.pt2scr(m.transform(Vector(sz.get_x(), sz.get_y())))
  };
  using std::min;
  const int min_x = min(min(corner[0].get_x(), corner[1].get_x()), 
			min(corner[2].get_x(), corner[3].get_x()));
  const int min_y = min(min(corner[0].get_y(), corner[1].get_y()), 
			min(corner[2].get_y(), corner[3].get_y()));
  view.get_win()->draw_pixbuf(view.get_gc(), screenpixbuf,
			      0, 0, min_x, min_y, -1, -1,
			      Gdk::RGB_DITHER_NORMAL, 0, 0);
  return done;
}

void Raster_Frame::act_on_zoom_factor_change() {
  clear_cache();
}

void Raster_Frame::clear_cache() {
  screenpixbuf.clear();
}

xmlpp::Element *Raster_Frame::save(xmlpp::Element& parent_node) const {
  xmlpp::Element *node = Basic_Frame::save(parent_node);
  node->add_attribute("type", "raster");
  node->add_attribute("file", Document::containing(*this)
		      .to_relative(association));

  node->add_attribute("width", tostr(width));
  node->add_attribute("height", tostr(height));

  return node;
}

namespace {
  struct Statistics{
    bool grayscale;
    bool suggest_rle;
  };

  Statistics calc_stats(Glib::RefPtr<Gdk::Pixbuf> pixbuf, int samples = 100) {
    // Todo: check that samples is well below number of pixels
    Statistics result = {true, false};
    const guint8 *data = pixbuf->get_pixels();
    const int pixnum = pixbuf->get_width() * pixbuf->get_height();
    const int bytes_per_pixel = pixbuf->get_n_channels();
    const int rowstride = pixbuf->get_rowstride();
    debug << "Rowstride: " << rowstride << std::endl;
    int rle_count = 0;
    for(int i = 0; i < 100; i++)
      debug << int(*(data + i)) << ", ";
    for(int i = 0; i < samples; i++) {
      guint32 index = bytes_per_pixel * i * (pixnum / samples);
      // rowstride does not seem to have to be divisible by bytes_per_pixel:
      index = (index / rowstride) * rowstride
	+ ((index % rowstride) / bytes_per_pixel) * bytes_per_pixel;
      const guint8 *pixel = data + index;
      
      // is it grayscale?
      bool grayscale = ((*pixel == *(pixel + 1))
			&& (*pixel == *(pixel + 2)));
      result.grayscale = result.grayscale && grayscale;
      debug << index << ": "
	    << int(*pixel) << " " 
	    << int(*(pixel + 1)) << " " 
	    << int(*(pixel + 2))
	    << " " << result.grayscale << std::endl;
      // Todo: check if b/w
	
      // count similar consecutive pixels:
      rle_count += 
	(grayscale 
	 && *pixel == *(pixel + bytes_per_pixel)
	 && *(pixel + 1) == *(pixel + bytes_per_pixel + 1)
	 && *(pixel + 2) == *(pixel + bytes_per_pixel + 2)
	 )
	? 1 : 0;
    }
    
    // minimum amount of similar pixels when suggesting rle:
    const int suggest_rle_threshold = 3; 
    result.suggest_rle = (rle_count >= suggest_rle_threshold);
    return result;
  }
}

void Raster_Frame::print(std::ostream &out, bool grayscale) const {
  if(!filepixbuf) {
    out << "% - - - no raster data to print for " << get_name() << std::endl;
    warning << "No raster data to print for " << get_name() << std::endl;
  } else {
    assert(filepixbuf->get_colorspace() == Gdk::COLORSPACE_RGB);
    
    Statistics statistics = calc_stats(filepixbuf);
    const bool print_rle = statistics.suggest_rle; 
    const bool print_gray = grayscale || statistics.grayscale;
    
    // Todo: config option for filter, and clever filter selection.
    PS::EncodeFilter *filter;
    PS::ASCIIHexEncodeFilter filterHex;
    PS::ASCII85EncodeFilter filter85;
    PS::RunLengthEncodeFilter filterRLE;
    PS::Filter_Sequence sequence;
    sequence.push_back(&filter85);
    if(print_rle)
      sequence.push_back(&filterRLE);
    PS::EncodeFilterCascade filterCascade(sequence);
    filter = &filterCascade;

    out << "% Raster image: " << basename(association)
	<< "\nsave\n"
	<< PS::Concat(get_matrix())
	<< (print_gray ? "/DeviceGray" : "/DeviceRGB")
	<< " setcolorspace\n"
	<< width << ' ' << height << " scale\n"
	<< "<<\n"
	<< "  /ImageType 1"
	<< "  /Width " << unit_size.get_x()
	<< "  /Height " << unit_size.get_y() << '\n'
	<< "  /BitsPerComponent 8\n"
	<< (print_gray ? "  /Decode [0 1]\n" : "  /Decode [0 1 0 1 0 1]\n")
	<< "  /ImageMatrix ["
	<< unit_size.get_x() << " 0 0 " << -unit_size.get_y() << " 0 "
	<< unit_size.get_y() << "]\n"
	<< "  /DataSource currentfile\n"
	<< "  " << filter->decode_command()
	<< "\n>> image" << std::endl;
    
    std::clock_t start = std::clock();
    // Dump the actual image data to postscript.
    guchar* data = filepixbuf->get_pixels();
    const int bytes_per_row = filepixbuf->get_rowstride();
    const int bytes_per_pixel = filepixbuf->get_n_channels();
    filter->begin(&out); // intialize filter
    for(int y = 0; y < unit_size.get_y(); ++y)
      for(int x = 0; x < unit_size.get_x(); ++x) {
	// Yucky pointer arithmetic ...
	guchar* pixel = data + bytes_per_row * y + bytes_per_pixel * x;
	// Note: This assumes that the 3 first channels are RGB, which seem to
	// be ok currently, even grayscale files are loaded to RGB pixbufs.

	// write one/three bytes:
	if(grayscale) {
	  // convert to grayscale according to the red book:
	  guchar g = guchar(0.3 * pixel[0]     // r
			    + 0.59 * pixel[1]  // g
			    + 0.11 * pixel[2]  // b
			    + 0.5); 
	  filter->write(&g, 1);
	} else
	  filter->write(pixel, print_gray ? 1 : 3); 
	// Todo; write more than 3 at a time to speed it up
      }
    filter->end(); // finalize filter
    verbose << "Time to print " << basename(association) << ": "
	    << (std::clock() - start) / 1000 << " ms ("
	    << (print_gray ? "grayscale" : "color")
	    << (print_rle ? ", RLE" : "") << ")" << std::endl;
    out << "\nrestore" << std::endl;
  }
}

namespace {
  class Set_Association_Action: public Action {
  public:
    Set_Association_Action(const Glib::ustring &_new_ass, 
			   Raster_Frame &_frame) :
      Action("Change associated image"), frame(_frame),
      old_ass(frame.get_association()), new_ass(_new_ass)
    {}
    void undo() const {frame.set_association(old_ass);}
    void redo() const {frame.set_association(new_ass);}
  private:
    Raster_Frame &frame;
    Glib::ustring old_ass, new_ass;
  };
}

void Raster_Frame::set_association(const Glib::ustring &s, bool push_undo) {
  if(s == association)
    return;

  if(push_undo)
    Document::containing(*this).push_action
      (new Set_Association_Action(s, *this));
  // important to create action object before association is set
  
  association = s;
  screenpixbuf.clear();
  try {
    filepixbuf = Gdk::Pixbuf::create_from_file(association);
    load_unit_size();
  } catch(Glib::Error e){
    warning << get_name() << ": " << e.what() << std::endl;
  }
  load_unit_size();
  ready_to_draw_signal(this);
}

Vector Raster_Frame::get_ppi() {
  return Vector(72 * unit_size.get_x() / width, 
		72 * unit_size.get_y() / height);
}

Vector Raster_Frame::get_default_ppi() {
  // could conceivably get resolution from image file
  Config::Floats &values = config.DefaultResolution.values;
  switch(values.size()) {
  case 1: return Vector(values[0], values[0]); break;
  case 2: return Vector(values[0], values[1]); break;
  default: return Vector(72, 72); break;
  }
}

namespace {
  class Set_PPI_Action: public Action {
  public:
    Set_PPI_Action(const Vector& ppi,
		   Raster_Frame &_frame) :
      Action("Change image resolution"), frame(_frame),
      old_ppi(frame.get_ppi()), new_ppi(ppi)
    {}
    void undo() const {frame.set_ppi(old_ppi);}
    void redo() const {frame.set_ppi(new_ppi);}
  private:
    Raster_Frame &frame;
    Vector old_ppi, new_ppi;
  };
}

void Raster_Frame::_set_ppi(const Vector& ppi) {
  width  = 72 * unit_size.get_x() / ppi.x;
  height = 72 * unit_size.get_y() / ppi.y;
}

void Raster_Frame::set_ppi(const Vector& ppi, bool push_undo) {
  if(push_undo)
    Document::containing(*this).push_action(new Set_PPI_Action(ppi, *this));
 // important to create action object before ppi is set
 
  _set_ppi(ppi);
  screenpixbuf.clear();
  geometry_changed_signal(this);
  ready_to_draw_signal(this);
}

void Raster_Frame::set_size(float w, float h, bool push_undo) {
  // override is just so we can know when to run the typesetter again
  if(w != width || h != height) {
    screenpixbuf.clear();
  }
  Pagent::set_size(w, h, push_undo);
}
