///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "streamdialog.h"
#include "widget/wmisc.h"
#include <gtkmm/box.h>
// #include <gtkmm/paned.h>
// #include <gtkmm/frame.h>
#include <gtkmm/button.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/table.h>

#include <util/warning.h>

#include "config.h"
#include "safeslot.h"
#include "document.h"
#include "textstream.h"
#include "globals.h"

/**
 * File-local utility classes in an unnamed namespace
 */
namespace {
  
  class MainModelColumns: public Gtk::TreeModel::ColumnRecord {
  public:
    MainModelColumns() { add(col_name); }
    
    Gtk::TreeModelColumn<Glib::ustring> col_name;
  };
  
  MainModelColumns main_model_columns;
  namespace Response {
    enum ReponseType { CLOSE, ADD, DELETE, SETNAME, SETFILE, SETXFRM, };
  }
}

// *** Stream_Dialog Methods:

Stream_Dialog::Stream_Dialog()
  : DialogWrap("Text Streams"),
    file_entry("Stream File"), 
    xfrm_entry("Stylesheet File", config.StylesheetPath.values.front()),
    document(0)
{
  Gtk::HBox *mainbox = manage(new Gtk::HBox(false, double_space));
  //  Gtk::HPaned *mainbox = manage(new Gtk::HPaned());
  {
    Gtk::VBox *vbox = manage(new Gtk::VBox(false, single_space));
    // listan should go here
    main_model = Gtk::ListStore::create(main_model_columns);
    stream_list.set_model(main_model);
    stream_list.set_headers_visible(false);
    stream_list.append_column("Stream", main_model_columns.col_name);
    // Todo:  Get a more motivated size, not a hardcoded number of pixels.
    stream_list.property_width_request() = 100;
    stream_list.property_height_request() = 100;
    
    Glib::RefPtr<Gtk::TreeSelection> selection = stream_list.get_selection();
    selection->signal_changed().connect
      (safeslot(*this, &Stream_Dialog::on_selection_changed));
    
    Gtk::ScrolledWindow *scroll = manage(new Gtk::ScrolledWindow());
    scroll->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
    scroll->set_shadow_type(Gtk::SHADOW_IN);
    scroll->add(stream_list);
    
    Gtk::Label *list_label = 
      manage(new Gtk::Label("S_treams", 0.0, 0.5, true));
    list_label->set_mnemonic_widget(stream_list);
    vbox->pack_start(*list_label, Gtk::PACK_SHRINK);
    vbox->pack_start(*scroll);

    vbox->pack_start(*fix_button(new Gtk::Button(Gtk::Stock::ADD),
				 Response::ADD),
		     Gtk::PACK_SHRINK);
    vbox->pack_start(*fix_button(new Gtk::Button(Gtk::Stock::DELETE),
				 Response::DELETE),
		     Gtk::PACK_SHRINK);
    
//     Gtk::Frame *frame = manage(new Gtk::Frame());
//     frame->add(*vbox); frame->set_shadow_type(Gtk::SHADOW_IN);
//     mainbox->pack1(*frame);
    mainbox->pack_start(*vbox);
  }
  {
    Gtk::VBox *propsbox = manage(new Gtk::VBox(false, single_space));
    this->propsbox = propsbox;
    {
      Gtk::Table *proptbl = manage(new Gtk::Table(2, 3));
      proptbl->set_row_spacings(single_space);
      proptbl->set_col_spacing(0, double_space);

      Gtk::Label *name_label = 
	manage(new Gtk::Label("_Name:", 1.0, 0.5, true));
      name_label->set_mnemonic_widget(name_entry);
      proptbl->attach(*name_label, 0, 1, 0, 1);
      proptbl->attach(name_entry, 1, 2, 0, 1);

      Gtk::Label *file_label = 
	manage(new Gtk::Label("_File:", 1.0, 0.5, true));
      file_label->set_mnemonic_widget(*file_entry.entry.get_entry());
      proptbl->attach(*file_label, 0, 1, 1, 2);
      proptbl->attach(file_entry, 1, 2, 1, 2, 
		      Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);

      Gtk::Label *xfrm_label = 
	manage(new Gtk::Label("_Stylesheet:", 1.0, 0.5, true));
      xfrm_label->set_mnemonic_widget(*xfrm_entry.entry.get_entry());
      proptbl->attach(*xfrm_label, 0, 1, 2, 3);
      proptbl->attach(xfrm_entry, 1, 2, 2, 3,
		      Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
      
      propsbox->pack_start(*proptbl, Gtk::PACK_SHRINK);
      propsbox->set_sensitive(false); // nothing is selected to start with
    }

//     Gtk::Frame *frame = manage(new Gtk::Frame());
//     frame->add(*propsbox); frame->set_shadow_type(Gtk::SHADOW_IN);
//     mainbox->pack2(*frame);
    mainbox->pack_start(*propsbox);
  }
  get_vbox()->pack_start(*mainbox);

  add_button(Gtk::Stock::CLOSE, Response::CLOSE);
  show_all_children();

  Document::streams_changed_signal.connect
    (safeslot(*this, &Stream_Dialog::update));
   
  // Todo: safeslot on the following?
  
  name_entry.signal_activate().connect
    (bind(slot(*this, &Stream_Dialog::on_response), Response::SETNAME));
  
  file_entry.entry.get_entry()->signal_activate().connect
    (bind(slot(*this, &Stream_Dialog::on_response), Response::SETFILE));
  
  xfrm_entry.entry.get_entry()->signal_activate().connect
    (bind(slot(*this, &Stream_Dialog::on_response), Response::SETXFRM));
}

Stream_Dialog::~Stream_Dialog()
{}

void Stream_Dialog::update(Document *document_) {
  if(document_ != document)
    return;
  
  main_model->clear();
  if(document) {     
    Document::StreamVec tmp = document->get_text_streams();
    for(Document::StreamVec::iterator i = tmp.begin();
	i != tmp.end();
	++i)
      {
	Gtk::TreeModel::Row row = *(main_model->append());
	row[main_model_columns.col_name] = (*i)->get_name();
      }
  }
  
  // don't disable the close button
  (*get_vbox()->get_children().begin())->set_sensitive(document);
}

void Stream_Dialog::on_response(int response_id) {
  switch(response_id) {
  case Response::CLOSE:
    hide();
    break;

  case Response::ADD:
    debug << "Stream_Dialog: add\n";
    document->add_text_stream("", document->make_up_new_name(), "");
    Document::streams_changed_signal(document);
    break;
    
  case Response::DELETE:
    debug << "Stream_Dialog: delete\n";
    if(document && current_stream)
      document->remove_text_stream(current_stream->get_name());
    Document::streams_changed_signal(document);
    break;
    
  case Response::SETNAME:
    if(current_stream) {
      Glib::ustring name = name_entry.get_text();
      if(name == current_stream->get_name())
	return;
      if(document->get_text_stream(name)) {
	Error_Dialog::view("A stream called \"" + name + "\" already exists\n"
			   "Names must be unique.");
 	return;
      }
      current_stream->set_name(name);
      if(Gtk::TreeModel::iterator i = 
	 stream_list.get_selection()->get_selected()) {
	(*i)[main_model_columns.col_name] = name;
      }
    } else
      warning << "ERROR: No current stream in Stream_Dialog!" << std::endl;
    break;
    
  case Response::SETFILE:
    if(current_stream)
      current_stream->set_association(file_entry.entry.get_text(false));
    else
      warning << "ERROR: No current stream in Stream_Dialog!" << std::endl;
    break;
    
  case Response::SETXFRM:
    if(current_stream)
      current_stream->set_transform(xfrm_entry.entry.get_text(false));
    else
      warning << "ERROR: No current stream in Stream_Dialog!" << std::endl;
    break;
    
  default:
    // well ...
    break;
  }
}

void Stream_Dialog::on_selection_changed() {
  if(!document) return;
  typedef Gtk::TreeModel Model;
  
  if(Model::iterator i = stream_list.get_selection()->get_selected()) {
    Model::Row row = *i;
    Glib::ustring name = row[main_model_columns.col_name];
    name_entry.set_text(name);
    
    current_stream = document->get_text_stream(name);
    file_entry.entry.set_text(current_stream->get_association());
    xfrm_entry.entry.set_text(current_stream->get_transform());
    
  } else {
    current_stream = 0;
  }
  
  propsbox->set_sensitive(current_stream);
}

void Stream_Dialog::show_raise() {
  show_all();
  DialogWrap::show_raise();
}

void Stream_Dialog::set_document(Document *document_) {
  document = document_;
  update(document);
}

//*** Stream_Menu methods ***

namespace{
  static const std::string no_stream("( No Stream )"); 
}

Stream_Menu::Stream_Menu()
  : document(0)
{
  update();
}

void Stream_Menu::set_document(Document *document_) 
{
  document=document_;
  update();
}

std::string Stream_Menu::get_stream() const 
{
  const Gtk::Label *tmp=dynamic_cast<const Gtk::Label*> (get_child());
  if(!tmp || tmp->get_text()==no_stream)
    return ""; 
  return tmp->get_text();
}
 
void Stream_Menu::update(std::string select_name) 
{
  Gtk::Menu dummy;
  set_menu(dummy); //so it will notice the change
  using namespace Gtk;
  using namespace Menu_Helpers;
  MenuList& menu_list=menu.items();
  menu_list.clear();
  if(!document)
    {
      menu.items().push_back(MenuElem("")); // cosmetic fix
      menu.set_active(0);
      set_menu(menu);
      return;
    }
  int index=0;
  Document::StreamVec streams=document->get_text_streams();
  for(Document::StreamVec::iterator i=streams.begin();
      i!=streams.end();
      i++)
    {
      std::string name=(*i)->get_name();
      menu.items().push_back(MenuElem(name));
      if(name==select_name)
	menu.set_active(index);
      index++;
    }
  menu.items().push_back(SeparatorElem());
  menu.items().push_back(MenuElem(no_stream));
  if(select_name.empty())
    menu.set_active(index+1);
  set_menu(menu);
}

