///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <sstream>
#include <fstream>

#include <libxml++/libxml++.h>

#include "safeslot.h"
#include "textframe.h"
#include "document.h"
#include "processman.h"
#include "config.h"
#include "globals.h"
#include "action.h"
#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>
#include <ps/misc.h>


Text_Frame::Text_Frame(Group *parent, Text_Stream *stream,
		       float w, float h) :
  Cached_Frame(parent, w, h, "Text " + (stream ? stream->get_name() : "")),
  parsed_file(process_manager.find_new_name()), text_stream(stream),
  num_columns(1), gutter_width(12.0)
{
  resizable = true;
  white_is_transparent = true;
  ps_exists = false;
  parsed_file_lock = false;

  if(text_stream)
    text_stream->add_frame(this);
  signal_end_write_ps.connect
    (safeslot(*this,  &Text_Frame::_end_write_ps));
}

Text_Frame::Text_Frame(Group *parent, const xmlpp::Element& node) :
  Cached_Frame(parent, node),
  parsed_file(process_manager.find_new_name()),
  text_stream(0), num_columns(1), gutter_width(12.0)
{
  resizable = true;
  white_is_transparent = true;
  ps_exists = false;
  parsed_file_lock = false;

  if(!node.get_attribute("type") 
     || node.get_attribute("type")->get_value() != "text")
    throw Error::Frame_Type("Bad text-frame type: " 
			    + node.get_attribute("type")->get_value());

  if(const xmlpp::Attribute* cols = node.get_attribute("num_columns"))
    num_columns = to<int>(cols->get_value());
  
  // Note: This is the gutter width as a plain numeric value, it /should/ be
  // possible to give it as a value with a unit, as "1em" or "5mm" ...
  if(const xmlpp::Attribute* gutter = node.get_attribute("gutter_width"))
    gutter_width = to<float>(gutter->get_value());
  
  if(const xmlpp::Attribute* stream = node.get_attribute("stream")) {
    set_stream(Document::containing(*this)
	       .get_text_stream(stream->get_value()));
    if(!text_stream)
      throw Error::Read("Failed to get text stream \"" + stream->get_value()
			+ "\" for text frame");
  } // If stream is not present text_stream is null. This is ok.
  signal_end_write_ps.connect
    (safeslot(*this,  &Text_Frame::_end_write_ps));
}

Text_Frame::~Text_Frame() {
  if(text_stream)
    text_stream->remove_frame(this);
  try { unlink(parsed_file); }
  catch(std::exception& err) {
    warning << "Failed to delete parsed_file: " << err.what()
	    << std::endl;
  }
}

xmlpp::Element *Text_Frame::save(xmlpp::Element& parent_node) const {
  xmlpp::Element *node = Cached_Frame::save(parent_node);
  node->add_attribute("type", "text");

  node->add_attribute("num_columns", tostr(num_columns));
  node->add_attribute("gutter_width", tostr(gutter_width));
  if(text_stream)
    node->add_attribute("stream", text_stream->get_name());

  return node;
}

void Text_Frame::print(std::ostream &out, bool grayscale) const {
  if(text_stream) {
    out << "% " << name << ", from " << text_stream->get_association() << '\n'
	<< "gsave\n"
	<< PS::Concat(get_matrix());
    text_stream->outputPageRaw(out, this);
    out << "grestore\n" << std::endl;

  } else {
    out << "% " << name << ", text stream without data\n";
  }
}

namespace {
  class Set_Stream_Action: public Action {
    // does not consider remove_from_old
    // Todo: make remove_from_old unnecessary
  public:
    Set_Stream_Action(Text_Stream *_new_stream, Text_Frame &_frame) :
      Action("Change text stream"), frame(_frame),
      old_stream(frame.get_stream()), new_stream(_new_stream)
    {}
    void undo() const {frame.set_stream(old_stream);}
    void redo() const {frame.set_stream(new_stream);}
  private:
    Text_Frame &frame;
    Text_Stream *old_stream, *new_stream;
  };
}

void Text_Frame::set_stream(Text_Stream *new_stream, 
			    bool remove_from_old, bool push_undo) 
{
  if(text_stream == new_stream)
    return;

  if(push_undo)
    Document::containing(*this).push_action
      (new Set_Stream_Action(new_stream, *this));
  // important to create action object before stream is set

  if(remove_from_old && text_stream)
    text_stream->remove_frame(this);

  text_stream = new_stream;
  if(text_stream)
    text_stream->add_frame(this);
  
  // Ok, make the GUI reflect the change
  clear_picture();
  props_changed_signal(this);
  ready_to_draw_signal(this);
}

namespace {
  class Set_Num_Columns_Action: public Action {
  public:
    Set_Num_Columns_Action(unsigned int _new_num_columns, Text_Frame &_frame) :
      Action("Change number of columns"), frame(_frame),
      old_num_columns(frame.get_num_columns()),
      new_num_columns(_new_num_columns)
    {}
    void undo() const {frame.set_num_columns(old_num_columns);}
    void redo() const {frame.set_num_columns(new_num_columns);}
  private:
    Text_Frame &frame;
    unsigned int old_num_columns, new_num_columns;
  };
}

void Text_Frame::set_num_columns(unsigned int columns, bool push_undo) { 
  if(num_columns != columns) {
    if(push_undo)
      Document::containing(*this).push_action
	(new Set_Num_Columns_Action(columns, *this));
    // important to create action object before columns is set

    num_columns = columns;
    props_changed_signal(this);
    if(text_stream)
      text_stream->generate_ps_request(this);
  }
}

namespace {
  class Set_Gutter_Action: public Action {
  public:
    Set_Gutter_Action(float _new_gutter, Text_Frame &_frame) :
      Action("Resize gutter"), frame(_frame),
      old_gutter(frame.get_gutter()),
      new_gutter(_new_gutter)
    {}
    void undo() const {frame.set_gutter(old_gutter);}
    void redo() const {frame.set_gutter(new_gutter);}
  private:
    Text_Frame &frame;
    float old_gutter, new_gutter;
  };
}

void Text_Frame::set_gutter(const float& gutter, bool push_undo) { 
  if(gutter_width != gutter) {
    if(push_undo)
      Document::containing(*this).push_action
	(new Set_Gutter_Action(gutter, *this));
    // important to create action object before gutter is set

    gutter_width = gutter; 
    props_changed_signal(this);
    if(text_stream)
      text_stream->generate_ps_request(this);
  }
}

void Text_Frame::set_size(float w, float h, bool push_undo) {
  // override is just so we can know when to run the typesetter again
  if(w != width || h != height) {
    ps_exists = false;
  }
  Pagent::set_size(w, h, push_undo);
}

void Text_Frame::begin_write_ps() {
  parsed_file_lock = true;
  if(pid != -1) {
    // kill running process
    debug << "Killing process " << pid << std::endl;
    process_manager.stop(pid);
    pid = -1;
  }
}

void Text_Frame::end_write_ps(bool _ps_exists) {
  parsed_file_lock = false;
  ps_exists = _ps_exists;
  signal_end_write_ps();
}

void Text_Frame::_end_write_ps() {
  clear_picture();
  ready_to_draw_signal(this);
}

void Text_Frame::generate_picture(View& view) {
  if(!text_stream)
    throw Gen_Pic_Error(ASSOCIATION, "No associated stream");
  
  Gdk::Point pv(int(view.pt2scr(width)), int(view.pt2scr(height)));
  if(pv.get_x() == 0 || pv.get_y() == 0) 
    throw Gen_Pic_Error(ZEROSIZE, "Picture has zero size");

  if(!ps_exists) {
    text_stream->generate_ps_request(this);
    return;
  }
  

  { // create the PostScript file
    std::ofstream out(parsed_file.c_str());
    text_stream->outputPageEps(out, this);
  }


  // don't try to start gs if Text_Stream is writing the file
  if(pid != -1 || parsed_file_lock) 
    return;

  std::string &psinterpreter = config.PSInterpreter.values.front();
  if(psinterpreter.empty())
    throw Gen_Pic_Error(GENERATION, "No PostScript interpreter specified");

  std::ostringstream tmp;
  // TODO: -sFONTPATH parameter ?
  tmp << psinterpreter //assuming it is ghostscript
      << " -q -dSAFER -dNOPAUSE -dBATCH -sDEVICE=ppmraw -r" 
      << view.get_scrres()
      << " -g" << pv.get_x() << 'x' << pv.get_y()
      << " -sOutputFile=- " << parsed_file
      << " > " << tmp_file;
  pid = process_manager.start(tmp.str());

  if(pid == -1)
    throw Gen_Pic_Error(GENERATION, "Failed to run "+psinterpreter);

  verbose << pid << ": " << tmp.str() << std::endl;
}
