///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <algorithm>
#include <sstream>
#include <fstream>
#include <sigc++/class_slot.h>

#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>
#include <util/os.h>
#include <xml2ps/typesetter.hh>
#include <fonts/fontmanager.hh>

#include "safeslot.h"
#include "textstream.h"
#include "textframe.h"
#include "globals.h"
#include "processman.h"
#include "page.h"
#include "config.h"

namespace {			// local
  class Sorting_Predicate {
  public:
    bool operator () (const Text_Frame* a, const Text_Frame* b) {
      int a_num, b_num;
      try {
	a_num = a->get_page_num(); 
	b_num = b->get_page_num();
      } catch (Error::Invalid_Page_Num e) {
	// shouldn't happen, but it has happened
      }
      if(a_num < b_num)
	return true;
      if(a_num > b_num)
	return false;
      
      // Same page, look at boxes.
      const Vector a_vec = a->get_box()->getCorner(Corner::UL);
      const Vector b_vec = b->get_box()->getCorner(Corner::UL);
      static const float k = 0.1;  // 10% slope
      const float m = a_vec.y - k * a_vec.x;
      return b_vec.y < k * b_vec.x + m;
    }
  };

  bool gtk_main_running = false;
}


Text_Stream::Text_Stream(const ustring& _name, const ustring& _association,
			 const ustring& _transform)
  :name(_name), association(_association), transform(_transform),
   pid(-1), stylesheet_applied(false),
   typesetter_thread(0)
{
  proc_stop_connection = process_manager.process_stopped.connect
    (safeslot(*this, &Text_Stream::process_stopped));
  parsed_file = process_manager.find_new_name();
  signal_typesetter_error.connect
    (safeslot(*this,  &Text_Stream::on_typesetter_error));

  // make sure Gtk::main() is running 
  Glib::signal_idle().connect
    (safeslot(*this, &Text_Stream::on_idle));
}

Text_Stream::~Text_Stream() {
  // wait for thread to finish
  if(typesetter_thread)
    typesetter_thread->join();

  proc_stop_connection.disconnect();
  try { unlink(parsed_file); }
  catch(std::exception& err) {
    warning << "Failed to delete parsed_file: " << err.what()
	    << std::endl;
  }

  // Todo:  Do this on a copy of frames, after clearing frames.
  // remove_frame has the property that removing an unexisting frame is not an
  // error, just a no-op.
  for(Frames::iterator i = frames.begin();
      i != frames.end();
      i++)
    (*i)->set_stream(0, false);
  // do not let the frame remove itself from this stream
}

void Text_Stream::file_modified() {
  apply_stylesheet();
}

void Text_Stream::add_frame(Text_Frame *text_frame) {
  // Don't insert a duplicate entry
  if(find(frames.begin(), frames.end(), text_frame) == frames.end())
    frames.push_back(text_frame);
}

void Text_Stream::remove_frame(Text_Frame *text_frame) {
  Frames::iterator i = find(frames.begin(), frames.end(),
			    text_frame);
  if(i != frames.end()) 
    frames.erase(i);
}

void Text_Stream::apply_stylesheet() {
  if(association.empty() || !access(association))
    throw Basic_Frame::Gen_Pic_Error(Basic_Frame::ASSOCIATION, 
  				     "Couldn't open " + association);

  if(transform.empty()) { // assume there is no need to apply a stylesheet
    debug << "No stylesheet to apply for " << association << std::endl;
    stylesheet_applied = true;
    // new content - OK do to the same thing again:
    run_typesetter();
    return;
  }

  debug << "Apply stylesheet" << std::endl;
  std::ostringstream cmd;
  // Assume xsltproc is in the path
  cmd << "xsltproc --nonet";
  typedef std::map<ustring, ustring> amap;
  for(amap::const_iterator i = parameters.begin(); i != parameters.end(); ++i)
    cmd << " --param " << i->first << " '\"" << i->second << "\"'";
  cmd << ' ' << transform << ' ' << association
      << " > " << parsed_file;
  
  if(pid != -1) {
    debug << "Stopping " << pid << std::endl;    
    process_manager.stop(pid);
  }

  pid = process_manager.start(cmd.str());
  verbose << pid<< ": " << cmd.str() << std::endl;
  if(pid == -1)
    throw Basic_Frame::Gen_Pic_Error(Basic_Frame::GENERATION, 
				     "Failed to run xsltproc");
}

void Text_Stream::generate_ps_request(Text_Frame *frame) {
  if(frame)
    debug << "Request by " << frame->get_name() << std::endl;
  if(pid == -1) {
    if(!stylesheet_applied)
      apply_stylesheet();
    else 
      run_typesetter();
  } else {
    run_typesetter();
  }
}

void Text_Stream::run_typesetter() {
  if(frames.empty() || !stylesheet_applied)
    return;

  if(!gtk_main_running) {
    debug << "not starting typsetter thread because Gtk::Main isn't running"
	  << std::endl;
    return;
  }

  // start typesetter in separate thread
  if(typesetter_thread) {
    debug << "joining previous thread" << std::endl;
    typesetter_thread->join();
  }
  debug << "starting typesetter thread" << std::endl;
  typesetter_barrier.reclose();
  typesetter_thread = Glib::Thread::create
    (SigC::slot_class(*this, &Text_Stream::typesetter_thread_function), true);
  // wait until the typesetter has started properly:
  typesetter_barrier.wait();
  debug << "resuming main thread" << std::endl;
}

void Text_Stream::typesetter_thread_function() {
  // this method is supposed to run in a separate thread
  debug << "inside typesetter thread" << std::endl;
  try {
    Frames sorted_frames(frames);
    sorted_frames.sort(Sorting_Predicate());

    const Glib::ustring &infile = 
      (!transform.empty()) ? parsed_file : association;
    std::ifstream in(infile.c_str());
    if(!in) {
      typesetter_error = "Could not read from " + infile;
      throw std::runtime_error(typesetter_error);      
    }

    xml2ps::Canvas::PageVec pages;
    for(Frames::const_iterator i = sorted_frames.begin(); 
	i != sorted_frames.end(); 
	i++) {
      (*i)->begin_write_ps();
      pages.push_back(xml2ps::PageBoundary((*i)->get_width(),
					   (*i)->get_height(),
					   (*i)->get_num_columns(),
					   (*i)->get_gutter()));
      
      // A boundary for the actual text frame, to make sure it doesn't try to
      // flow around itself.
      const Boundary boundary = (*i)->get_obstacle_boundary();
    
      // Transform from page coords to local textframe coords.
      const Matrix xform((*i)->get_matrix().inv());
      
      // List of boundaries the text should flow around
      const BoundaryVect obstacle_list = Page::containing(**i).obstacle_list();
      for(BoundaryVect::const_iterator j = obstacle_list.begin();
	  j != obstacle_list.end();
	  j++) {
 	if(boundary != *j) {
	  pages.back().addObstacle(*j * xform);
	}
      }
    }

    // ok for caller to proceed:
    typesetter_barrier.open();

    debug << "running internal xml2ps ..." << std::endl;
    try {
      Glib::RefPtr<xml2ps::PsCanvas> new_canvas
	(new xml2ps::PsCanvas(pages, true /* extra pages */));
      new_canvas->setSubstituteFontAliases(true);
      xml2ps::PsConverter parser(*new_canvas.operator->());
      parser.parse_stream(in);
      used_fonts = new_canvas->getUsedFonts();

      // The new canvas is now ready to get data from
      pageno_map.clear();
      swap(new_canvas, canvas);
      int num = 1;
      for(Frames::const_iterator i = sorted_frames.begin(); 
	  i != sorted_frames.end(); 
	  ++i, ++num) {
	pageno_map[*i] = num;
 	(*i)->end_write_ps();
      }
//       const font::FontManager *fm = font::FontInfo::getFontManager();
//       for(font::Fonts::const_iterator i = used_fonts.begin();
// 	  i != used_fonts.end();
// 	  i++) {
// 	debug << "% " << *i << " - "; 
// 	if(fm)
// 	  debug << "" << fm->unalias(*i);
// 	debug << "" << std::endl;
//       }
    } catch(const std::exception& e) {
      warning << e.what() << std::endl;
      debug << "internal xml2ps failed" << std::endl;
      return;
    }
    debug << "internal xml2ps done" << std::endl;

  } catch (...) {
    debug << "typesetter error: " << typesetter_error << std::endl;
    // have to let main thread resume
    // even if there is an error
    if(typesetter_error == "") {
      try { throw; }
      catch (const std::exception& err) {
	typesetter_error = err.what();
      } catch(...) {
	typesetter_error = "unknown error";
      }
    }
    signal_typesetter_error();
  }
}

void Text_Stream::on_typesetter_error() {
  warning << get_name() << ": typesetter error - " 
	  << typesetter_error << std::endl;
}

void Text_Stream::process_stopped(pid_t _pid, bool exited_normally, 
				  int exit_code)
{
  if(_pid != pid)
    return;
  pid = -1;
  if(!exited_normally) {
    warning << "Process " << this->pid << " exited abnormally" << std::endl;
    return; // Todo: do something clever
  }

  debug << pid << ", exit code: " << exit_code << std::endl;

  if(exit_code == 127) { // the shell could not find xsltproc
    warning << "xsltproc is not in $PATH" << std::endl;
    return; // Todo: deal with this
  }
  // Todo: check that the return code 127 is not unique to Bash.

  debug << "Stylesheet done" << std::endl;
  stylesheet_applied = true;
  run_typesetter();
}

void Text_Stream::set_association(const ustring &s) {
  // Todo: run ready_to_draw_signal for all frames. Maybe.
  association = s;
  try {
    apply_stylesheet();
  } catch(Basic_Frame::Gen_Pic_Error e) {
    warning << e.what() << std::endl;
  }
}

const Glib::ustring &Text_Stream::get_association() const {
  return association;
} 

void Text_Stream::set_name(const ustring &s) {
  name = s;
}

const Glib::ustring &Text_Stream::get_name() const {
  return name;
} 

void Text_Stream::set_transform(const ustring& s) {
  if(s == transform)
    return;
  
  transform = s;
  stylesheet_applied = false;
  apply_stylesheet();		// Right on ...
}

const Glib::ustring &Text_Stream::get_transform() const {
  return transform;
}

void Text_Stream::set_parameter(const ustring& name, const ustring& value) {
  parameters[name] = value;
}

void Text_Stream::outputPageRaw(std::ostream& out, const Text_Frame* frame) {
  if(canvas) {
    canvas->flush();
    canvas->appendPage(pageno_map[frame], out);
  } else
    throw std::runtime_error("Missing canvas while running outputPageRaw");
}
  
void Text_Stream::outputPageEps(std::ostream& out, const Text_Frame* frame) {
  if(canvas) {
    canvas->flush();
    canvas->pageEps(pageno_map[frame], out);
  } else
    throw std::runtime_error("Missing canvas while running outputPageEps");
}

// a hack to make sure the gtkmm main loop is running before we
// start any threads
bool Text_Stream::on_idle() {
  if(!gtk_main_running) {
    debug << "Gtk::Main is running" << std::endl;
    // if there has been an idle signal, then everything is cool
    gtk_main_running = true;
  }
  run_typesetter();
  // disconnect from handler
  return false;
}
