#ifndef TEXTSTREAM_H		// -*- c++ -*-
#define TEXTSTREAM_H
///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "filewatcher.h"
#include <xml2ps/pscanvas.hh>
#include <sys/types.h>		// pid_t
#include <list>
#include <glibmm/thread.h>
#include <fonts/fontinfo.hh>
#include <util/barrier.h>

/*
Here's how it works:

Text_Stream keeps a list of Text_Frames.  When association is set,
Text_Stream runs apply_stylesheet to convert the association into the
xml2ps dtd. This only needs to be done when the association is changed
or the files modification date has been changed. When the xsltproc
process has finished, the xml2ps thread is started in run_typesetter,
but only if there are any frames associated with the stream. When
xml2ps has finished, the resulting ps is chopped into one file for
each frame, and a ready_to_draw_signal is emitted from each frame.

When the docview tells the page to draw itself, the text frame looks
at its ps_exists member to detemine if it should read its
parsed_file. If it is false, the frame calls
Text_Stream::generate_ps_request which will run xml2ps if
necessary. This also happens when a frame is resized.
*/

class Text_Frame;

class Text_Stream: public File_Watcher {
public:
  typedef Glib::ustring ustring;
  
  Text_Stream(const ustring& _name, const ustring& _association,
	      const ustring& _transform);
  ~Text_Stream();

  void add_frame(Text_Frame *text_frame);
  void remove_frame(Text_Frame *text_frame);

  void generate_ps_request(Text_Frame *frame = 0);
  
  void set_association(const ustring &s);
  const ustring &get_association() const;

  void set_name(const ustring &s);
  const ustring &get_name() const;

  void set_transform(const ustring& s);
  const ustring &get_transform() const;
  
  void set_parameter(const ustring& name, const ustring& value);

  // This doesn't really feel good, for an interface ...
  typedef std::map<ustring, ustring> ParamMap;
  typedef ParamMap::const_iterator ParamIter;
  ParamIter param_begin() const { return parameters.begin(); }
  ParamIter param_end() const { return parameters.end(); }

  void run_typesetter(); // public, so Document_View can run it 

  const font::Fonts &get_used_fonts() const {return used_fonts;}
  
  /** Get the raw postscript data for a specific frame from the canvas */
  void outputPageRaw(std::ostream& out, const Text_Frame* frame);
  
  /** Get the eps data for a specific frame from the canvas */
  void outputPageEps(std::ostream& out, const Text_Frame* frame);
  
protected:
  typedef std::list<Text_Frame*> Frames;
  ustring name, association, transform, parsed_file;
  Frames frames;
  pid_t pid;
  SigC::Connection proc_stop_connection;
  bool stylesheet_applied;
  
  Glib::Thread *typesetter_thread;
  Barrier typesetter_barrier;
  Glib::Dispatcher signal_typesetter_error;
  ustring typesetter_error;
  
  void typesetter_thread_function();
  void on_typesetter_error();
  void apply_stylesheet();
  std::string file_to_watch() const
  {return association;}
  void file_modified();
  void process_stopped(pid_t  _pid, bool exited_normally, int exit_code);

  bool on_idle();

private:
  font::Fonts used_fonts;
  ParamMap parameters;

  Glib::RefPtr<xml2ps::PsCanvas> canvas;
  std::map<const Text_Frame*, unsigned int> pageno_map;
  
  // Undefined ctors, avoid defaults
  Text_Stream(const Text_Stream&);
  Text_Stream();
  void operator = (const Text_Stream&);
};

#endif
