#include "undodialog.h"
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include "action.h"
#include "document.h"
#include "safeslot.h"
#include "globals.h"

namespace {
  class ModelColumns: public Gtk::TreeModel::ColumnRecord {
  public:
    ModelColumns(){
      add(col_redo); add(col_action); add(col_index);
    }
    Gtk::TreeModelColumn<bool> col_redo;
    Gtk::TreeModelColumn<Glib::ustring> col_action;
    Gtk::TreeModelColumn<int> col_index;
  };
  const ModelColumns columns;
};

Undo_Dialog::Undo_Dialog()
  : DialogWrap("Undo history"),
  locked(false)
{
  set_default_size(200, 300);
  //  get_vbox()->set_border_width(border_width);

  action_list = manage(new Gtk::TreeView()); 
  Gtk::ScrolledWindow *scroller = manage(new Gtk::ScrolledWindow());
  scroller->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  scroller->add(*action_list);
  get_vbox()->pack_start(*scroller);

  tree_model = Gtk::ListStore::create(columns);
  action_list->set_model(tree_model);

  Gtk::CellRendererText *text_renderer = manage(new Gtk::CellRendererText());
  int col_count = action_list->append_column("Action", *text_renderer);
  Gtk::TreeViewColumn* column = action_list->get_column(col_count - 1);
  if(column) {
    column->add_attribute(text_renderer->property_markup(), 
			  columns.col_action);
    column->add_attribute(text_renderer->property_strikethrough(), 
			  columns.col_redo);
  }

  Document::undo_changed_signal.connect
    (safeslot(*this, &Undo_Dialog::on_undo_changed));

  Glib::RefPtr<Gtk::TreeSelection> selection =
    action_list->get_selection();
  selection->signal_changed().connect
    (safeslot(*this, &Undo_Dialog::on_selection_changed));

  undo_button = add_button(Gtk::Stock::UNDO, 1);
  undo_button->set_sensitive(false);
  redo_button = add_button(Gtk::Stock::REDO, 2);
  redo_button->set_sensitive(false);
  add_button(Gtk::Stock::CLOSE, 0);
}

Undo_Dialog::~Undo_Dialog() {
}

void Undo_Dialog::show_raise() {
  update(false);

  // TreeView is stupid. I don't want anything selected!
  locked = true;
  show_all();
  locked = false;
  action_list->get_selection()->unselect_all();
  
  DialogWrap::show_raise();
}

void Undo_Dialog::set_document(Document *_document) {
  if(_document != document) {
    document = _document;
    update();
  }
}

bool Undo_Dialog::on_button_release_event(GdkEventButton* event) {
  // This is a hack to make the selection work as a clicked signal
  action_list->get_selection()->unselect_all();
  return true;
}

void Undo_Dialog::on_selection_changed() {
  if(locked)
    return;
  if(!document)
    return;
  locked = true;
  Glib::RefPtr<Gtk::TreeSelection> selection =
    action_list->get_selection();
  Gtk::TreeModel::iterator i = selection->get_selected();
  if(i) {
    Gtk::TreeModel::Row row = *i;
    if(row[columns.col_redo]) {
      for(int i = 0; i < -row[columns.col_index]; i++) {
	document->redo();
      }
    } else {
      for(int i = 0; i <= row[columns.col_index]; i++) {
	document->undo();
      }
    }
  }
  update();
  locked = false;
}

void Undo_Dialog::on_response(int response_id) {
  switch(response_id) {
  case 0: hide(); break;
  case 1: if(document) document->undo(); break;
  case 2: if(document) document->redo(); break;
  default: break;
  }
}

void Undo_Dialog::on_undo_changed(Document *_document) {
  if(_document == document && !locked)
    update();
}

void Undo_Dialog::update(bool only_if_visible) {
  // If a dialog window is updated in the forest and nobody is
  // watching, does it waste CPU cycles?
  if(!only_if_visible || is_visible()) { 
    // Can't clear the list when inside the on_selection_changed callback.
    // The "locked" member is a workaround.
    if(!locked) {
      action_list->get_selection()->unselect_all();
      tree_model->clear();
    }
    if(!document)
      return;
    int uc = document->get_undo_count();
    int rc = document->get_redo_count();
    undo_button->set_sensitive(uc);
    redo_button->set_sensitive(rc);
    typedef Gtk::TreeModel::Children Children;
    Children children = tree_model->children();
    Children::iterator iter = children.begin();
    for(int i = uc - 1; i >= -rc; --i) {
      Gtk::TreeModel::Row row;
      if(!locked) {
	const Action *action = 
	  dynamic_cast<const Action*>(document->get_action(i)); 
	row = *(tree_model->append());
	row[columns.col_action] = 
	  action 
	  ? action->get_desc() 
	  : "<description not available>";
      } else {
	row = *(iter++);
      }
      row[columns.col_redo] = (i < 0);
      row[columns.col_index] = i;
    }
  }
}

