///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "filesel.h"
#include <util/stringutil.h>
#include <util/warning.h>

// *** End_Entry methods ***

End_Entry::End_Entry(int _history_max) 
  :history_max(_history_max) 
{
  disable_activate();
}

void End_Entry::set_text(const Glib::ustring &text) {
  get_entry()->set_text(text); 
  // move cursor to end so the filename is visible
  get_entry()->set_position(text.length());  
}

Glib::ustring End_Entry::get_text(bool remember_text) {
  if(remember_text) 
    remember();
  return get_entry()->get_text();
}

void End_Entry::remember() {
  Glib::ustring text = get_entry()->get_text();
  for(History::iterator i = history.begin();
      i != history.end(); i++)
    if(*i == text) {
      History::iterator j = i++;
      history.erase(j);
    }
  history.push_front(text); 
  // put text in front, or the entry will be altered
  while(history.size() > history_max)
    history.pop_back();
  set_popdown_strings(history);
}

// *** File_Entry methods ***

File_Entry::File_Entry(const Glib::ustring& window_title_, 
		       const Glib::ustring& default_path_):
  window_title(window_title_), default_path(default_path_),
  button(" Browse ... "), filesel(0)
{
  pack_start(entry, Gtk::PACK_EXPAND_WIDGET, 0);
  pack_start(button, Gtk::PACK_SHRINK, 0);
  button.show();
  entry.show();
  button.set_size_request(-1, entry.get_height());
  button.signal_clicked().connect(slot(*this, &File_Entry::show_filesel));
}

File_Entry::~File_Entry() {
  delete filesel;
}

bool File_Entry::on_mnemonic_activate(bool group_cycling) {
  // Not much documentet in gtkmm, but it seems this metod gets called when
  // this widget is activated (by a mnemonic e.g. from a label) so I can put
  // focus on the proper part of the widget.
  entry.get_entry()->grab_focus();
  return true;
}

void File_Entry::filesel_done() {
  if(!filesel->was_cancelled()) {
    entry.set_text(filesel->get_filename());
    entry.get_entry()->activate();
  }
  delete filesel;
  filesel = 0;
}

void File_Entry::show_filesel() {
  Gtk::Window *toplevel = dynamic_cast<Gtk::Window*>(this->get_toplevel());
  assert(toplevel);
  filesel = new Filesel(*toplevel, window_title);
  filesel->signal_hide().connect(slot(*this, &File_Entry::filesel_done));
  filesel->set_filename(entry.get_text().empty() 
			? default_path 
			: entry.get_text());
  filesel->show();
}

// *** Filesel methods ***

Filesel::Filesel(Gtk::Window &parent, const Glib::ustring& title)
  :Gtk::FileSelection(title), cancelled(false)
{
  set_modal(true);
  set_has_separator(false);
  set_transient_for(parent);

//   get_ok_button()->signal_clicked().connect(slot(*this, &Filesel::ok));
//   get_cancel_button()->signal_clicked().connect(slot(*this, &Filesel::cancel));
}

void Filesel::on_response(int response_id) {
  switch(response_id) {
  case Gtk::RESPONSE_OK:
    {
      std::string filename = get_filename();
      // don't accept directories
      if(filename.at(filename.length() - 1) != '/') {
       cancelled = false;
       hide();
      }
    }
    break;
  case Gtk::RESPONSE_CANCEL: case Gtk::RESPONSE_DELETE_EVENT:
    // Gtk::RESPONSE_DELETE_EVENT is sent when the user tries to close
    // the window
    set_filename("");
    cancelled = true;
    hide();
    break;
  default:
    warning << "Impossible error (you didn't see this)"
	    << " in Filesel::on_response()" << std::endl;
    break;
  }
}
