///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "window.h"
#include "globals.h"
#include "widget/wmisc.h"
#include <iostream>
#include <sstream>
#include <fstream>
#include <vector>
#include <set>
#include <gtkmm/adjustment.h>
#include <gtkmm/label.h>
#include <gtkmm/stock.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/table.h>
#include <gtkmm/image.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/optionmenu.h>
#include <gtkmm/menubar.h>
#include <gtkmm/toolbar.h>

#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>
#include <util/os.h>

#include "widget/filesel.h"
#include "icons/icons.h"

#include "processman.h"
#include "safeslot.h"
#include "textframe.h"
#include "textstream.h"
#include "page.h"
#include "errordialog.h"
#include "config.h"
#include "inspiration.h"
#include "printdialog.h"
#include "aboutdialog.h"
#include "propertiesdialog.h"
#include "docpropsdialog.h"
#include "streamdialog.h"
#include "undodialog.h"
#include "pagesel.h"
#include "webbrowser.h"

PropertiesDialog *Frame_Window::properties_dialog(0);
Stream_Dialog *Frame_Window::stream_dialog(0);
Frame_Window *Frame_Window::active_window(0);
Doc_Props_Dialog *Frame_Window::doc_props_dialog(0);
Undo_Dialog *Frame_Window::undo_dialog(0);

namespace{
  About_Dialog* about_dialog(0);
  typedef std::set<Frame_Window*> Windows;
  Windows windows; // a list of all windows

  typedef std::vector<Glib::ustring> ZoomFactors;
  ZoomFactors zoom_factors;

  class Image_Filesel: public Filesel {
  public:
    Image_Filesel(Gtk::Window &parent, const Glib::ustring& title):
      Filesel(parent, title), res(0, true)
    {
      res.set(config.DefaultResolution.values.front());
      Gtk::Box *box = manage(new Gtk::HBox(false, double_space));
      box->pack_end(*manage(new Gtk::Label("ppi")),
		      Gtk::PACK_SHRINK);
      box->pack_end(res, Gtk::PACK_SHRINK);
      box->pack_end(*manage(new Gtk::Label("Image resolution:")),
		      Gtk::PACK_SHRINK);
      get_main_vbox()->pack_start(*box,Gtk::PACK_SHRINK, single_space);
    }
    float get_res() const {return res.get();}
  private:
    Spinner res; // resolution;
  };
}

// A bit more clever than a usual conversion function
// - it understands per cent signs.
float str2float(const std::string& value) {
  std::istringstream s(value.c_str());
  float f, factor = 1;
  if(s >> f) {
    char ch;
    if(s >> ch)
      if(ch == '%') factor = 0.01;
      else throw std::runtime_error("Strange unit in \"" + value + '"');
    return f * factor;
  } else
    throw std::runtime_error("Not a float value: \"" + value + '"');
}


Frame_Window::Frame_Window(const Glib::ustring &filename, Document *document):
  document_view(0), icons(new Icons())
{
  // temp file for gv to read from when doing a print preview:
  preview_tmp_file = process_manager.find_new_name();

  pagesel = manage(new Pagesel(document_view));

  // These are deleted in the destructor
  print_dialog = new Print_Dialog(*this, document_view);
  open_dialog = new Filesel(*this, "Open");
  save_dialog = new Filesel(*this, "Save As");
  import_dialog = new Image_Filesel(*this, "Import Image");

  if(!doc_props_dialog)
    doc_props_dialog = new Doc_Props_Dialog();
  if(!properties_dialog)
    properties_dialog = new PropertiesDialog();
  if(!stream_dialog)
    stream_dialog = new Stream_Dialog();
  if(!about_dialog)
    about_dialog = new About_Dialog();
  if(!undo_dialog)
    undo_dialog = new Undo_Dialog();

  toolbar = manage(new Gtk::Toolbar()); // before create_menus()
  menubar = manage(new Gtk::MenuBar()); // before create_menus()
  create_menus(); 
  
  zoom_lock = false;
  set_default_size(450, 600);

  Error_Dialog::init(*this);

  if(zoom_factors.empty()) {
    zoom_factors.push_back("50 %");
    zoom_factors.push_back("75 %");
    zoom_factors.push_back("100 %");
    zoom_factors.push_back("150 %");
    zoom_factors.push_back("200 %");
  }

  Gtk::VBox *mainbox = manage(new Gtk::VBox());
  add(*mainbox);

  toolbar->set_orientation(Gtk::ORIENTATION_VERTICAL);
  toolbar->set_toolbar_style(Gtk::TOOLBAR_ICONS);

  Gtk::HBox *hbox = manage(new Gtk::HBox(false, 0));
  hbox->pack_start(*toolbar, Gtk::PACK_SHRINK, 2);

  Gtk::Table *table = manage(new Gtk::Table(2, 2, false));
  hbox->pack_end(*table, Gtk::PACK_EXPAND_WIDGET, 0);

  mainbox->pack_start(*menubar, Gtk::PACK_SHRINK);
  mainbox->pack_start(*hbox);

  scroller = manage(new Gtk::ScrolledWindow());
  table->attach(*scroller, 1, 2, 1, 2, 
		Gtk::EXPAND | Gtk::FILL, Gtk::EXPAND | Gtk::FILL);

  scroller->add(document_view);
  scroller->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

  {
    using namespace Gtk::Toolbar_Helpers;
    ToolList &tools = toolbar->tools();

    tools.push_back(StockElem(Gtk::Stock::SAVE,
			      safeslot(*this, &Frame_Window::save),
			      "Save file"));
    save_button = static_cast<Gtk::Button*>(tools.back().get_widget());
    tools.push_back(Space());
    tools.push_back(StockElem(Gtk::Stock::PRINT_PREVIEW,
			      safeslot(*this, &Frame_Window::print_to_viewer),
			      "Print to external viewer"));
    preview_button = static_cast<Gtk::Button*>(tools.back().get_widget());
    tools.push_back(Space());
    tools.push_back(ButtonElem("Text frame", 
			       icons->new_frame, 
			       safeslot(document_view, 
					&Document_View::new_text_frame),
			       "Create new text frame"));
    text_frame_button = static_cast<Gtk::Button*>(tools.back().get_widget());
    tools.push_back(ButtonElem("Image",
			       icons->moose, 
			       safeslot(*import_dialog, &Filesel::show_all),
			       "Import image file"));
    image_frame_button = static_cast<Gtk::Button*>(tools.back().get_widget());
    tools.push_back(Space());
    tools.push_back(StockElem(Gtk::Stock::PROPERTIES,
			      safeslot(*properties_dialog, 
				       &PropertiesDialog::show_raise),
			       "Show object properties"));
    properties_button = static_cast<Gtk::Button*>(tools.back().get_widget());
    tools.push_back(ButtonElem("Streams", 
			       icons->streams, 
			       safeslot(*stream_dialog, 
					&Stream_Dialog::show_raise),
			       "Show list of text streams"));
    tools.push_back(Space());
    tools.push_back(ToggleElem("Snap to grid", 
			       icons->grid, 
			       safeslot(*this,
					&Frame_Window::on_grid_button_clicked),
			       "Align objects with grid"));
    grid_button = 
      static_cast<Gtk::ToggleButton*>(tools.back().get_widget());
    tools.push_back(Space());
    tools.push_back(StockElem(Gtk::Stock::GOTO_TOP,
			      bind(slot(document_view, 
					&Document_View::rearrange_selected), 
				   TOP),
			      "Move object(s) to top"));

    tools.push_back(StockElem(Gtk::Stock::GO_UP,
			      bind(slot(document_view, 
					&Document_View::rearrange_selected), 
				   UP),
		    "Move object(s) up"));

    tools.push_back(StockElem(Gtk::Stock::GO_DOWN,
			      bind(slot(document_view, 
					&Document_View::rearrange_selected), 
				   DOWN),
		    "Move object(s) down"));

    tools.push_back(StockElem(Gtk::Stock::GOTO_BOTTOM,
			      bind(slot(document_view, 
					&Document_View::rearrange_selected), 
				   BOTTOM),
		    "Move object(s) to bottom"));

    streams_button = static_cast<Gtk::Button*>(tools.back().get_widget());
  }
  
  cafe_opera = manage(new Gtk::Statusbar());
  cafe_opera->set_has_resize_grip(false);
  cafe_opera->property_spacing() = double_space;
  
  cafe_opera->pack_end(*pagesel, Gtk::PACK_SHRINK, 0);
  Gtk::Label *page_label = manage(new Gtk::Label("P_age:", true));
  page_label->set_mnemonic_widget(pagesel->get_optionmenu());
  cafe_opera->pack_end(*page_label,
		       Gtk::PACK_SHRINK, 0);
  zoom_factor = manage(new Gtk::OptionMenu());
  cafe_opera->pack_end(*zoom_factor, Gtk::PACK_SHRINK, 0);
  Gtk::Label *zoom_label = manage(new Gtk::Label("_Zoom:", true));
  zoom_label->set_mnemonic_widget(*zoom_factor);
  cafe_opera->pack_end(*zoom_label, 
		       Gtk::PACK_SHRINK, 0);
  
  mainbox->pack_end(*cafe_opera, Gtk::PACK_SHRINK);
  
  document_view.zoom_change_signal.connect
    (safeslot(*this, &Frame_Window::zoom_change_action));
  zoom_factor->set_menu(*create_zoom_menu());
  zoom_factor->get_menu()->signal_selection_done().connect
    (safeslot(*this, &Frame_Window::zoom_factor_changed_action));
  
  open_dialog->signal_hide().connect
    (safeslot(*this, &Frame_Window::open_dialog_done));
  save_dialog->signal_hide().connect
    (safeslot(*this, &Frame_Window::save_dialog_done));
  import_dialog->signal_hide().connect
    (safeslot(*this, &Frame_Window::import_dialog_done));

  document_view.document_set_signal.connect
    (safeslot(*this, &Frame_Window::on_document_changed));

  Document::undo_changed_signal.connect
    (safeslot(*this, &Frame_Window::on_document_updated));
  Document::changed_signal.connect
    (safeslot(*this, &Frame_Window::on_document_updated));
  Document::filename_changed_signal.connect
    (safeslot(*this, &Frame_Window::on_document_filename_changed));

  if(!filename.empty()) {
    // If we don't specify false, it will try to find a template in filename:
    document_view.set_document(new Document(filename, false)); 
  } else 
    document_view.set_document(document);
  
  set_filename(basename(filename));

  // put window in global list
  windows.insert(this);

  show_all();

#ifndef ENABLE_UNDO
  undo_item->hide();
  redo_item->hide();
#endif

  active_window = this;

  // Make sure not everything is visible:
  on_document_updated(document_view.get_document()); 
}

Frame_Window::~Frame_Window() {
  // remove window from list
  windows.erase(this);

  // delete non-widgets
  delete icons;

  // delete dialogs
  delete open_dialog; delete save_dialog; 
  delete import_dialog; delete print_dialog;

  try { unlink(preview_tmp_file); }
  catch(const std::exception& err) {
    warning << "Failed to clean up: " << err.what() << std::endl;
  }
  
  // set active window
  active_window = 0; // don't know which one it'll be

  // delete utility windows
  if(windows.empty()) {
    delete doc_props_dialog; delete properties_dialog;
    delete stream_dialog; delete about_dialog; delete undo_dialog;
    Gtk::Main::quit();
  } else
    debug << "window count = " << windows.size() << std::endl;
}

void Frame_Window::on_grid_button_clicked() {
  document_view.set_snap(grid_button->get_active() ? snap::GRID : snap::NONE);
}

void Frame_Window::do_nothing() const {
  Error_Dialog::view("Not implemented yet.\nSorry.");
}

void Frame_Window::set_filename(const Glib::ustring &filename) {
  if(!filename.empty()) {
    set_title(basename(filename) + " - Passepartout");
  } else
    set_title("Passepartout");
}

void Frame_Window::zoom_change_action(float factor) {
  if(!zoom_lock) {
    Gtk::Menu_Helpers::MenuList::iterator i;
    int j;
    for(i = zoom_factor->get_menu()->items().begin(), j=0;
	i != zoom_factor->get_menu()->items().end();
	i++, j++)
      {
	if(str2float(dynamic_cast<Gtk::Label&>
		     (*(i->get_child())).get_text()) == factor) {
	  zoom_factor->get_menu()->set_active(j);
	  return;
	} else if(str2float(dynamic_cast<Gtk::Label&>
			    (*(i->get_child())).get_text())<factor) {
	  warning << "Zoom factor " << factor 
		  << " is not defined" << std::endl;
	}
      }
  }
}

void Frame_Window::zoom_factor_changed_action() {
  zoom_lock = true;
  // GtkOtptionMenu (GTK+) behaves in a rather perverse manner:
  // The child of the active MenuItem is reparented to
  // reside within the actual button instead of inside the
  // MenuItem. I went a little crazy before I read the GTK+ FAQ.
  // /Fredrik
  if(Gtk::Label* label = dynamic_cast<Gtk::Label*>(zoom_factor->get_child()))
    document_view.set_zoom_factor(str2float(label->get_text()));
  else ;//run around in circles and scream!

  zoom_lock = false;
  scroller->get_vadjustment()->set_value(0);
  scroller->get_hadjustment()->set_value(0);
}

Gtk::Menu* Frame_Window::create_zoom_menu() {
  using namespace Gtk;
  using namespace Menu_Helpers;

  Menu *menu = manage(new Menu());
  MenuList& menu_list = menu->items();
  for(ZoomFactors::const_iterator i = zoom_factors.begin();
      i != zoom_factors.end();  ++i)
    menu_list.push_back(MenuElem(*i));

  return menu;
}

void Frame_Window::on_document_updated(Document *document_) {
  Document *document = document_view.get_document();
  if(!delete_page_item || document != document_)
    return;
  on_document_changed();
}

void Frame_Window::on_document_filename_changed(Document *document_) {
  Document *document = document_view.get_document();
  if(document_ == document)
    set_filename(document ? document->get_filename() : "");
}

void Frame_Window::on_document_changed() {
  Document *document = document_view.get_document();
  // enable/disable stuff
  bool on = document;
  bool have_pages = on && document->get_num_of_pages() > 0;
  if(!(save_item && save_as_item && print_item && preview_item 
       && props_item && prefs_item
       && save_button && text_frame_button && image_frame_button 
       && preview_button
       && page_menu && edit_menu)) {
    warning << "Some menu stuff is NULL" << std::endl;
    return;
  }
  save_item->set_sensitive(on);
  save_button->set_sensitive(on);
  save_as_item->set_sensitive(on);
  print_item->set_sensitive(have_pages);
  preview_item->set_sensitive(have_pages);
  preview_button->set_sensitive(have_pages);
  props_item->set_sensitive(on);

  text_frame_button->set_sensitive(have_pages);
  image_frame_button->set_sensitive(have_pages);
  toolbar->set_sensitive(on);
  pagesel->set_sensitive(have_pages);
  zoom_factor->set_sensitive(on);

  Gtk::Menu_Helpers::MenuList::iterator i;
  for(i = edit_menu->items().begin(); i != edit_menu->items().end(); i++)
    i->set_sensitive(on);
  // but show preferences anyway
  prefs_item->set_sensitive(true);

  for(i = page_menu->items().begin(); i != page_menu->items().end(); i++)
    i->set_sensitive(on);
  delete_page_item->set_sensitive(have_pages);

  undo_item->set_sensitive(on && document->get_undo_count());
  redo_item->set_sensitive(on && document->get_redo_count());
 
  on_document_filename_changed(document);
}

void Frame_Window::new_document() {
  doc_props_dialog->create(&document_view);
}

void Frame_Window::create_menus() {
  using namespace Gtk;
  using namespace Menu_Helpers;
  using SigC::bind;

  /// *** file menu ***

  Menu *file_menu = manage(new Menu());
  MenuList& list_file = file_menu->items();
  list_file.push_back
    (ImageMenuElem("_New ...", AccelKey("<control>N"),
		   *manage(new Gtk::Image(Gtk::Stock::NEW, 
					  Gtk::ICON_SIZE_MENU)),
		   safeslot(*this, &Frame_Window::new_document)));
  list_file.push_back
    (ImageMenuElem("_Open ...", AccelKey("<control>O"),
		   *manage(new Gtk::Image(Gtk::Stock::OPEN, 
					  Gtk::ICON_SIZE_MENU)),
		   safeslot(*open_dialog, &Filesel::show_all)));
  list_file.push_back(SeparatorElem());
  list_file.push_back(StockMenuElem(Gtk::Stock::SAVE,
				    safeslot(*this, &Frame_Window::save)));
  save_item = &list_file.back();
  list_file.push_back
    (ImageMenuElem("Save _As ...", AccelKey("<shift><control>S"),
		   *manage(new Gtk::Image(Gtk::Stock::PRINT, 
					  Gtk::ICON_SIZE_MENU)),
		   safeslot(*save_dialog, 
			    &Filesel::show_all)));
  save_as_item = &list_file.back();
  list_file.push_back(SeparatorElem());
  list_file.push_back
    (ImageMenuElem("_Print ...", AccelKey("<control>P"),
		   *manage(new Gtk::Image(Gtk::Stock::PRINT, 
					  Gtk::ICON_SIZE_MENU)),
		   safeslot(*print_dialog, 
			    &Print_Dialog::show_it)));
  print_item = &list_file.back();
  list_file.push_back(StockMenuElem(Gtk::Stock::PRINT_PREVIEW,
				    AccelKey("<shift><control>P"),
				    safeslot(*this, 
					     &Frame_Window::print_to_viewer)));
  preview_item = &list_file.back();
  list_file.push_back(SeparatorElem());
  list_file.push_back
    (MenuElem("P_roperties ...",
	      safeslot(*this, 				   
		       &Frame_Window::show_doc_props_dialog)));
  props_item = &list_file.back();
  list_file.push_back(SeparatorElem());
  list_file.push_back(StockMenuElem(Gtk::Stock::CLOSE,
				    safeslot(*this, &Frame_Window::close)));
  list_file.push_back(StockMenuElem(Gtk::Stock::QUIT,
				    SigC::slot(&Frame_Window::quit)));
  menubar->items().push_back(MenuElem("_File", AccelKey("<alt>f"), 
				      *file_menu));

  /// *** edit menu ***

  edit_menu = manage(new Menu());
  MenuList& list_edit = edit_menu->items();
  list_edit.push_back(StockMenuElem(Gtk::Stock::UNDO, 
				    AccelKey("<control>Z"),
				    safeslot(document_view, 
					     &Document_View::undo)));
  undo_item = &list_edit.back();
  list_edit.push_back(StockMenuElem(Gtk::Stock::REDO,
				    AccelKey("<shift><control>Z"),
				    safeslot(document_view, 
					     &Document_View::redo)));
  redo_item = &list_edit.back();
#ifdef ENABLE_UNDO
  list_edit.push_back(SeparatorElem());
#endif
  list_edit.push_back(StockMenuElem(Gtk::Stock::COPY,
			       safeslot(document_view, 
					&Document_View::copy)));
  list_edit.push_back(StockMenuElem(Gtk::Stock::CUT,
			       safeslot(document_view, 
					&Document_View::cut)));
  list_edit.push_back(StockMenuElem(Gtk::Stock::PASTE,
			       safeslot(document_view, 
					&Document_View::paste)));
  list_edit.push_back
    (StockMenuElem(Gtk::Stock::DELETE,
		   AccelKey(guint(GDK_Delete), 
			    Gdk::ModifierType(0)),
		   safeslot(document_view, 
			    &Document_View::delete_selected)));

  list_edit.push_back(SeparatorElem());
  list_edit.push_back(MenuElem("Select _All", AccelKey("<control>A"),
				safeslot(document_view, 
					 &Document_View::select_all_frames)));
  list_edit.push_back
    (MenuElem("Dese_lect All",	AccelKey("<shift><control>A"),
	      safeslot(document_view, 
		       &Document_View::unselect_all_frames)));
  list_edit.push_back(SeparatorElem());
  // *** arrange submenu ***
  Menu *arrange_menu = manage(new Menu());
  MenuList& list_arrange = arrange_menu->items();
  list_arrange.push_back(MenuElem("_Group",
				  safeslot(document_view, 
					   &Document_View::group_selected)));
  list_arrange.push_back
    (MenuElem("_Ungroup",
	      safeslot(document_view, 
		       &Document_View::ungroup_selected)));
  list_arrange.push_back(SeparatorElem());
  list_arrange.push_back
    (MenuElem("Move to _top",
	      bind(slot(document_view, 
			&Document_View::rearrange_selected), TOP)));
  list_arrange.push_back
    (MenuElem("Move _up",
	      bind(slot(document_view, 
			&Document_View::rearrange_selected), UP)));
  list_arrange.push_back
    (MenuElem("Move _down",
	      bind(slot(document_view, 
			&Document_View::rearrange_selected), DOWN)));
  list_arrange.push_back
    (MenuElem("Move to _bottom",
	      bind(slot(document_view, 
			&Document_View::rearrange_selected), BOTTOM)));
  list_edit.push_back(MenuElem("A_rrange", *arrange_menu));
  list_edit.push_back(SeparatorElem());
  list_edit.push_back(MenuElem("Insert Te_xt Frame", 
				safeslot(document_view,
					 &Document_View::new_text_frame)));
  list_edit.push_back(MenuElem("Insert _Image ...", 
				safeslot(*import_dialog, &Filesel::show_all)));
  list_edit.push_back(SeparatorElem());
  list_edit.push_back(MenuElem("Pre_ferences ...", 
			       safeslot(config, &P_File::dialog_show)));
  prefs_item = &list_edit.back();
  menubar->items().push_back(MenuElem("_Edit", AccelKey("<alt>e"), 
				      *edit_menu));


  // *** page menu ***

  page_menu = manage(new Menu());
  MenuList& list_page = page_menu->items();
  list_page.push_back(MenuElem("_Delete", 
			       safeslot(document_view, 
					&Document_View::delete_page)));
  list_page.push_back(SeparatorElem());
  list_page.push_back(MenuElem("Insert _Before ...", 
			       safeslot(document_view, 
					&Document_View::insert_page_before)));
  list_page.push_back(MenuElem("Insert _After ...", 
			       safeslot(document_view, 
					&Document_View::insert_page_after)));
  delete_page_item = &list_page.back();
  //  list_page.push_back(SeparatorElem());
  //  list_page.push_back(MenuElem("_Template Page ...", safeslot(this, &Frame_Window::show_template_page_dialog)));
  // it is not yet possible to change the template of an existing page
  menubar->items().push_back(MenuElem("_Page", AccelKey("<alt>p"),*page_menu));


  // *** view menu ***

  Menu *view_menu = manage(new Menu());
  MenuList& list_view = view_menu->items();
  list_view.push_back
    (CheckMenuElem("_Toolbar", 
		   safeslot(*this, 
			    &Frame_Window::toggle_toolbar)));
  static_cast<CheckMenuItem*>(&list_view.back())->set_active(true);
  list_view.push_back(SeparatorElem());
  list_view.push_back
    (StockMenuElem(Gtk::Stock::PROPERTIES,
		   safeslot(*properties_dialog, 
			    &PropertiesDialog::show_raise)));
  list_view.push_back(MenuElem("_Streams", 
			       safeslot(*stream_dialog, 
					&Stream_Dialog::show_raise)));
#ifdef ENABLE_UNDO
  list_view.push_back(MenuElem("_Undo history", 
			       safeslot(*undo_dialog, 
					&Undo_Dialog::show_raise)));
#endif
  menubar->items().push_back(MenuElem("_View", 
				      AccelKey("<alt>v"), 
				      *view_menu));


  // *** help menu ***

  Menu *help_menu = manage(new Menu());
  MenuList& list_help = help_menu->items();
  list_help.push_back(MenuElem("_User's guide", AccelKey("F1"),
			       SigC::slot(&open_docs)));
  list_help.push_back(MenuElem("_About", 
			       safeslot(*about_dialog, 
					&About_Dialog::show_all)));
  list_help.push_back(SeparatorElem());
  list_help.push_back(MenuElem("_Homepage", 
			       SigC::slot(&open_homepage)));
  list_help.push_back(MenuElem("_Inspiration", 
			       safeslot(*inspiration, 
					&Inspiration::show_all)));
  menubar->items().push_back(MenuElem("_Help", AccelKey("<alt>h"), 
				      *help_menu));
  menubar->items().back().set_right_justified();
}

void Frame_Window::show_doc_props_dialog() {
  doc_props_dialog->modify(&document_view);
}

void Frame_Window::open_dialog_done() {
  if(!open_dialog->was_cancelled()) {
    try {
      if(document_view.get_document())
	new Frame_Window(open_dialog->get_filename());
      else {
	document_view.set_document
	  (new Document(open_dialog->get_filename(), false));
	set_filename(basename(open_dialog->get_filename()));
      }
    } catch(const Error::Read &e) {
      Error_Dialog::view(e.message);
    }
  }
}

void Frame_Window::import_dialog_done() {
  if(!import_dialog->was_cancelled())
    document_view.new_image_frame
      (import_dialog->get_filename(),
       dynamic_cast<Image_Filesel*>(import_dialog)->get_res());
}

void Frame_Window::save_dialog_done() {
  Document *document = document_view.get_document();

  if(!save_dialog->was_cancelled() && document) {
    document->set_filename(save_dialog->get_filename());
    try {
      document->save();
    } catch (Error::Write e) {
      Error_Dialog::view(e.message);
    }
  }
}

void Frame_Window::close() {
  Document *document = document_view.get_document();
  if(windows.size() > 1 || !document) {
    windows.erase(this);
    delete this;
  } else {
    document_view.set_document(0);
    properties_dialog->set_document(0);
    stream_dialog->set_document(0);
    undo_dialog->set_document(0);
    delete document;
  }
}

void Frame_Window::save() {
  Document *document = document_view.get_document();
  
  if(document && !document->get_filename().empty()) {
    try {
      document->save();
    } catch (Error::Write e) {
      Error_Dialog::view(e.message);
    }
  } else
    save_dialog->show();
}

void Frame_Window::toggle_toolbar() {
  if(toolbar->is_visible())
    toolbar->hide();
  else
    toolbar->show();
}

void Frame_Window::toggle_cafe_opera() {
  if(cafe_opera->is_visible())
    cafe_opera->hide();
  else
    cafe_opera->show();
}

bool Frame_Window::on_delete_event(GdkEventAny *event) {
  close();
  return true;
}

bool Frame_Window::on_focus_in_event(GdkEventFocus *event) {
  // the nonmodal windows are updated to reflect the content in the window
  // that is currently in focus
  
  active_window = this;
  Document *document = document_view.get_document();
  if(properties_dialog) {
      properties_dialog->set_transient_for(*this);
      properties_dialog->set_document(document);
    }
  if(stream_dialog) {
      stream_dialog->set_transient_for(*this);
      stream_dialog->set_document(document);
    }
  if(undo_dialog) {
      undo_dialog->set_transient_for(*this);
      undo_dialog->set_document(document);
    }
  return false;
}

bool Frame_Window::on_key_press_event(GdkEventKey *k) {
  Window::on_key_press_event(k);
  document_view.on_key_press_event(k);
  return false;
}

void Frame_Window::print_to_viewer() const {
  const Document *document = document_view.get_document();

  // try to run gv (or whatever you want to use instead)
  if(document) {
    try {
      const std::string &psviewer = config.PSViewer.values.front();
      if(!psviewer.empty()) {
	std::ofstream out(preview_tmp_file.c_str());
	if(!out) {
	  Error_Dialog::view("Could not print to file:\n"
			     + preview_tmp_file);
	  return;
	}
	// print all pages:
	document->print(out, 
			document->get_first_page_num(), 
			document->get_first_page_num() 
			+ document->get_num_of_pages() - 1);
	verbose << psviewer + " " + preview_tmp_file << std::endl;
	try {
	  Glib::spawn_command_line_async(psviewer + " " + preview_tmp_file);
	} catch(Glib::SpawnError e) {
	  Error_Dialog::view("Could not run: " + psviewer + "\n\n"
			     + e.what());
	  return;
	}
      } else
	Error_Dialog::view("No PostScript viewer has been defined");
    } 
    catch(Error::Print &e) {
      Error_Dialog::view(e.message);
    } 
    catch(std::exception &e) {
      Error_Dialog::view("Failed to print\n\n" + std::string(e.what()));
    }
  }
}

void Frame_Window::quit() {
  // make sure all destructors are executed
  for(Windows::iterator i = windows.begin(); i != windows.end(); i++)
    delete *i;

  Gtk::Main::quit();
}
