#ifndef CONFIGFILE_H		// -*- c++ -*-
#define CONFIGFILE_H
#include <string>
#include <vector>
#include <iostream>

/*
A configuration file parser.
Each statement can only be one line of the format:
name = value
The variable name may contain any character except '=' and newline.
All variables must be declared, so that the parser knows what format
to expect the value to be in. Actually, the parsing of the value is handled
by the declared variable object itself, making it easy to add new types.

Existing types are lists of booleans, strings and floats. The values in a list
are separated by whitespace. Strings may be quoted with '"'. The '"' character
may be escaped (in strings only) with '\"'.
 */

namespace Config {

struct Var {
  std::string name;
  Var(std::string name_): name(name_) {}
  virtual ~Var() {}
  virtual void print(std::ostream &out) const=0;
  virtual void parse(const std::string &input)=0;
};

//  std::ostream& operator<<(std::ostream &out, const Var &var)
//  {
//    var.print(out);
//    return out;
//  }

typedef std::vector<bool> Bools;

struct Bool_Var: public Var {
  Bools values;
  Bool_Var(std::string name_): Var(name_) {}
  virtual ~Bool_Var() {}
  void print(std::ostream &out) const;
  void parse(const std::string &input);
};

typedef std::vector<std::string> Strings;

struct String_Var: public Var {
  Strings values;
  String_Var(std::string name_): Var(name_) {}
  virtual ~String_Var() {}
  void print(std::ostream &out) const;
  void parse(const std::string &input);
};

typedef std::vector<float> Floats;

struct Float_Var: public Var {
  Floats values;
  Float_Var(std::string name_): Var(name_) {}
  virtual ~Float_Var() {}
  void print(std::ostream &out) const;
  void parse(const std::string &input);
};

typedef std::vector<Var*> Vars;

class File {
public:
  File(): ignore_unknown(false) {}
  virtual ~File() {}
  void set_filename(std::string filename);
  void declare_var(Var *var);
  void read(const std::string &filename);
  void write(const std::string &filename);
protected:
  bool ignore_unknown; // don't complain about unknown variables
  // feel free to override this method:
  virtual void error(const std::string &message)
  {std::cerr << message << std::endl;}
  Vars vars;
};

}

#endif
