#include "typesetter.hh"
#include <libxml++/libxml++.h>
#include <util/stringutil.h>
#include "line.hh"
#include "paragraph.hh"
#include "pscanvas.hh"
#include "pdfcanvas.hh"
#ifdef XCANVAS
#include "xcanvas.hh"
#endif
#include "blockcontainer.hh"

namespace {
  class Parser_Error: public xmlpp::exception
  {
  public:
    Parser_Error(std::string s) : xmlpp::exception(s) {}
    virtual ~Parser_Error() throw () {};
    virtual void Raise() const {throw *this;}
    virtual Parser_Error * Clone() const {return new Parser_Error(*this);}
  };
}

xml2ps::PsConverter::PsConverter(xml2ps::Canvas& output)
  : out(output), root(out), node(&root)
{}

xml2ps::PsConverter::~PsConverter() {}

void xml2ps::PsConverter::on_start_element(const std::string& n, 
				   const xmlpp::SaxParser::AttributeMap &p) {
      // remember which element we are in, for better error messages:
      element_stack.push_back(n);
      bool unknown = false;
      try {
	xml2ps::Element* newnode;
	if (n == "para") {
	  newnode = new xml2ps::Paragraph(out, *node, n, p);
	} else if (n == "font") {
	  newnode = new xml2ps::TextContainer(*node, n, p);
	} else if (n == "bp") {
	  newnode = new xml2ps::BreakPoint(*node);
	} else if (n == "leader") {
	  newnode = new xml2ps::LeaderNode(*node, p);
	} else if (n == "obstacle") {
	  newnode = new xml2ps::ObstacleNode(*node, p);
	} else if (n == "block-container") {
	  newnode = new xml2ps::BlockContainer(out, *node, n, p);
	} else if (n == "linebreak") {
	  newnode = new xml2ps::LineBreak(*node);
	} else {
	  unknown = true;
	  throw Parser_Error("element <" + n + "> is unknown");
	}
	node->add(newnode);
	node = newnode;
      } catch(const std::exception& err) {
	if(unknown) // don't catch what it throws itself
	  throw;
	throw(Parser_Error("open <" 
			   + n + "> - " + err.what()));
      }
    }
void xml2ps::PsConverter::on_end_element(const std::string &n) {
      try {      
	node->close();
	node = &node->getParent();
      } catch(const std::exception& err) {
	throw(Parser_Error("close <" 
			   + n + "> - " + err.what()));
      }
      if(!element_stack.empty())
	element_stack.pop_back();
    }
void xml2ps::PsConverter::on_characters(const std::string& s) {
      try {
	using xml2ps::TextContainer;
	if(TextContainer* tc = dynamic_cast<TextContainer*>(node)) {
	  tc->add(s);
	} else {
	  if(strip_whitespace(s).length() > 0) {// Todo: unicode
	    if(!element_stack.empty()) {
	      std::cerr << "ignored text inside <" 
			<< element_stack.back()
			<< ">: \"" << s << '"' << std::endl;
	    }
	  }
	}
      } catch(const std::exception& err) {
	throw(Parser_Error("parsing text inside <" + element_stack.back()
			   + "> - " + err.what()));
      }
    
    }
void xml2ps::PsConverter::on_warning(const std::string &s) {
      if(!element_stack.empty())
	std::cerr << "inside <" << element_stack.back() << "> - ";
      std::cerr << s << std::endl;
    }
void xml2ps::PsConverter::on_error(const std::string &s) {
      std::string tmp;
      if(!element_stack.empty())
	tmp = "inside <" + element_stack.back() + "> - ";
      throw(Parser_Error(tmp + s));
    }
void xml2ps::PsConverter::on_fatal_error(const std::string &s) {on_error(s);}

namespace xml2ps {
  Typesetter::~Typesetter() {
  }

  void Typesetter::addPage(const PageBoundary &page_boundary) {
    pages.push_back(page_boundary);
  }

  void Typesetter::addObstacle(Boundary obstacle) {
    if(pages.size())
      pages.back().addObstacle(obstacle);
    else {
      throw std::runtime_error("Obstacle before page boundary"
			       " is not allowed");
    }
  }

  void Typesetter::run(std::istream &input, std::ostream &output) {
    xml2ps::Canvas *out;
    switch(output_format) {
    case FORMAT_PS:
      out = new xml2ps::PsCanvas(pages, extra_pages);
      break;
    case FORMAT_PDF:
      out = new xml2ps::PDFCanvas(output.rdbuf(), 
				  pages, extra_pages);
      break;
#ifdef XCANVAS
    case FORMAT_X:
      out = new xml2ps::XCanvas(pages, extra_pages);
      break;
#endif
    default:
      throw std::runtime_error("Uncaught output format");
      break;
    }
    try {
      out->setSubstituteFontAliases(subst_font_aliases);
      PsConverter parser(*out);
      parser.parse_stream(input);
      used_fonts = out->getUsedFonts();
      if(output_format == FORMAT_PS) {
	// must flush before merge // Todo: fix this
	dynamic_cast<PsCanvas*>(out)->flush();
	dynamic_cast<PsCanvas*>(out)->merge(output);
      }
      
      delete out;
    } catch (...) {
      delete out;
      throw;
    }
  }
};
