/*
 * interdiff - create incremental patch between two against a common source
 * combinediff - create cumulative patch from two incremental patches
 * Copyright (C) 2000, 2001, 2002 Tim Waugh <twaugh@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#ifdef HAVE_ERROR_H
# include <error.h>
#endif /* HAVE_ERROR_H */
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <getopt.h>
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif /* HAVE_SYS_WAIT_H */

#include "util.h"
#include "diff.h"

#ifndef DIFF
#define DIFF "diff"
#endif

#ifndef PATCH
#define PATCH "patch"
#endif

/* This can be invoked as interdiff or combinediff. */
static int combining;

struct file_list 
{
	char *file;
	long pos;
	struct file_list *next;
};

struct lines {
	char *line;
	unsigned long n;
	struct lines *next;
	struct lines *prev;
};

struct lines_info {
	char *unline;
	struct lines *head;
	struct lines *tail;
};

static int human_readable = 1;
static char diff_opts[4];
static int max_context_real = 3, max_context = 3;
static int ignore_components = 0;
static int unzip = 0;
static int debug = 0;

static struct patlist *pat_drop_context = NULL;

static struct file_list *files_done = NULL;
static struct file_list *files_in_patch2 = NULL;

/* checks whether file needs processed and sets context */
static int check_filename (const char *fn)
{
	if (patlist_match(pat_drop_context, fn)) {
		max_context = 0;
	} else {
		max_context = max_context_real;
	}
	return 1;
}

static void add_to_list (struct file_list **list, const char *file, long pos)
{
	struct file_list *make;
	make = xmalloc (sizeof *make);
	make->next = *list;
	make->file = xstrdup (file);
	make->pos = pos;
	*list = make;
}

static const char *stripped (const char *name)
{
	const char *basename;
	int i = 0;
	if (!strncmp (name, "./", 2))
		name += 2;
	basename = strrchr (name, '/');
	if (!basename)
		basename = name;
	while (i < ignore_components &&
	       (name = strchr (name, '/')) != NULL) {
		while (*name == '/')
			name++;
		i++;
	}
	return name ? name : basename;
}

static long file_in_list (struct file_list *list, const char *file)
{
	file = stripped (file);
	while (list) {
		if (!strcmp (stripped (list->file), file))
			return list->pos;
		list = list->next;
	}
	return -1;
}

static void free_list (struct file_list *list)
{
	struct file_list *next;
	for (; list; list = next) {
		next = list->next;
		free (list->file);
		free (list);
	}
}

static void insert_line_before (struct lines_info *lines,
				struct lines *at, struct lines *make)
{
	make->next = at;
	make->prev = at->prev;
	if (at->prev)
		at->prev->next = make;
	else
		lines->head = make;
	at->prev = make;
}

static int add_line (struct lines_info *lines, const char *line,
		     unsigned long n)
{
	struct lines *make;
	struct lines *at;
	
	make = xmalloc (sizeof *make);
	make->n = n;
	make->line = xstrdup (line);

	if (!lines->tail) {
		/* New */
		make->prev = NULL;
		make->next = NULL;
		lines->head = make;
		lines->tail = make;
		return 0;
	}
	if (lines->tail->n < n) {
		/* End */
		lines->tail->next = make;
		make->next = NULL;
		make->prev = lines->tail;
		lines->tail = make;
		return 0;
	}

	if (!lines->head)
		error (EXIT_FAILURE, 0, "List corrupted: no head");

	at = lines->head;
	while (at) {
		if (n < at->n) {
			insert_line_before (lines, at, make);
			return 0;
		}
		if (n == at->n) {
			free (make->line);
			free (make);
			if (!strcmp (at->line, line))
				/* Already there. */
				return 0;

			/* Clash: first one wins. */
			return 1;
		}
		make->prev = at;
		at = at->next;
	}

	if (!at)
		error (EXIT_FAILURE, 0, "List corrupted: ordering");

	return 1;
}

static void merge_lines (struct lines_info *lines1, struct lines_info *lines2)
{
	struct lines *at, *at_prev, *make;


	/* 
	 * Note: we don't care about the tail member here - not needed
	 * anymore.
	 */

	if (!lines1->head) {
		/* first list empty - only take second */
		lines1->head = lines2->head;
		return;
	}

	/* merge lines in one pass */
	at = lines1->head;
	make = lines2->head;
	while (at) {
		while (make) {
			if (make->n > at->n)
				break;

			if (make->n < at->n) {
				struct lines *next = make->next;

				insert_line_before (lines1, at, make);
				make = next;
				continue;
			}

			make = make->next; /* line number equal - first wins */
		}
		at_prev = at;
		at = at->next;
	}

	/* append the rest of list 2 to list 1 */
	if (make)
		at_prev->next = make;
}

static void free_lines (struct lines *list)
{
	struct lines *next;
	for (; list; list = next) {
		next = list->next;
		free (list->line);
		free (list);
	}
}

static struct lines *create_orig (FILE *f, struct lines_info *file,
				  int reverted)
{
	unsigned long linenum;
	char *line = NULL, first_char;
	size_t linelen = 0;
	int last_was_add;
	long pos = ftell (f);

	do {
		if (getline (&line, &linelen, f) == -1)
			break;
	} while (strncmp (line, "@@ ", 3));

	while (!feof (f)) {
		/* Find next hunk */
		unsigned long orig_lines, new_lines, newline;
		int file_is_removed = 0;
		char *p, *q;

		if (strncmp (line, "@@", 2)) {
			fseek (f, pos, SEEK_SET);
			break;
		}

		if (reverted) {
			new_lines = orig_num_lines (line);
			orig_lines = new_num_lines (line);
		} else {
			orig_lines = orig_num_lines (line);
			new_lines = new_num_lines (line);
		}

		p = strchr (line, reverted ? '+' : '-');
		if (!p)
			break;

		p++;
		linenum = strtoul (p, &q, 10);
		if (p == q)
			break;
		if (linenum == 0)
			file_is_removed = 1;

		/* Now copy the relevant bits of the hunk */
		last_was_add = 0;
		newline = 1;
		while (orig_lines || new_lines || newline) {
			pos = ftell (f);
			if (getline (&line, &linelen, f) == -1)
				break;

			if (!orig_lines && !new_lines &&
			    line[0] != '\\')
				break;

			first_char = line[0];
			if (reverted) {
				if (first_char == '-')
					first_char = '+';
				else if (first_char == '+')
					first_char = '-';
			}

			switch (first_char) {
			case ' ':
				if (new_lines) new_lines--;
			case '-':
				if (orig_lines) orig_lines--;
				if (!file_is_removed)
					add_line (file, line + 1, linenum++);
				break;
			case '+':
				if (new_lines) new_lines--;
				break;
			case '\\':
				newline = 0;
				if (!file_is_removed && !last_was_add) {
					struct lines *at, *prev = NULL;
					char *p;
					for (at = file->head; at; at = at->next)
						prev = at;
					if (!prev)
						error (EXIT_FAILURE, 0,
						       "Garbled patch");
					p = prev->line;
					p += strlen (p) - 1;
					if (*p == '\n')
						*p = '\0';
				}
				break;
			}

			last_was_add = (line[0] == '+');
		}

		if (!newline) {
			pos = ftell (f);
			if (getline (&line, &linelen, f) == -1)
				break;
		}
	}

	if (line)
		free (line);

	return file->head;
}

static int output_patch1_only (FILE *p1, int not_reverted)
{
	char *line;
	char *oldname;
	char first_char;
	size_t linelen;
	long pos;

	oldname = NULL;
	linelen = 0;
	if (getline (&oldname, &linelen, p1) < 0)
		error (EXIT_FAILURE, errno, "Bad patch #1");

	if (strncmp (oldname, "--- ", 4))
		error (EXIT_FAILURE, 0, "Bad patch #1");

	line = NULL;
	linelen = 0;
	if (getline (&line, &linelen, p1) < 0)
		error (EXIT_FAILURE, errno, "Bad patch #1");

	if (strncmp (line, "+++ ", 4))
		error (EXIT_FAILURE, 0, "Bad patch #1");

	if (not_reverted) {
		/* Combinediff: copy patch */
		if (human_readable)
			printf ("unchanged:\n");
		fputs (oldname, stdout);
		fputs (line, stdout);
	} else {
		if (human_readable)
			printf ("reverted:\n");
		printf ("--- %s", line + 4);
		printf ("+++ %s", oldname + 4);
	}
	free (oldname);

	pos = ftell (p1);
	if (getline (&line, &linelen, p1) < 0)
		error (EXIT_FAILURE, errno, "Bad patch #1");

	for (;;) {
		unsigned long orig_lines;
		unsigned long new_lines;
		unsigned long newline;
		char *d1, *d2, *p;
		size_t h;

		if (strncmp (line, "@@ ", 3)) {
			fseek (p1, pos, SEEK_SET);
			break;
		}

		p = line + 3;
		h = strcspn (p, " \t");
		d1 = xstrndup (p, h);
		p += h;
		p += strspn (p, " \t");
		h = strcspn (p, " \t");
		d2 = xstrndup (p, h);
		if (!*d1 || !*d2)
			error (EXIT_FAILURE, 0, "Bad patch #1");

		if (not_reverted) {
			/* Combinediff: copy patch */
			fputs (line, stdout);
			new_lines = orig_num_lines (d1);
			orig_lines = new_num_lines (d2);
		} else {
			/* Interdiff: revert patch */
			printf ("@@ -%s +%s @@\n", d2 + 1, d1 + 1);
			orig_lines = orig_num_lines (d1);
			new_lines = new_num_lines (d2);
		}
		

		free (d1);
		free (d2);

		newline = 1;
		while (orig_lines || new_lines || newline) {
			pos = ftell (p1);
			if (getline (&line, &linelen, p1) == -1)
				break;

			if (!orig_lines && !new_lines &&
			    line[0] != '\\')
				break;

			first_char = line[0];
			if (not_reverted) {
				if (first_char == '+')
					first_char = '-';
				else if (first_char == '-')
					first_char = '+';
			}

			switch (first_char) {
			case ' ':
				if (orig_lines) orig_lines--;
				if (new_lines) new_lines--;
				fputs (line, stdout);
				break;
			case '+':
				if (new_lines) new_lines--;
				printf ("-%s", line + 1);
				break;
			case '-':
				if (orig_lines) orig_lines--;
				printf ("+%s", line + 1);
				break;
			case '\\':
				newline = 0;
				printf ("%s", line);
				break;
			}
		}

		if (!newline) {
			pos = ftell (p1);
			if (getline (&line, &linelen, p1) == -1)
				break;
		}
	}

	if (line)
		free (line);

	return 0;
}

static void construct_unline (struct lines_info *file_info)
{
	char *un;
	struct lines *at;
	size_t i;

	if (file_info->unline)
		/* Already done. */
		return;

	un = file_info->unline = xmalloc (7);

	/* First pass: construct a small line not in the file. */
	for (i = 0, at = file_info->head; at && i < 5; i++, at = at->next) {
		size_t j = strlen (at->line) - 1;
		if (i < j)
			j = i;
		un[i] = at->line[j] + 1;
		if (iscntrl (un[i]))
			un[i] = '!';
	}
	for (; i < 5; i++)
		un[i] = '!';
	un[i++] = '\n';
	un[i] = '\0';

	for (i = 4; i > 0; i--) {
		/* Is it unique yet? */
		for (at = file_info->head; at; at = at->next) {
			if (!strcmp (at->line, un))
				break;
		}
		if (!at)
			/* Yes! */
			break;

		un[i] = '\n';
		un[i + 1] = '\0';
	}

	if (i == 0) {
		/* Okay, we'll do it the hard way, and generate a long line. */
		size_t maxlength = 0;
		for (at = file_info->head; at; at = at->next) {
			size_t len = strlen (at->line);
			if (len > maxlength)
				maxlength = len;
		}

		free (un);
		un = file_info->unline = xmalloc (maxlength + 3);
		for (i = 0; i < maxlength; i++)
			un[i] = '!';
		un[i++] = '\n';
		un[i] = '\0';
	}
}

static int write_file (struct lines_info *file_info, int fd)
{
	unsigned long linenum;
	struct lines *at;
	FILE *fout = fdopen (fd, "w");

	construct_unline (file_info);
	linenum = 1;
	for (at = file_info->head; at; at = at->next) {
		unsigned long i = at->n - linenum;
		while (i--) {
			fprintf (fout, "%s", file_info->unline);
			linenum++;
		}
		fprintf (fout, "%s", at->line);
		linenum++;
	}
	fclose (fout);
	return 0;
}

static int apply_patch (FILE *patch, const char *file, int reverted)
{
	unsigned long orig_lines, new_lines;
	size_t linelen;
	char *line;
	pid_t child;
	FILE *w;

	w = xpipe(PATCH, &child, "w", PATCH,
		  reverted ? "-Rsp0" : "-sp0", NULL);

	fprintf (w, "--- %s\n+++ %s\n", file, file);
	line = NULL;
	linelen = 0;
	orig_lines = new_lines = 0;
	for (;;) {
		if (getline (&line, &linelen, patch) == -1)
			break;

		if (!orig_lines && !new_lines && !strncmp (line, "--- ", 4))
			break;

		fputs (line, w);

		if (!strncmp (line, "@@ ", 3)) {
			orig_lines = orig_num_lines (line);
			new_lines = new_num_lines (line);
			continue;
		}

		if (orig_lines && line[0] != '+')
			orig_lines--;
		if (new_lines && line[0] != '-')
			new_lines--;
	}
	fclose (w);
	waitpid (child, NULL, 0);

	if (line)
		free (line);

	return 0;
}

static int trim_context (FILE *f /* positioned at start of @@ line */,
			 const char *unline /* drop this line */)
{
	/* For each hunk, trim the context so that the number of
	 * pre-context lines does not exceed the number of
	 * post-context lines.  See the fuzz1 test case. */
	char *line = NULL;
	ssize_t linelen;

	for (;;) {
		fpos_t pos;
		unsigned long pre = 0, pre_seen = 0, post = 0, strip_post = 0;
		unsigned long orig_offset, new_offset;
		unsigned long orig_count, new_orig_count;
		unsigned long new_count, new_new_count;
		unsigned long total_count = 0;
		unsigned long trim = 0;

		/* Read @@ line. */
		if (getline (&line, &linelen, f) < 0)
			break;

		if (line[0] == '\\') {
			/* Pass '\' lines through unaltered. */
			fputs (line, stdout);
			continue;
		}

		if (read_atatline (line, &orig_offset, &orig_count,
				   &new_offset, &new_count))
			error (EXIT_FAILURE, 0, "Line not understood: %s",
			       line);

		new_orig_count = orig_count;
		new_new_count = new_count;
		fgetpos (f, &pos);
		while (orig_count || new_count) {
			if (getline (&line, &linelen, f) < 0)
				break;

			total_count++;
			switch (line[0]) {
			case ' ' :
				if (orig_count) orig_count--;
				if (new_count) new_count--;
				if (!pre_seen)
					pre++;
				else
					post++;
				if (!strcmp (line + 1, unline))
					strip_post++;
				break;
			case '-':
				if (orig_count) orig_count--;
				pre_seen = 1;
				post = 0;
				break;
			case '+':
				if (new_count) new_count--;
				pre_seen = 1;
				post = 0;
				break;
			}
		}

		if (pre > (post - strip_post)) {
			trim = pre - post + strip_post;
			orig_offset += trim;
			new_orig_count -= trim + strip_post;
			new_offset += trim;
			new_new_count -= trim + strip_post;
		}

		fsetpos (f, &pos);
		printf ("@@ -%lu", orig_offset);
		if (new_orig_count != 1)
			printf (",%lu", new_orig_count);
		printf (" +%lu", new_offset);
		if (new_new_count != 1)
			printf (",%lu", new_new_count);
		printf (" @@\n");

		while (total_count--) {
			assert (getline (&line, &linelen, f) > 0);

			if (trim) {
				trim--;
				continue;
			}

			if (strcmp (line + 1, unline))
				fputs (line, stdout);
		}
	}

	free (line);
	return 0;
}

static int output_delta (FILE *p1, FILE *p2)
{
	const char *tmpdir = getenv ("TMPDIR");
	unsigned int tmplen;
	const char tail1[] = "/interdiff-1.XXXXXX";
	const char tail2[] = "/interdiff-2.XXXXXX";
	char *tmpp1, *tmpp2;
	int tmpp1fd, tmpp2fd;
	struct lines_info file = { NULL, NULL, NULL };
	struct lines_info file2 = { NULL, NULL, NULL };
	char *oldname = NULL, *newname = NULL;
	pid_t child;
	FILE *in;
	ssize_t namelen;
	long pos1 = ftell (p1), pos2 = ftell (p2);
	long pristine1, pristine2;
	long start1, start2;
	char options[100];
	int diff_is_empty = 1;

	pristine1 = ftell (p1);
	pristine2 = ftell (p2);

	if (!tmpdir)
		tmpdir = P_tmpdir;

	tmplen = strlen (tmpdir);
	tmpp1 = alloca (tmplen + sizeof (tail1));
	memcpy (tmpp1, tmpdir, tmplen);
	memcpy (tmpp1 + tmplen, tail1, sizeof (tail1));
	tmpp2 = alloca (tmplen + sizeof (tail2));
	memcpy (tmpp2, tmpdir, tmplen);
	memcpy (tmpp2 + tmplen, tail2, sizeof (tail2));

	if (max_context == 3)
		sprintf(options, "-%su", diff_opts);
	else
		sprintf (options, "-%sU%d", diff_opts, max_context);

	tmpp1fd = xmkstemp (tmpp1);
	tmpp2fd = xmkstemp (tmpp2);

	do {
		if (oldname) {
			free (oldname);
			oldname = NULL;
		}
		if (getline (&oldname, &namelen, p1) < 0)
			error (EXIT_FAILURE, errno, "Bad patch1");

	} while (strncmp (oldname, "+++ ", 4));
	oldname[strlen (oldname) - 1] = '\0';

	do {
		if (newname) {
			free (newname);
			newname = NULL;
		}
		if (getline (&newname, &namelen, p2) < 0)
			error (EXIT_FAILURE, errno, "Bad patch #2");

	} while (strncmp (newname, "+++ ", 4));
	newname[strlen (newname) - 1] = '\0';

	start1 = ftell (p1);
	start2 = ftell (p2);
	fseek (p1, pos1, SEEK_SET);
	fseek (p2, pos2, SEEK_SET);
	create_orig (p2, &file, 0);
	fseek (p1, pos1, SEEK_SET);
	fseek (p2, pos2, SEEK_SET);
	create_orig (p1, &file2, combining);
	merge_lines(&file, &file2);
	pos1 = ftell (p1);

	/* Write it out. */
	write_file (&file, tmpp1fd);
	write_file (&file, tmpp2fd);
	free_lines (file.head);

	fseek (p1, start1, SEEK_SET);
	fseek (p2, start2, SEEK_SET);

	apply_patch (p1, tmpp1, combining);

	apply_patch (p2, tmpp2, 0);
	fseek (p1, pos1, SEEK_SET);

	fflush (NULL);

	in = xpipe(DIFF, &child, "r", DIFF, options, tmpp1, tmpp2, NULL);
	
	/* Eat the first line */
	for (;;) {
		int ch = fgetc (in);
		if (ch == EOF || ch == '\n')
			break;
		diff_is_empty = 0;
	}

	/* Eat the second line */
	for (;;) {
		int ch = fgetc (in);
		if (ch == EOF || ch == '\n')
			break;
	}

	if (!diff_is_empty) {
		/* ANOTHER temporary file!  This is to catch the case
		 * where we just don't have enough context to generate
		 * a proper interdiff. */
		FILE *tmpdiff = tmpfile ();
		char *line = NULL;
		ssize_t linelen;
		for (;;) {
			if (getline (&line, &linelen, in) < 0)
				break;
			fputs (line, tmpdiff);
			if (!strcmp (line + 1, file.unline)) {
				/* Uh-oh.  We're trying to output a
				 * line that made up (we never saw the
				 * original).  As long as this is at
				 * the end of a hunk we can safely
				 * drop it (done in trim_context
				 * later). */
				if (getline (&line, &linelen, in) < 0)
					continue;
				else if (strncmp (line, "@@ ", 3)) {
					/* An interdiff isn't possible.
					 * Evasive action: just revert the
					 * original and copy the new
					 * version. */
					fclose (tmpdiff);
					free (line);
					goto evasive_action;
				}
				fputs (line, tmpdiff);
			}
		}
		free (line);

		/* First character */
		if (human_readable) {
			char *p, *q, c, d;
			c = d = '\0'; /* shut gcc up */
			p = strchr (oldname + 4, '\t');
			if (p) {
				c = *p;
				*p = '\0';
			}
			q = strchr (newname + 4, '\t');
			if (q) {
				d = *q;
				*q = '\0';
			}
			printf (DIFF " %s %s %s\n", options, oldname + 4,
				newname + 4);
			if (p) *p = c;
			if (q) *q = d;
		}
		printf ("--- %s\n", oldname + 4);
		printf ("+++ %s\n", newname + 4);
		rewind (tmpdiff);
		trim_context (tmpdiff, file.unline);
		fclose (tmpdiff);
	}

	fclose (in);
	waitpid (child, NULL, 0);
	if (debug)
		printf ("reconstructed orig1=%s orig2=%s\n", tmpp1, tmpp2);
	else {
		unlink (tmpp1);
		unlink (tmpp2);
	}
	free (oldname);
	free (newname);
	return 0;

 evasive_action:
	if (debug)
		printf ("reconstructed orig1=%s orig2=%s\n", tmpp1, tmpp2);
	else {
		unlink (tmpp1);
		unlink (tmpp2);
	}
	if (human_readable)
		printf ("interdiff impossible; taking evasive action\n");
	fseek (p1, pristine1, SEEK_SET);
	fseek (p2, pristine2, SEEK_SET);
	output_patch1_only (p1, 0);
	output_patch1_only (p2, 1);
	return 0;
}

static int copy_residue (FILE *p2)
{
	struct file_list *at;
	char *line = NULL;
	size_t linelen = 0;
	int first;

	for (at = files_in_patch2; at; at = at->next) {

		if (file_in_list (files_done, at->file) != -1)
			continue;

		/* check if we need to process it and init context */
		if (!check_filename(at->file))
			continue;

		fseek (p2, at->pos, SEEK_SET);
		first = 1;
		if (human_readable)
			printf ("only in patch2:\n");

		do {
			if (getline (&line, &linelen, p2) == -1)
				break;
			printf ("%s", line);
		} while (strncmp (line, "--- ", 4));
		if (getline (&line, &linelen, p2) == -1)
			break;
		printf ("%s", line);
		if (getline (&line, &linelen, p2) == -1)
			break;
		for (;;) {
			unsigned long orig_lines, new_lines, newline;
			if (strncmp (line, "@@ ", 3))
				break;
			printf ("%s", line);
			orig_lines = orig_num_lines (line);
			new_lines = new_num_lines (line);
			newline = 1;
			while (orig_lines || new_lines || newline) {
				if (getline (&line, &linelen, p2) == -1)
					break;
				
				if (!orig_lines && !new_lines &&
				    line[0] != '\\')
					break;

				switch (line[0]) {
				case ' ':
					if (orig_lines) orig_lines--;
					if (new_lines) new_lines--;
					break;
				case '-':
					if (orig_lines) orig_lines--;
					break;
				case '+':
					if (new_lines) new_lines--;
					break;
				case '\\':
					newline = 0;
					break;
				}
				printf ("%s", line);
			}

			if (!newline &&
			    getline (&line, &linelen, p2) == -1)
				break;
		}
	}

	free (line);
	return 0;
}

static int index_patch2 (FILE *p2)
{
	char *line = NULL;
	size_t linelen = 0;
	int is_context = 0;

	/* Index patch2 */
	while (!feof (p2)) {
		unsigned long skip;
		char *names[2];
		char *p, *end;
		size_t h;
		long pos = ftell (p2);

		if (getline (&line, &linelen, p2) == -1)
			break;

		if (strncmp (line, "--- ", 4)) {
			is_context = !strncmp (line, "*** ", 4);
			continue;
		}

		if (is_context)
			error (EXIT_FAILURE, 0,
			       "I don't understand context diffs yet.");

		h = strcspn (line + 4, "\t\n");
		names[0] = xstrndup (line + 4, h);

		if (getline (&line, &linelen, p2) == -1) {
			free (names[0]);
			break;
		}

		if (strncmp (line, "+++ ", 4)) {
			free (names[0]);
			continue;
		}

		h = strcspn (line + 4, "\t\n");
		names[1] = xstrndup (line + 4, h);

		if (getline (&line, &linelen, p2) == -1) {
			free (names[0]);
			free (names[1]);
			break;
		}

		if (strncmp (line, "@@ ", 3))
			goto try_next;

		p = strchr (line + 3, '+');
		if (!p)
			goto try_next;
		p = strchr (p, ',');
		if (p) {
			/* Like '@@ -1,3 +1,3 @@' */
			p++;
			skip = strtoul (p, &end, 10);
			if (p == end)
				goto try_next;
		} else
			/* Like '@@ -1 +1 @@' */
			skip = 1;

		add_to_list (&files_in_patch2, best_name (2, names), pos);

		while (skip--) {
			if (getline (&line, &linelen, p2) == -1)
				break;

			if (line[0] == '-')
				/* Doesn't count towards skip count */
				skip++;
		}

	try_next:
		free (names[0]);
		free (names[1]);
	}

	if (line)
		free (line);

	return 0;
}

static int interdiff (FILE *p1, FILE *p2)
{
	char *line = NULL;
	size_t linelen = 0;
	int is_context = 0;

	index_patch2 (p2);

	/* Search for next file to patch */
	while (!feof (p1)) {
		char *names[2];
		char *p;
		size_t h;
		long pos, start_pos = ftell (p1);

		if (line) {
			free (line);
			line = NULL;
			linelen = 0;
		}

		if (getline (&line, &linelen, p1) == -1)
			break;

		if (strncmp (line, "--- ", 4)) {
			is_context = !strncmp (line, "*** ", 4);
			continue;
		}

		if (is_context)
			error (EXIT_FAILURE, 0,
			       "I don't understand context diffs yet.");

		h = strcspn (line + 4, "\t\n");
		p = xstrndup (line + 4, h);
		names[0] = p;

		if (getline (&line, &linelen, p1) == -1)
			break;

		if (strncmp (line, "+++ ", 4))
			continue;

		h = strcspn (line + 4, "\t\n");
		p = xstrndup (line + 4, h);
		names[1] = p;

		p = xstrdup (best_name (2, names));
		free (names[0]);
		free (names[1]);
		
		/* check if we need to process it and init context */
		if (!check_filename(p)) {
			add_to_list (&files_done, p, 0);
			continue;
		}

		fseek (p1, start_pos, SEEK_SET);
		pos = file_in_list (files_in_patch2, p);
		if (pos == -1) {
			output_patch1_only (p1, combining);
		} else {
			fseek (p2, pos, SEEK_SET);
			output_delta (p1, p2);
		}

		add_to_list (&files_done, p, 0);
	}

	copy_residue (p2);

	free_list (files_in_patch2);
	free_list (files_done);
	if (line)
		free (line);
	return 0;
}

NORETURN
static void syntax (int err)
{
	const char *const syntax_str =
"usage: %s [OPTIONS] patch1 patch2\n"
"       %s --version|--help\n"
"OPTIONS are:\n"
"  -U N            max lines of context to carry\n"
"  -i              Consider upper- and lower-case to be the same\n"
"  -w              ignore whitespace changes in patches\n"
"  -b              ignore changes in the amount of whitespace\n"
"  -B              ignore changes whose lines are all blank\n"
"  -p N            pathname components to ignore\n"
"  -q              don't add rationale text\n"
"  -d PAT          drop context on matching files\n"
"  -z              decompress .gz and .bz2 files\n"
"  --interpolate   run as 'interdiff'\n"
"  --combine       run as 'combinediff'\n";

	fprintf (err ? stderr : stdout, syntax_str, progname, progname);
	exit (err);
}

static void set_interdiff (void)
{
	/* This is interdiff. */
	set_progname ("interdiff");
	combining = 0;
}

static void set_combinediff (void)
{
	/* This is combinediff. */
	set_progname ("combinediff");
	combining = 1;
}

static void interdiff_or_combinediff (const char *argv0)
{
	/* This is interdiff, unless it is named 'combinediff'. */
	const char *p = strrchr (argv0, '/');
	if (!p++)
		p = argv0;
	if (strstr (p, "combine"))
		set_combinediff ();
	else
		set_interdiff ();
}

int main (int argc, char *argv[])
{
	FILE *p1, *p2;
	int num_diff_opts = 0;
	int ret;

	interdiff_or_combinediff (argv[0]);
	diff_opts[0] = '\0';
	for (;;) {
		static struct option long_options[] = {
			{"help", 0, 0, 1000 + 'H'},
			{"version", 0, 0, 1000 + 'V'},
			{"interpolate", 0, 0, 1000 + 'I' },
			{"combine", 0, 0, 1000 + 'C' },
			{"debug", 0, 0, 1000 + 'D' },
			{0, 0, 0, 0}
		};
		char *end;
		int c = getopt_long (argc, argv, "BU:bd:hip:qwz",
				     long_options, NULL);
		if (c == -1)
			break;

		switch (c) {
		case 1000 + 'V':
			printf("%s - patchutils version %s\n", progname,
			       VERSION);
			exit(0);
		case 1000 + 'H':
			syntax (0);
			break;
		case 'h':
			break;
		case 'U':
			max_context_real = strtoul (optarg, &end, 0);
			if (optarg == end)
				syntax (1);
			break;
		case 'p':
			ignore_components = strtoul (optarg, &end, 0);
			if (optarg == end)
				syntax (1);
			break;
		case 'q':
			human_readable = 0;
			break;
		case 'd':
			patlist_add (&pat_drop_context, optarg);
			break;
		case 'z':
			unzip = 1;
			break;
		case 'B':
		case 'b':
		case 'i':
		case 'w':
			if (!memchr (diff_opts, c, num_diff_opts)) {
				diff_opts[num_diff_opts++] = c;
				diff_opts[num_diff_opts] = '\0';
			}
			break;
		case 1000 + 'I':
			set_interdiff ();
			break;
		case 1000 + 'C':
			set_combinediff ();
			break;
		case 1000 + 'D':
			debug = 1;
			break;
		default:
			syntax(1);
		}
	}
	
	if (optind + 2 != argc)
		syntax (1);
	
	if (unzip) {
		p1 = xopen_unzip (argv[optind], "rb");
		p2 = xopen_unzip (argv[optind + 1], "rb");
	} else {
		p1 = xopen_seekable (argv[optind], "rb");
		p2 = xopen_seekable (argv[optind + 1], "rb");
	}

	ret = interdiff (p1, p2);

	fclose (p1);
	fclose (p2);
	patlist_free (&pat_drop_context);
	return 0;
}
