#!/usr/bin/perl -w
# $RCSfile: FileLRUCache.pm,v $ $Revision: 1.1 $ $Date: 1999/06/15 16:47:34 $ $Author: snowhare $
package Tie::FileLRUCache;

use strict;
use Carp;
use Fcntl qw (:flock);
use Tie::Hash;
use Digest::SHA1 qw(sha1_hex);
use Storable qw (nstore nfreeze retrieve);
use File::Path;
use Class::NamedParms;
use Class::ParmList;
use vars qw (@ISA $VERSION);

@ISA     = qw (Tie::Hash Class::NamedParms);
$VERSION = "1.01";

=head1 NAME

Tie::FileLRUCache - A lightweight but robust filesystem based persistent LRU cache 

=head1 CHANGES

1.01 19991209 Added 'detainting' to cache management code.

=head1 SYNOPSIS

=head2 OBJECT INTERFACE

 use Tie::FileLRUCache;
 my $cache = Tie::FileLRUCache->new({ -cache_dir => $directory, -keep_last => 100 });

 # Inserting value into LRU cache using '-key'
 $cache->update({ -key => $key, -value => $value });


 # Inserting value into LRU cache using '-cache_key'
 my $cache_key = $cache->make_cache_key({ -key => $key });
 $cache->update({ -cache_key => $cache_key, -value => $value });


 # Checking LRU cache
 my ($in_cache,$value) = $cache->check({ -key => $key });
 if ($in_cache) {
     return $value;
 }
 # Not in cache - do something else


 # Checking LRU cache with speed up hack for objects, hashes, arrays etc used as keys
 my $cache_key = $cache->make_cache_key({ -key => $something });
 my ($in_cache,$value) = $cache->check({ -cache_key => $cache_key });
 if ($in_cache) {
     return $value;
 }
 # Not in cache - do something else


 # Deleting a key and its value from the cache
 $cache->delete({ -key => $key });


 # Clearing LRU cache
 $cache->clear;

=head2 TIED INTERFACE 

 use Tie::FileLRUCache;

 [$X =] tie %hash,  'Tie::FileLRUCache', $cache_dir, $keep_last_n;

 # Adding a key/value to the cache
 $hash{$key} = $value;

 # Checking the cache
 if (not exists $hash{$key}) {;
     # No match
       .
       .
       .

 } else {
     my $value = $hash{$key};
       .
       .
       .

 }

 # Removing a value from the cache;
 delete $hash{$key};

 # Clearing the cache
 %hash = ();

 Note: Iteration over the cache (each, keys, values) is _NOT_
 supported. 

=cut

=head1 DESCRIPTION

Provides a persistent filesystem based LRU cache.  

It uses the 'last accessed' timestamp generated by the file system 
to determine the 'oldest' cache entry and discards the oldest 
cache entries when needed to stay under the -keep_last limit. 

If you store thing very fast (such that many entries receive the 
_same_ time stamp), it is essentially a coin toss which entry 
within a single timestamped second gets purged from the cache
to make room for new ones.

=cut

=head1 OBJECT METHODS 

=cut

#######################################################################

=over 4

=item C<new($parm_ref);>

Creates and optionally initializes a Tie::FileLRUCache object:

Example: 

  my $cache = Tie::FileLRUCache->new({
                       -cache_dir => '/tmp/testing',
                       -keep_last => 100,
                     });

=back

=cut

sub new {
    my $proto = shift;
    my $class = ref ($proto) || $proto;
    my $self  = Class::NamedParms->new(-cache_dir, -keep_last);
    bless $self,$class;

    # Read any passed parms
    my ($parm_ref) = {};
    if ($#_ == 0) {
        $parm_ref  = shift; 
    } elsif ($#_ > 0) { 
        %$parm_ref = @_; 
    }
    my $parms = Class::ParmList->new({ -parms => $parm_ref,
                                       -legal => [-cache_dir, -keep_last],
                                    -required => [],
                                    -defaults => { -keep_last => 100,
                                                   -cache_dir => undef,
                                                   },
                            });
    if (not defined $parms) {
        my $error_message = Class::ParmList->error;
        croak (__PACKAGE__ . "::new() - Parameter error '$error_message'\n");
        $self->set($parm_ref);
    }

    # Save settings
    my ($cache_dir,$keep_last) = $parms->get(-cache_dir,-keep_last);
    $self->set(-cache_dir => $cache_dir, -keep_last => $keep_last);

    $self;
}

#######################################################################

=over 4

=item C<check($parm_ref);>

Reads the cache for the key. 

Returns two values: $cache_hit (true if a hit was found, false if not)
                    $value     (the cached value, undef if no hit)

Examples:

   my ($cache_hit,$value) = $cache->check({ -key => $key });
   
   my ($cache_hit,$value) = $cache->check({ -cache_key => $cache_key });

The '-key' form is used when you just want to use a raw key. It can use
blessed objects, hash refs, scalars, or array refs as keys. The more complex
structures take a speed penalty for computing a canonical form.
You can minimize this penalty by using the '-cache_key' form instead.

The '-cache_key' form is used for performance reasons when using keys
such as complex blessed objects or hashes as a key. The -cache_key
is obtained with a call to 'make_cache_key'. It is legal to mix
-cache_key and -key based calls - they are cross-compatible.

=back

=cut

sub check {
    my $self = shift;

    # Read any passed parms
    my ($parm_ref) = {};
    if ($#_ == 0) {
        $parm_ref  = shift; 
    } elsif ($#_ > 0) { 
        %$parm_ref = @_; 
    }
    my $parms = Class::ParmList->new({ -parms => $parm_ref,
                                       -legal => [-cache_key, -key],
                                    -required => [],
                                    -defaults => {}, 
                                  });
    if (not defined $parms) {
        my $error_message = Class::ParmList->error;
        croak (__PACKAGE__ . "::check() - $error_message\n");
        $self->set($parm_ref);
    }
    my ($key,$cache_key) = $parms->get(-key,-cache_key);
    if (not (defined ($key) or defined ($cache_key))) {
        croak (__PACKAGE__ . "::check() - Called without either a -key or -cache_key\n");
    }

    my ($cache_dir) = $self->get(-cache_dir);
    if (not defined $cache_dir) {
        croak (__PACKAGE__ . "::check - No cache directory set.\n");
    }

    # Ok. Set our lock on the cache directory
    $self->_lock_cache;

    # Generate the cache_key (done by making a cannonical 
    # network order Storable string out of the key) if we
    # don't already have it 
    if (not defined $cache_key) {
        $cache_key = $self->make_cache_key({ -key => $key });
    }

    # Generate a unique cache file
    # name by taking a SHA1 hash of 
    # $cache_key
    my $cache_hash = sha1_hex($cache_key);
    $cache_hash    =~ s/\s//go;
    my $cache_file ="$cache_dir/cacheline_$cache_hash";

    # Check if there is a cache entry for this key
    if (not -e $cache_file) {
        $self->_unlock_cache;
        return (0,undef);
    }

    # Yes. Get it. A side effect of this is to
    # update the 'last accessed' date of the file.
    my $entry;
    eval { $entry = retrieve($cache_file); };
    if ($@) {
        my $error = $@;
        $self->_unlock_cache;
        croak(__PACKAGE__ . "::check - Error while retrieving cache entry file '$cache_file': $error\n");
    }
    if (not defined $entry) {
        my $error = $!;
        $self->_unlock_cache;
        croak(__PACKAGE__ . "::update - Failed to retrieve cache entry file '$cache_file': $error\n");
    }

    # Since we are using a hash of the actual object
    # for the cache file name, there is the (incredibly remote)
    # possibility that this is not the actual data we want
    # - so we double check the actual read key against the requested key.
    my $verifier = $entry->{-cache_key};
    if ($verifier ne $cache_key) {
        $self->_unlock_cache;
        return (0,undef);
    }


    # Release the lock.
    $self->_unlock_cache;

    # Give them their cupie doll
    return (1,$entry->{-value});
}

#######################################################################

=over 4

=item C<make_cache_key($parm_ref);>

Generates a cache key by canonicalizing a passed 
key as a network ordered canonical Storable string.

Example:

 my $cache_key = $cache->make_cache_key({ -key => $key });

=back

=cut

sub make_cache_key {
    my ($self) = shift;

    # Read passed parms
    my ($parm_ref) = {};
    if ($#_ == 0) {
        $parm_ref  = shift; 
    } elsif ($#_ > 0) { 
        %$parm_ref = @_; 
    }
    my $parms = Class::ParmList->new({ -parms => $parm_ref,
                                       -legal => [],
                                    -required => [-key],
                                    -defaults => {}, 
                                  });
    if (not defined $parms) {
        my $error_message = Class::ParmList->error;
        croak (__PACKAGE__ . "::make_cache_key() - $error_message\n");
        $self->set($parm_ref);
    }
    my ($key) = $parms->get(-key);

    my $temp =  $Storable::canonical;
    my $result = nfreeze(\$key);
    $Storable::canonical = $temp;

    if (not $result) {
        croak (__PACKAGE__ . "::check() - Unable to serialize passed -key value: $!");
    }
    $result;
}

#######################################################################

=over 4

=item C<clear;>

Completely clears the cache.

=back

=cut

sub clear {
    my ($self) = shift;

    my ($cache_dir) = $self->get(-cache_dir);
    if (not defined $cache_dir) {
        croak (__PACKAGE__ . "::clear - No cache directory set.\n");
    }
    if ($cache_dir eq '') {
        croak (__PACKAGE__ . "::clear - Cannot use root directory as cache directory.\n");
    }
    if ((-e $cache_dir) and (not -d _)) {
        croak (__PACKAGE__ . "::clear - '$cache_dir' already exists and is not a directory.\n");
    }
    $self->_lock_cache;
    if (not opendir (CACHE_DIR,$cache_dir)) {
        my $error = $!;
        $self->_unlock_cache;
        croak (__PACKAGE__ . "::clear - Failed to open directory '$cache_dir' for reading: $error\n");
    }
    my (@raw_file_list) = map { "$cache_dir/$_" } grep(/^cacheline_.*$/,readdir(CACHE_DIR));
    # 'Taint' needs a talking to. It is _MUCH_ too aggressive.
    my @file_list = ();
    foreach my $item (@raw_file_list) {
        my ($filename) = $item =~ m/^(.*)$/s;
        push (@file_list,$filename);
    }
    if( not closedir (CACHE_DIR)) {
        my $error = $!;
        $self->_unlock_cache;
        croak (__PACKAGE__ . "::clear - Failed to close directory '$cache_dir': $!\n");
    }
    foreach my $file (@file_list) {
        if (not unlink $file) {
            my $error = $!;
            $self->_unlock_cache;
            croak(__PACKAGE__ . "::clear - Failed to unlink file '$file': $!");
        }
    }
    $self->_unlock_cache;
}

#######################################################################

=over 4

=item C<update($parm_ref);>

Updates the Least Recently Used cache for the specified
key with the passed value.  '-keep_last' is optional after the first access
to a dataset. It will use the _most recent_ 'keep_last' used
if not specified.

It is legal to use ordinary scalars, hash references, or array references
as keys as well as objects as -keys or -values.  Basically, anything that 
Storable can serialize can be used.

Examples:

 $cache->update({ -key => $key, -value => $value });

 $cache->update({ -key => $key, -value => $value, -keep_last => 100});

 my $cache_key = $cache->make_cache_key({ -key => $key });
 $cache->update({ -cache_key => $cache_key, -value => $value });

 my $cache_key = $cache->make_cache_key({ -key => $key });
 $cache->update({ -cache_key => $cache_key, -value => $value, -keep_last => 50 });

=back

=cut

sub update {
    my ($self) = shift;

    # Read passed parms
    my ($parm_ref) = {};
    if ($#_ == 0) {
        $parm_ref  = shift; 
    } elsif ($#_ > 0) { 
        %$parm_ref = @_; 
    }
    my $parms = Class::ParmList->new({ -parms => $parm_ref,
                                       -legal => [-keep_last, -key, -cache_key],
                                    -required => [-value],
                                    -defaults => { -keep_last => $self->get(-keep_last) }, 
                                  });
    if (not defined $parms) {
        my $error_message = Class::ParmList->error;
        croak (__PACKAGE__ . "::update() - $error_message\n");
        $self->set($parm_ref);
    }
    my ($key,$cache_key,$value,$keep_last) = $parms->get(-key, -cache_key, -value, -keep_last);
    if (not (defined ($key) or defined ($cache_key))) {
        croak (__PACKAGE__ . "::update() - Called without either a -key or -cache_key\n");
    }

    my ($cache_dir) = $self->get(-cache_dir);
    if (not defined $cache_dir) {
        croak (__PACKAGE__ . "::update - No cache directory set.\n");
    }

    # Generate the cache_key (done by making a cannonical 
    # network order Storable string out of the key) if we
    # don't already have it 
    if (not defined $cache_key) {
        $cache_key = $self->make_cache_key({ -key => $key });
    }

    # Generate a unique cache file
    # name by taking a SHA1 hash of 
    # $cache_key
    my $cache_hash = sha1_hex($cache_key);
    $cache_hash    =~ s/\s//go;
    my $cache_file ="$cache_dir/cacheline_$cache_hash";

    # Serialize the $value for storage
    my $entry = { -cache_key => $cache_key, -value => $value };

    # Set our lock on the cache directory
    $self->_lock_cache;

    # Store it.
    my $result;
    eval { $result = nstore($entry,$cache_file); };
    if ($@) {
        my $error = $@;
        $self->_unlock_cache;
        croak(__PACKAGE__ . "::update - Error while saving cache entry file '$cache_file': $error");
    }
    if (not defined $result) {
        my $error = $!;
        $self->_unlock_cache;
        croak(__PACKAGE__ . "::update - Error while saving cache entry file '$cache_file': $error\n");
    }

    # Check if we need to purge old entries
    if (not opendir (CACHE_DIR,$cache_dir)) {
        my $error = $!;
        $self->_unlock_cache;
        croak (__PACKAGE__ . "::update - Failed to open directory '$cache_dir' for reading: $error\n");
    }
    my (@raw_file_list) = map { "$cache_dir/$_" } grep(/^cacheline_.*$/,readdir(CACHE_DIR));
    # 'Taint' needs a talking to. It is _MUCH_ too aggressive.
    my @file_list = ();
    foreach my $item (@raw_file_list) {
        my ($filename) = $item =~ m/^(.*)$/s;
        push (@file_list,$filename);
    }

    closedir (CACHE_DIR)
        or croak (__PACKAGE__ . "::update - Failed to close directory '$cache_dir': $!\n");

    # Nope. All done. 
    if ($#file_list < $keep_last) {
        $self->_unlock_cache;
        return;
    }
    # Yes. Delete the excess entries (usually only one)
    my %file_last_access = map { $_ => -A $_ } @file_list;
    @file_list =  sort { $file_last_access{$a} <=> $file_last_access{$b} } @file_list;

    for (my $count=$keep_last;$count<=$#file_list;$count++) {
        my $file = $file_list[$count];    
        if (not unlink $file) {
            my $error = $!;
            $self->_unlock_cache;
            croak(__PACKAGE__ . "::update - Failed to unlink file '$file': $!");
        }
    }

    # Release our lock and return
    $self->_unlock_cache;
}

#######################################################################

=over 4

=item C<delete($parm_ref);>

Forces the deletion of a specific key from the cache.

Example:

 $cache->delete({ -key => $key });

=back

=cut

sub delete {
    my ($self) = shift;

    # Read passed parms
    my ($parm_ref) = {};
    if ($#_ == 0) {
        $parm_ref  = shift; 
    } elsif ($#_ > 0) { 
        %$parm_ref = @_; 
    }
    my $parms = Class::ParmList->new({ -parms => $parm_ref,
                                       -legal => [-key, -cache_key],
                                    -required => [],
                                    -defaults => {}, 
                                  });
    if (not defined $parms) {
        my $error_message = Class::ParmList->error;
        croak (__PACKAGE__ . "::delete() - $error_message\n");
        $self->set($parm_ref);
    }
    my ($key,$cache_key) = $parms->get(-key, -cache_key);
    if (not (defined ($key) or defined ($cache_key))) {
        croak (__PACKAGE__ . "::delete() - Called without either a -key or -cache_key\n");
    }

    my ($cache_dir) = $self->get(-cache_dir);
    if (not defined $cache_dir) {
        croak (__PACKAGE__ . "::delete - No cache directory set.\n");
    }
    if ($cache_dir eq '') {
        croak (__PACKAGE__ . "::delete - Cannot use root directory as cache directory.\n");
    }

    # Generate the cache_key (done by making a cannonical 
    # network order Storable string out of the key) if we
    # don't already have it 
    if (not defined $cache_key) {
        $cache_key = $self->make_cache_key({ -key => $key });
    }

    # Generate a unique cache file
    # name by taking a SHA1 hash of 
    # $cache_key
    my $cache_hash = sha1_hex($cache_key);
    $cache_hash    =~ s/\s//go;
    my $cache_file ="$cache_dir/cacheline_$cache_hash";

    # Ok. Set our lock on the cache directory
    $self->_lock_cache;

    # If it is in the cache, remove it
    if ((-e $cache_file) and (not unlink $cache_file)) {
        my $error = $!;
        $self->_unlock_cache;
        croak(__PACKAGE__ . "::delete - Failed to unlink file '$cache_file': $!");
    }

    # Release our lock and return
    $self->_unlock_cache;
}

#######################################################################
#                                                                     #
# PRIVATE METHODS                                                     #
#                                                                     #
# Internals. Documented for maintainance reasons only.                #
# Do not use these methods.                                           #
#                                                                     #
#######################################################################

# _lock_cache;
# 
# Obtains a lock on the 'cache.lock' file for this LRU cache.
# 
#  Example:
#     $self->_lock_cache;
# 
# This will create the 'cache.lock' file if it does not already exist,
# creating any intermediate directories as needed.
# 
# It also writes the current PID to the lock file.

sub _lock_cache {
    my ($self) = shift;

    my ($cache_dir) = $self->get(-cache_dir);
    if (not defined $cache_dir) {
        croak (__PACKAGE__ . "::_lock_cache - No cache directory set.\n");
    }
    if ($cache_dir eq '') {
        croak (__PACKAGE__ . "::_lock_cache - Cannot use root directory as cache directory.\n");
    }
    if ((-e $cache_dir) and (not -d _)) {
        croak (__PACKAGE__ . "::_lock_cache - '$cache_dir' already exists and is not a directory.\n");
    }
    if (not -e $cache_dir) {
        mkpath ($cache_dir) 
            or croak (__PACKAGE__ . "::_lock_cache - unable to create directory '$cache_dir'\n");
    }
    if (not ((-e $cache_dir) and (-d _))) {
        croak (__PACKAGE__ . "::_lock_cache - Unable to create directory '$cache_dir'\n");
    }
    my $document_name = "$cache_dir/.cache.lock";
    open (CACHE_LOCK,">>$document_name") 
        or croak (__PACKAGE__ . "::_lock_cache - Unable to open '$document_name': $!\n");
    my $lock_timeout = 100;
    while (not flock(CACHE_LOCK,LOCK_EX()|LOCK_NB())) {
        $lock_timeout--;
        select (undef,undef,undef,0.1);
        if ($lock_timeout == 0) { 
            croak (__PACKAGE__ . "::_lock_cache - Unable to get an exclusive lock on '$document_name': $!\n"); 
        }
    }
    my $fh = select (CACHE_LOCK);
    $|++;
    select ($fh);
    truncate (CACHE_LOCK,0)
        or croak (__PACKAGE__ . "::_lock_cache - Unable to truncate '$document_name': $!\n");
    print CACHE_LOCK "$$\n";
}

#######################################################################

# _unlock_cache;
# 
# Release a lock on the 'cache.lock' file for this LRU cache.
# 
#  Example:
#     $self->_unlock_cache;
 
sub _unlock_cache {
    my ($self) = shift;

    truncate (CACHE_LOCK,0)
        or croak (__PACKAGE__ . "::_lock_cache - Unable to truncate cache.lock file: $!\n");
    close (CACHE_LOCK) or
        croak (__PACKAGE__ . "::_unlock_cache - Error while closing cache.lock file: $!\n");
}

####################################################################

sub TIEHASH {
    my $proto = shift;
    my $class = ref ($proto) || $proto;
    my $self  = Class::NamedParms->new(-cache_dir, -keep_last);
    bless $self,$class;

    my ($cache_dir,$keep_last) = @_;

    if (not defined $keep_last) {
        $keep_last = 100;
    }
    if (not defined $cache_dir) {
        croak(__PACKAGE__ . ": Missing required parameter (cache_dir)\n");
    }
    $self->set(-cache_dir => $cache_dir, -keep_last => $keep_last);
    $self;
}

#######################################################################

sub STORE {
    my $self = shift;

    my ($key,$value) = @_;

    $self->update({ -key => $key, -value => $value });
}

#######################################################################

sub FETCH {
    my ($self) = shift;

    my ($key)  = @_;

    my ($cache_hit,$value) = $self->check({ -key => $key });

    $value;
}

#######################################################################

sub DELETE {
    my ($self) = shift;
    
    my ($key) = @_;

    $self->delete({ -key => $key });
}

#######################################################################

sub CLEAR {
    my $self = shift;

    $self->clear;
}

#######################################################################

sub EXISTS {
    my ($self) = shift;
    
    my ($key) = @_;

    my ($cache_hit,$value) = $self->check({ -key => $key });

    $cache_hit;
}

#######################################################################


# DESTROY;
#
# Called when the tied object is being destroyed. 

sub DESTROY {
    my $self = shift;

}

#######################################################################

#
# Iteration over the cache is not supported
#

sub FIRSTKEY { undef; }

#######################################################################

#
# Iteration over the cache is not supported
#

sub NEXTKEY { undef; }

#######################################################################

=head1 COPYRIGHT

Copyright 1999, Benjamin Franz (<URL:http://www.nihongo.org/snowhare/>) and 
FreeRun Technologies, Inc. (<URL:http://www.freeruntech.com/>). All Rights Reserved.
This software may be copied or redistributed under the same terms as Perl itelf.

=head1 AUTHOR

Benjamin Franz

=head1 TODO

Debugging.

=cut

1;
